<?php
/**
 * Theme Palace functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Theme Palace
 * @subpackage Personify Pro
 * @since Personify Pro 1.0
 */

if ( ! function_exists( 'personify_pro_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function personify_pro_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Theme Palace, use a find and replace
		 * to change 'personify-pro' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'personify-pro' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		add_theme_support( 'register_block_pattern' ); 

		add_theme_support( 'register_block_style' ); 

		add_theme_support( 'responsive-embeds' );
		
		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Load Footer Widget Support.
		require_if_theme_supports( 'footer-widgets', get_template_directory() . '/inc/footer-widget.php' );
		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 295, 400, true );

		// Set the default content width.
		$GLOBALS['content_width'] = 525;
		
		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' 	=> esc_html__( 'Primary', 'personify-pro' ),
			'social' 	=> esc_html__( 'Social', 'personify-pro' ),
			'footer' 	=> esc_html__( 'Footer', 'personify-pro' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'personify_pro_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// This setup supports logo, site-title & site-description
		add_theme_support( 'custom-logo', array(
			'height'      => 60,
			'width'       => 135,
			'flex-height' => true,
			'flex-width'  => true,
			'header-text' => array( 'site-title', 'site-description' ),
		) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		//  Add theme support for excerpt for pages
		add_post_type_support( 'page', 'excerpt' );

		/*
		 * This theme styles the visual editor to resemble the theme style,
		 * specifically font, colors, icons, and column width.
		 */
		add_editor_style( array( 'assets/css/editor-style' . personify_pro_min() . '.css', personify_pro_fonts_url() ) );
			// Gutenberg support
		add_theme_support( 'editor-color-palette', array(
	       	array(
				'name' => esc_html__( 'Red', 'personify-pro' ),
				'slug' => 'red',
				'color' => '#d91717',
	       	),
	       	array(
	           	'name' => esc_html__( 'Black', 'personify-pro' ),
	           	'slug' => 'black',
	           	'color' => '#333',
	       	),
	       	array(
	           	'name' => esc_html__( 'Grey', 'personify-pro' ),
	           	'slug' => 'grey',
	           	'color' => '#959595',
	       	),
	   	));

		add_theme_support( 'align-wide' );
		add_theme_support( 'editor-font-sizes', array(
		   	array(
		       	'name' => esc_html__( 'small', 'personify-pro' ),
		       	'shortName' => esc_html__( 'S', 'personify-pro' ),
		       	'size' => 12,
		       	'slug' => 'small'
		   	),
		   	array(
		       	'name' => esc_html__( 'regular', 'personify-pro' ),
		       	'shortName' => esc_html__( 'M', 'personify-pro' ),
		       	'size' => 16,
		       	'slug' => 'regular'
		   	),
		   	array(
		       	'name' => esc_html__( 'larger', 'personify-pro' ),
		       	'shortName' => esc_html__( 'L', 'personify-pro' ),
		       	'size' => 36,
		       	'slug' => 'larger'
		   	),
		   	array(
		       	'name' => esc_html__( 'huge', 'personify-pro' ),
		       	'shortName' => esc_html__( 'XL', 'personify-pro' ),
		       	'size' => 48,
		       	'slug' => 'huge'
		   	)
		));
		add_theme_support('editor-styles');
		add_theme_support( 'wp-block-styles' );
	}
endif;
add_action( 'after_setup_theme', 'personify_pro_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function personify_pro_content_width() {

	/**
	 * Filter Theme Palace content width of the theme.
	 *
	 * @since Personify Pro 1.0
	 *
	 * @param int $content_width Content width in pixels.
	 */
	$GLOBALS['content_width'] = apply_filters( 'personify_pro_content_width', 1180 );
}
add_action( 'template_redirect', 'personify_pro_content_width', 0 );

if ( ! function_exists( 'personify_pro_fonts_url' ) ) :
/**
 * Register Google fonts
 *
 * @return string Google fonts URL for the theme.
 */
function personify_pro_fonts_url() {
	$fonts_url = '';

	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	// Header Font
	
	/* translators: If there are characters in your language that are not supported by Abhaya Libre, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Abhaya Libre font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Abhaya Libre:400,600,700';
	}

	/* translators: If there are characters in your language that are not supported by Rajdhani, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Rajdhani font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Rajdhani';
	}

	/* translators: If there are characters in your language that are not supported by Cherry Swash, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Cherry Swash font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Cherry Swash';
	}

	/* translators: If there are characters in your language that are not supported by Philosopher, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Philosopher font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Philosopher';
	}

	/* translators: If there are characters in your language that are not supported by Slabo 27px, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Slabo 27px font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Slabo 27px';
	}

	/* translators: If there are characters in your language that are not supported by Dosis, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Dosis font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Dosis';
	}

	// Body Fonts

	/* translators: If there are characters in your language that are not supported by Open Sans, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Open Sans font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Open Sans:300,400,600,700';
	}

	/* translators: If there are characters in your language that are not supported by |News Cycle, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'News Cycle font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'News Cycle';
	}

	/* translators: If there are characters in your language that are not supported by Pontano Sans, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Pontano Sans font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Pontano Sans';
	}

	/* translators: If there are characters in your language that are not supported by Gudea, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Gudea font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Gudea';
	}

	/* translators: If there are characters in your language that are not supported by Quattrocento, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Quattrocento font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Quattrocento';
	}

	/* translators: If there are characters in your language that are not supported by Khand, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Khand font: on or off', 'personify-pro' ) ) {
		$fonts[] = 'Khand';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
		'family' => urlencode( implode( '|', $fonts ) ),
		'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}
endif;

/**
 * Add preconnect for Google Fonts.
 *
 * @since Personify Pro 1.0
 *
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function personify_pro_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'personify-pro-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'personify_pro_resource_hints', 10, 2 );

/**
 * Enqueue scripts and styles.
 */
function personify_pro_scripts() {
	$options = personify_pro_get_theme_options();

	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'personify-pro-fonts', wptt_get_webfont_url( personify_pro_fonts_url() ), array(), null );

	// slick
	wp_enqueue_style( 'slick', get_template_directory_uri() . '/assets/css/slick' . personify_pro_min() . '.css' );

	// blocks
	wp_enqueue_style( 'personify-pro-blocks', get_template_directory_uri() . '/assets/css/blocks' . personify_pro_min() . '.css' );

	// slick theme
	wp_enqueue_style( 'slick-theme', get_template_directory_uri() . '/assets/css/slick-theme' . personify_pro_min() . '.css' );

	wp_enqueue_style( 'personify-pro-style', get_stylesheet_uri() );

	// Load the colorscheme.
	$color_scheme = ! empty( $options['colorscheme'] ) ? $options['colorscheme'] : 'red';
	if ( $color_scheme !== 'custom' || is_customize_preview() ) {
		wp_enqueue_style( 'personify-pro-colors', get_template_directory_uri() . '/assets/css/' . esc_attr( $color_scheme ) . personify_pro_min() . '.css', array( 'personify-pro-style' ), '' );
	}

	// Load the html5 shiv.
	wp_enqueue_script( 'html5', get_template_directory_uri() . '/assets/js/html5' . personify_pro_min() . '.js', array(), '3.7.3' );
	wp_script_add_data( 'html5', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'personify-pro-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix' . personify_pro_min() . '.js', array(), '20160412', true );

	wp_enqueue_script( 'personify-pro-navigation', get_template_directory_uri() . '/assets/js/navigation' . personify_pro_min() . '.js', array( 'jquery' ), '20151215', true );
	
	$personify_pro_l10n = array(
		'quote'          => personify_pro_get_svg( array( 'icon' => 'quote-right' ) ),
		'expand'         => esc_html__( 'Expand child menu', 'personify-pro' ),
		'collapse'       => esc_html__( 'Collapse child menu', 'personify-pro' ),
		'icon'           => personify_pro_get_svg( array( 'icon' => 'angle-down', 'fallback' => true ) ),
	);
	
	wp_localize_script( 'personify-pro-navigation', 'personify_pro_l10n', $personify_pro_l10n );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	// jquery slick
	wp_enqueue_script( 'jquery-slick', get_template_directory_uri() . '/assets/js/slick' . personify_pro_min() . '.js', array( 'jquery' ), '', true );

	// Custom Js
	wp_enqueue_script( 'personify-pro-custom', get_template_directory_uri() . '/assets/js/custom' . personify_pro_min() . '.js', array( 'jquery' ), '20151215', true );

	if ( 'infinite' == $options['pagination_type'] ) {
		// infinite scroll js
		wp_enqueue_script( 'personify-pro-infinite-scroll', get_template_directory_uri() . '/assets/js/infinite-scroll' . personify_pro_min() . '.js', array( 'jquery' ), '', true );
	}
}
add_action( 'wp_enqueue_scripts', 'personify_pro_scripts' );

/**
 * Enqueue editor styles for Gutenberg
 *
 * @since Food Restro Pro 1.0.0
 */
function personify_pro_block_editor_styles() {
	// Block styles.
	wp_enqueue_style( 'personify-pro-block-editor-style', get_theme_file_uri( '/assets/css/editor-blocks' . personify_pro_min() . '.css' ) );
	// Add custom fonts.
	wp_enqueue_style( 'personify-pro-fonts', wptt_get_webfont_url( personify_pro_fonts_url() ), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'personify_pro_block_editor_styles' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Load core file
 */
require get_template_directory() . '/inc/core.php';