<?php
/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package Bizroot
 */

if ( ! function_exists( 'bizroot_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function bizroot_setup() {
		/*
		 * Make theme available for translation.
		 */
		load_theme_textdomain( 'bizroot-pro', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'bizroot-thumb', 380, 260 );
		add_image_size( 'bizroot-portfolio-slider', 1180, 510 );

		// This theme uses wp_nav_menu() in four location.
		register_nav_menus( array(
			'primary'  => esc_html__( 'Primary Menu', 'bizroot-pro' ),
			'footer'   => esc_html__( 'Footer Menu', 'bizroot-pro' ),
			'social'   => esc_html__( 'Social Menu', 'bizroot-pro' ),
			'notfound' => esc_html__( '404 Menu', 'bizroot-pro' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		/*
		 * Enable support for Post Formats.
		 * See https://developer.wordpress.org/themes/functionality/post-formats/
		 */
		add_theme_support( 'post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'bizroot_custom_background_args', array(
			'default-color' => 'FFFFFF',
			'default-image' => '',
		) ) );

		/*
		 * Enable support for selective refresh of widgets in Customizer.
		 */
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Editor style.
		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		add_editor_style( 'css/editor-style' . $min . '.css' );

		/*
		 * Enable support for custom logo.
		 */
		add_theme_support( 'custom-logo' );

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Add WooCommerce Support.
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-lightbox' );

		// Load Supports.
		require get_template_directory() . '/inc/support.php';

		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for editor styles.
		add_theme_support( 'editor-styles' );

		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__( 'Small', 'bizroot-pro' ),
					'shortName' => esc_html__( 'S', 'bizroot-pro' ),
					'size'      => 13,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__( 'Normal', 'bizroot-pro' ),
					'shortName' => esc_html__( 'M', 'bizroot-pro' ),
					'size'      => 16,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__( 'Large', 'bizroot-pro' ),
					'shortName' => esc_html__( 'L', 'bizroot-pro' ),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__( 'Huge', 'bizroot-pro' ),
					'shortName' => esc_html__( 'XL', 'bizroot-pro' ),
					'size'      => 56,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support( 'editor-color-palette', array(
			array(
				'name'  => esc_html__( 'White', 'bizroot-pro' ),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__( 'Black', 'bizroot-pro' ),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__( 'Gray', 'bizroot-pro' ),
				'slug'  => 'gray',
				'color' => '#f4f4f4',
			),
			array(
				'name'  => esc_html__( 'Yellow', 'bizroot-pro' ),
				'slug'  => 'yellow',
				'color' => '#e5ae4a',
			),
			array(
				'name'  => esc_html__( 'Blue', 'bizroot-pro' ),
				'slug'  => 'blue',
				'color' => '#357fe2',
			),
		) );

		// Do not remove. Used later.
		global $bizroot_default_options;
        $bizroot_default_options = bizroot_get_default_theme_options();
	}
endif;

add_action( 'after_setup_theme', 'bizroot_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function bizroot_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'bizroot_content_width', 640 );
}
add_action( 'after_setup_theme', 'bizroot_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function bizroot_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'bizroot-pro' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'bizroot-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar', 'bizroot-pro' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar.', 'bizroot-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Front Page Widget Area', 'bizroot-pro' ),
		'id'            => 'sidebar-front-page-widget-area',
		'description'   => esc_html__( 'Add widgets here to appear in your Front Page.', 'bizroot-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s"><div class="container">',
		'after_widget'  => '</div></aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2><span class="separator"></span>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'bizroot-pro' ), 1 ),
		'id'            => 'extra-sidebar-1',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'bizroot-pro' ), 2 ),
		'id'            => 'extra-sidebar-2',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'bizroot-pro' ), 3 ),
		'id'            => 'extra-sidebar-3',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'bizroot-pro' ), 4 ),
		'id'            => 'extra-sidebar-4',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'bizroot_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function bizroot_scripts() {

	$theme_version = wp_get_theme()->get( 'Version' );

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	// Add custom fonts.
	wp_enqueue_style( 'bizroot-fonts', bizroot_fonts_url(), array(), null );

	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '4.7.0' );

	wp_enqueue_style( 'animate', get_template_directory_uri() . '/third-party/wow/css/animate' . $min . '.css', '', '3.5' );

	$deps = array();

	wp_enqueue_style( 'jquery-sidr', get_template_directory_uri() .'/third-party/sidr/css/jquery.sidr.dark' . $min . '.css', '', '2.2.1' );

	wp_enqueue_style( 'bizroot-style', get_stylesheet_uri(), $deps, $theme_version );

	// Theme block stylesheet.
	wp_enqueue_style( 'bizroot-block-style', trailingslashit( esc_url ( get_template_directory_uri() ) ) . 'css/blocks.css', array( 'bizroot-style' ), $theme_version );

	wp_enqueue_script( 'bizroot-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix' . $min . '.js', array(), '20130115', true );

	wp_enqueue_script( 'jquery-cycle2', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2' . $min . '.js', array( 'jquery' ), '2.1.6', true );

    $transition_effect = bizroot_get_option( 'featured_slider_transition_effect' );

	if ( in_array( $transition_effect, array( 'tileBlind', 'tileSlide' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-tile', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.tile' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	if ( in_array( $transition_effect, array( 'flipHorz', 'flipVert' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-flip', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.flip' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	if ( in_array( $transition_effect, array( 'shuffle' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-shuffle', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.shuffle' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	wp_enqueue_script( 'jquery-sidr', get_template_directory_uri() . '/third-party/sidr/js/jquery.sidr' . $min . '.js', array( 'jquery' ), '2.2.1', true );

	wp_enqueue_script( 'wow', get_template_directory_uri() . '/third-party/wow/js/wow' . $min . '.js', array( 'jquery' ), '1.1.2', true );

	wp_enqueue_script( 'bizroot-custom', get_template_directory_uri() . '/js/custom' . $min . '.js', array( 'jquery' ), '2.0.0', true );
	$custom_args = array(
		'go_to_top_status' => ( true === bizroot_get_option( 'go_to_top' ) ) ? 1 : 0,
		);
	wp_localize_script( 'bizroot-custom', 'Bizroot_Custom_Options', $custom_args );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'bizroot_scripts' );

/**
 * Enqueue editor styles for Gutenberg
 */
function bizroot_block_editor_styles() {
	// Block styles.
	wp_enqueue_style( 'bizroot-block-editor-style', trailingslashit( esc_url ( get_template_directory_uri() ) ) . 'css/editor-blocks.css' );

	// Add custom fonts.
	wp_enqueue_style( 'bizroot-fonts', bizroot_fonts_url(), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'bizroot_block_editor_styles' );

/**
 * Enqueue admin scripts and styles.
 */
function bizroot_admin_scripts( $hook ) {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	if ( in_array( $hook, array( 'post.php', 'post-new.php' ) ) ) {
		wp_enqueue_style( 'bizroot-metabox', get_template_directory_uri() . '/css/metabox' . $min . '.css', '', '1.0.0' );
		wp_enqueue_script( 'bizroot-custom-admin', get_template_directory_uri() . '/js/admin' . $min . '.js', array( 'jquery', 'jquery-ui-core', 'jquery-ui-tabs' ), '1.0.0', true );
	}

	if ( 'widgets.php' === $hook ) {
		wp_enqueue_style( 'wp-color-picker' );
	    wp_enqueue_script( 'wp-color-picker' );
	    wp_enqueue_media();
		wp_enqueue_style( 'bizroot-custom-widgets-style', get_template_directory_uri() . '/css/widgets' . $min . '.css', array(), '1.0.0' );
		wp_enqueue_script( 'bizroot-custom-widgets', get_template_directory_uri() . '/js/widgets' . $min . '.js', array( 'jquery' ), '1.0.0', true );
	}

}
add_action( 'admin_enqueue_scripts', 'bizroot_admin_scripts' );

/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';

/**
 * Load theme updater functions.
 */
function bizroot_theme_updater() {
	if ( is_admin() ) {
		require_once get_template_directory() . '/updater/theme-updater.php';
	}
}
add_action( 'after_setup_theme', 'bizroot_theme_updater' );
