<?php
/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package Nature_Bliss
 */

if ( ! function_exists( 'nature_bliss_setup' ) ) :

	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function nature_bliss_setup() {

		// Make theme available for translation.
		load_theme_textdomain( 'nature-bliss-pro', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 */
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'nature-bliss-thumb', 380, 360 );
		add_image_size( 'nature-bliss-works', 800, 800, true );

		// This theme uses wp_nav_menu() in four location.
		register_nav_menus( array(
			'primary'  => esc_html__( 'Primary Menu', 'nature-bliss-pro' ),
			'footer'   => esc_html__( 'Footer Menu', 'nature-bliss-pro' ),
			'social'   => esc_html__( 'Social Menu', 'nature-bliss-pro' ),
			'notfound' => esc_html__( '404 Menu', 'nature-bliss-pro' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'nature_bliss_custom_background_args', array(
			'default-color' => 'FFFFFF',
			'default-image' => '',
		) ) );

		/*
		 * Enable support for selective refresh of widgets in Customizer.
		 */
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Editor style.
		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		add_editor_style( 'css/editor-style' . $min . '.css' );

		/*
		 * Enable support for custom logo.
		 */
		add_theme_support( 'custom-logo' );

		// Load default block styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Add WooCommerce Support.
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-lightbox' );

		// Load Supports.
		require get_template_directory() . '/inc/support.php';

	}
endif;

add_action( 'after_setup_theme', 'nature_bliss_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function nature_bliss_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'nature_bliss_content_width', 640 );
}
add_action( 'after_setup_theme', 'nature_bliss_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function nature_bliss_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'nature-bliss-pro' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'nature-bliss-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar', 'nature-bliss-pro' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar.', 'nature-bliss-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Front Page Widget Area', 'nature-bliss-pro' ),
		'id'            => 'sidebar-front-page-widget-area',
		'description'   => esc_html__( 'Add widgets here to appear in your Front Page.', 'nature-bliss-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s"><div class="container">',
		'after_widget'  => '</div></aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'nature-bliss-pro' ), 1 ),
		'id'            => 'extra-sidebar-1',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'nature-bliss-pro' ), 2 ),
		'id'            => 'extra-sidebar-2',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'nature-bliss-pro' ), 3 ),
		'id'            => 'extra-sidebar-3',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'nature-bliss-pro' ), 4 ),
		'id'            => 'extra-sidebar-4',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'nature_bliss_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function nature_bliss_scripts() {

	$theme_version = wp_get_theme()->get( 'Version' );

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '4.7.0' );

	wp_enqueue_style( 'animate', get_template_directory_uri() . '/third-party/wow/css/animate' . $min . '.css', '', '3.5' );

	$fonts_url = nature_bliss_fonts_url();

	if ( ! empty( $fonts_url ) ) {
		wp_enqueue_style( 'nature-bliss-google-fonts', $fonts_url, array(), null );
	}

	wp_enqueue_style( 'nature-bliss-style', get_stylesheet_uri(), array(), $theme_version );

	wp_enqueue_script( 'nature-bliss-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix' . $min . '.js', array(), '20130115', true );

	//wp_enqueue_script( 'nature-bliss-keyboard-image-nav', get_theme_file_uri() . '/js/keyboard-image-navigation' . $min . '.js', array(), date("ymd-Gis", filemtime( get_theme_file_path() . '/js/keyboard-image-navigation' . $min . '.js' ) ), true );

	wp_enqueue_script( 'jquery-cycle2', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2' . $min . '.js', array( 'jquery' ), '2.1.6', true );

    $transition_effect = nature_bliss_get_option( 'featured_slider_transition_effect' );

	if ( in_array( $transition_effect, array( 'tileBlind', 'tileSlide' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-tile', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.tile' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	if ( in_array( $transition_effect, array( 'flipHorz', 'flipVert' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-flip', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.flip' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	if ( in_array( $transition_effect, array( 'shuffle' ) ) ) {
		wp_enqueue_script( 'jquery-cycle2-shuffle', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2.shuffle' . $min . '.js', array( 'jquery' ), '2.1.6', true );
	}

	wp_enqueue_script( 'wow', get_template_directory_uri() . '/third-party/wow/js/wow' . $min . '.js', array( 'jquery' ), '1.1.2', true );

	wp_enqueue_script( 'nature-bliss-custom', get_theme_file_uri() . '/js/custom' . $min . '.js', array( 'jquery' ), date("ymd-Gis", filemtime( get_parent_theme_file_path() . '/js/custom' . $min . '.js' ) ), true );

	$custom_args = array(
		'goToTopStatus' => ( true === nature_bliss_get_option( 'go_to_top' ) ) ? 1 : 0,
		'screenReaderText' => array(
			'expand'   => esc_html__( 'expand child menu', 'nature-bliss-pro' ),
			'collapse' => esc_html__( 'collapse child menu', 'nature-bliss-pro' ),
		)
	);
	
	wp_localize_script( 'nature-bliss-custom', 'natureBlissCustomOptions', $custom_args );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'nature_bliss_scripts' );

/**
 * Enqueue admin scripts and styles.
 */
function nature_bliss_admin_scripts( $hook ) {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	if ( in_array( $hook, array( 'post.php', 'post-new.php' ) ) ) {
		wp_enqueue_style( 'nature-bliss-metabox', get_template_directory_uri() . '/css/metabox' . $min . '.css', '', '1.0.0' );
		wp_enqueue_script( 'nature-bliss-custom-admin', get_template_directory_uri() . '/js/admin' . $min . '.js', array( 'jquery', 'jquery-ui-core', 'jquery-ui-tabs' ), '1.0.0', true );
	}

	if ( 'widgets.php' === $hook ) {
		wp_enqueue_style( 'wp-color-picker' );
	    wp_enqueue_script( 'wp-color-picker' );
	    wp_enqueue_media();
		wp_enqueue_style( 'nature-bliss-custom-widgets-style', get_template_directory_uri() . '/css/widgets' . $min . '.css', array(), '1.0.0' );
		wp_enqueue_script( 'nature-bliss-custom-widgets', get_template_directory_uri() . '/js/widgets' . $min . '.js', array( 'jquery' ), '1.0.0', true );
	}

}
add_action( 'admin_enqueue_scripts', 'nature_bliss_admin_scripts' );

/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';

/**
 * Load theme updater functions.
 */
function nature_bliss_theme_updater() {
	if ( is_admin() ) {
		require_once get_template_directory() . '/updater/theme-updater.php';
	}
}
add_action( 'after_setup_theme', 'nature_bliss_theme_updater' );

// Load Demo Import Class
if ( class_exists( 'CatchThemesDemoImportPlugin' ) ) :
	require( get_template_directory() . '/inc/support/ctdi/demo-import.php' );
endif;

if ( ! function_exists( 'nature_bliss_blocks_support' ) ) :
	/**
	 * Create add default blocks support
	 */
	function nature_bliss_blocks_support() {
		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );

		// Add support for editor styles.
		add_theme_support( 'editor-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__( 'Small', 'nature-bliss-pro' ),
					'shortName' => esc_html__( 'S', 'nature-bliss-pro' ),
					'size'      => 14,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__( 'Normal', 'nature-bliss-pro' ),
					'shortName' => esc_html__( 'M', 'nature-bliss-pro' ),
					'size'      => 18,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__( 'Large', 'nature-bliss-pro' ),
					'shortName' => esc_html__( 'L', 'nature-bliss-pro' ),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__( 'Huge', 'nature-bliss-pro' ),
					'shortName' => esc_html__( 'XL', 'nature-bliss-pro' ),
					'size'      => 54,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support( 'editor-color-palette', array(
			array(
				'name'  => esc_html__( 'White', 'nature-bliss-pro' ),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__( 'Black', 'nature-bliss-pro' ),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__( 'Gray', 'nature-bliss-pro' ),
				'slug'  => 'gray',
				'color' => '#f4f4f4',
			),
			array(
				'name'  => esc_html__( 'Blue', 'nature-bliss-pro' ),
				'slug'  => 'blue',
				'color' => '1b8be0',
			),
			array(
				'name'  => esc_html__( 'Orange', 'nature-bliss-pro' ),
				'slug'  => 'orange',
				'color' => '#fab702',
			),
			array(
				'name'  => esc_html__( 'Green', 'nature-bliss-pro' ),
				'slug'  => 'green',
				'color' => '#568c11',
			),
		) );
	}
	add_action( 'after_setup_theme', 'nature_bliss_blocks_support', 20 );
endif; //nature_bliss_blocks_support

if ( ! function_exists( 'nature_bliss_add_blocks_style' ) ) :
	/**
	 * Add Blocks Style
	 */
	function nature_bliss_add_blocks_style() {
		// Theme block stylesheet.
		wp_enqueue_style( 'nature-bliss-block-style', get_theme_file_uri( '/css/blocks.css' ), array( 'nature-bliss-style' ), date( 'Ymd-Gis', filemtime( get_template_directory() . '/css/blocks.css' ) ) );
	}
	add_action( 'wp_enqueue_scripts', 'nature_bliss_add_blocks_style' );
endif; //nature_bliss_add_blocks_style

if ( ! function_exists( 'nature_bliss_block_editor_styles' ) ) :
	/**
	 * Enqueue editor styles for Blocks
	 */
	function nature_bliss_block_editor_styles() {
		// Block styles.
		wp_enqueue_style( 'nature-bliss-block-editor-style', get_theme_file_uri( '/css/editor-blocks.css' ), null, date( 'Ymd-Gis', filemtime( get_template_directory() . '/css/editor-blocks.css' ) ) );

		// Add custom fonts.
		wp_enqueue_style( 'nature-bliss-fonts', nature_bliss_fonts_url(), array(), null );
	}
	add_action( 'enqueue_block_editor_assets', 'nature_bliss_block_editor_styles' );
endif; //nature_bliss_block_editor_styles
