<?php
/**
 * Helper functions related to customizer and options.
 *
 * @package Nature_Bliss
 */

if ( ! function_exists( 'nature_bliss_get_global_layout_options' ) ) :

	/**
	 * Returns global layout options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_global_layout_options() {

		$choices = array(
			'left-sidebar'            => esc_html__( 'Primary Sidebar - Content', 'nature-bliss-pro' ),
			'right-sidebar'           => esc_html__( 'Content - Primary Sidebar', 'nature-bliss-pro' ),
			'three-columns'           => esc_html__( 'Three Columns ( Secondary - Content - Primary )', 'nature-bliss-pro' ),
			'three-columns-pcs'       => esc_html__( 'Three Columns ( Primary - Content - Secondary )', 'nature-bliss-pro' ),
			'three-columns-cps'       => esc_html__( 'Three Columns ( Content - Primary - Secondary )', 'nature-bliss-pro' ),
			'three-columns-psc'       => esc_html__( 'Three Columns ( Primary - Secondary - Content )', 'nature-bliss-pro' ),
			'three-columns-pcs-equal' => esc_html__( 'Three Columns ( Equal Primary - Content - Secondary )', 'nature-bliss-pro' ),
			'three-columns-scp-equal' => esc_html__( 'Three Columns ( Equal Secondary - Content - Primary )', 'nature-bliss-pro' ),
			'no-sidebar'              => esc_html__( 'No Sidebar', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_layout_options', $choices );
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_pagination_type_options' ) ) :

	/**
	 * Returns pagination type options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_pagination_type_options() {

		$choices = array(
			'default'               => esc_html__( 'Default (Older / Newer Post)', 'nature-bliss-pro' ),
			'numeric'               => esc_html__( 'Numeric', 'nature-bliss-pro' ),
			'infinite-scroll'       => esc_html__( 'Infinite Scroll - Scroll', 'nature-bliss-pro' ),
			'infinite-scroll-click' => esc_html__( 'Infinite Scroll - Click', 'nature-bliss-pro' ),
		);
		return $choices;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_breadcrumb_type_options' ) ) :

	/**
	 * Returns breadcrumb type options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_breadcrumb_type_options() {

		$choices = array(
			'disabled' => esc_html__( 'Disabled', 'nature-bliss-pro' ),
			'simple'   => esc_html__( 'Enabled', 'nature-bliss-pro' ),
		);
		return $choices;

	}

endif;


if ( ! function_exists( 'nature_bliss_get_archive_layout_options' ) ) :

	/**
	 * Returns archive layout options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_archive_layout_options() {

		$choices = array(
			'full'    => esc_html__( 'Full Post', 'nature-bliss-pro' ),
			'excerpt' => esc_html__( 'Post Excerpt', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_archive_layout_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_image_sizes_options' ) ) :

	/**
	 * Returns image sizes options.
	 *
	 * @since 1.0.0
	 *
	 * @param bool  $add_disable True for adding No Image option.
	 * @param array $allowed Allowed image size options.
	 * @return array Image size options.
	 */
	function nature_bliss_get_image_sizes_options( $add_disable = true, $allowed = array(), $show_dimension = true ) {

		global $_wp_additional_image_sizes;
		$get_intermediate_image_sizes = get_intermediate_image_sizes();
		$choices = array();
		if ( true === $add_disable ) {
			$choices['disable'] = esc_html__( 'No Image', 'nature-bliss-pro' );
		}
		$choices['thumbnail'] = esc_html__( 'Thumbnail', 'nature-bliss-pro' );
		$choices['medium']    = esc_html__( 'Medium', 'nature-bliss-pro' );
		$choices['large']     = esc_html__( 'Large', 'nature-bliss-pro' );
		$choices['full']      = esc_html__( 'Full (original)', 'nature-bliss-pro' );

		if ( true === $show_dimension ) {
			foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
				$choices[ $_size ] = $choices[ $_size ] . ' (' . get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
			}
		}

		if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {
			foreach ( $_wp_additional_image_sizes as $key => $size ) {
				$choices[ $key ] = $key;
				if ( true === $show_dimension ) {
					$choices[ $key ] .= ' (' . $size['width'] . 'x' . $size['height'] . ')';
				}
			}
		}

		if ( ! empty( $allowed ) ) {
			foreach ( $choices as $key => $value ) {
				if ( ! in_array( $key, $allowed ) ) {
					unset( $choices[ $key ] );
				}
			}
		}

		return $choices;

	}

endif;


if ( ! function_exists( 'nature_bliss_get_image_alignment_options' ) ) :

	/**
	 * Returns image options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_image_alignment_options() {

		$choices = array(
			'none'   => _x( 'None', 'Alignment', 'nature-bliss-pro' ),
			'left'   => _x( 'Left', 'Alignment', 'nature-bliss-pro' ),
			'center' => _x( 'Center', 'Alignment', 'nature-bliss-pro' ),
			'right'  => _x( 'Right', 'Alignment', 'nature-bliss-pro' ),
		);
		return $choices;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_slider_transition_effects' ) ) :

	/**
	 * Returns the featured slider transition effects.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_slider_transition_effects() {

		$choices = array(
			'fade'       => _x( 'fade', 'Transition Effect', 'nature-bliss-pro' ),
			'fadeout'    => _x( 'fadeout', 'Transition Effect', 'nature-bliss-pro' ),
			'none'       => _x( 'none', 'Transition Effect', 'nature-bliss-pro' ),
			'scrollHorz' => _x( 'scrollHorz', 'Transition Effect', 'nature-bliss-pro' ),
			'tileSlide'  => _x( 'tileSlide', 'Transition Effect', 'nature-bliss-pro' ),
			'tileBlind'  => _x( 'tileBlind', 'Transition Effect', 'nature-bliss-pro' ),
			'flipHorz'   => _x( 'flipHorz', 'Transition Effect', 'nature-bliss-pro' ),
			'flipVert'   => _x( 'flipVert', 'Transition Effect', 'nature-bliss-pro' ),
			'shuffle'    => _x( 'shuffle', 'Transition Effect', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_slider_transition_effects', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_slider_content_options' ) ) :

	/**
	 * Returns the featured slider content options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_slider_content_options() {

		$choices = array(
			'home-page' => esc_html__( 'Static Front Page Only', 'nature-bliss-pro' ),
			'disabled'  => esc_html__( 'Disabled', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_slider_content_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_slider_type' ) ) :

	/**
	 * Returns the featured slider type.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_slider_type() {

		$choices = array(
			'featured-page'     => __( 'Featured Pages', 'nature-bliss-pro' ),
			'featured-category' => __( 'Featured Category', 'nature-bliss-pro' ),
			'featured-post'     => __( 'Featured Posts', 'nature-bliss-pro' ),
			'featured-image'    => __( 'Featured Images', 'nature-bliss-pro' ),
			'featured-tag'      => __( 'Featured Tag', 'nature-bliss-pro' ),
			'demo-slider'       => __( 'Demo Slider', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_slider_type', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_content_status_options' ) ) :

	/**
	 * Returns the featured content options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_content_status_options() {

		$choices = array(
			'home-page' => esc_html__( 'Home Page Only', 'nature-bliss-pro' ),
			'disabled'  => esc_html__( 'Disabled', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_content_status_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_content_type' ) ) :

	/**
	 * Returns the featured content type.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_content_type() {

		$choices = array(
			'featured-page' => esc_html__( 'Featured Pages', 'nature-bliss-pro' ),
			'featured-post' => esc_html__( 'Featured Posts', 'nature-bliss-pro' ),
			'demo-content'  => esc_html__( 'Demo Content', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_content_type', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if( ! function_exists( 'nature_bliss_get_customizer_font_options' ) ) :

	/**
	 * Returns customizer font options.
	 *
	 * @since 1.0.0
	 */
	function nature_bliss_get_customizer_font_options(){

		$web_fonts = nature_bliss_get_web_fonts();
		$os_fonts  = nature_bliss_get_os_fonts();

		$web_fonts = array_merge( $web_fonts, $os_fonts );

		if ( ! empty( $web_fonts ) ) {
			ksort( $web_fonts );
		}

		$choices = array();

		if ( ! empty( $web_fonts ) ) {
			foreach ( $web_fonts as $k => $v ) {
				$choices[$k] = esc_html( $v['label'] );
			}
		}
		return $choices;

	}

endif;

if( ! function_exists( 'nature_bliss_get_web_fonts' ) ) :

	/**
	 * Returns web font options.
	 *
	 * @since 1.0.0
	 */
	function nature_bliss_get_web_fonts(){

		$choices = array(
			'philosopher' => array(
				'name'  => 'Philosopher',
				'label' => "'Philosopher', sans-serif",
				),
			'open-sans' => array(
				'name'  => 'Open Sans',
				'label' => "'Open Sans', sans-serif",
				),
			'pt-sans' => array(
				'name'  => 'PT Sans',
				'label' => "'PT Sans', sans-serif",
				),
			'merriweather-sans' => array(
				'name'  => 'Merriweather Sans',
				'label' => "'Merriweather Sans', sans-serif",
				),
			'montserrat' => array(
				'name'  => 'Montserrat',
				'label' => "'Montserrat', sans-serif",
				),
			'roboto' => array(
				'name'  => 'Roboto',
				'label' => "'Roboto', sans-serif",
				),
			'arizonia' => array(
				'name'  => 'Arizonia',
				'label' => "'Arizonia', cursive",
				),
			'raleway' => array(
				'name'  => 'Raleway',
				'label' => "'Raleway', sans-serif",
				),
			'droid-sans' => array(
				'name'  => 'Droid Sans',
				'label' => "'Droid Sans', sans-serif",
				),
			'lato' => array(
				'name'  => 'Lato',
				'label' => "'Lato', sans-serif",
				),
			'dosis' => array(
				'name'  => 'Dosis',
				'label' => "'Dosis', sans-serif",
				),
			'slabo-27px' => array(
				'name'  => 'Slabo 27px',
				'label' => "'Slabo 27px', serif",
				),
			'oswald' => array(
				'name'  => 'Oswald',
				'label' => "'Oswald', sans-serif",
				),
			'pt-sans-narrow' => array(
				'name'  => 'PT Sans Narrow',
				'label' => "'PT Sans Narrow', sans-serif",
				),
			'josefin-slab' => array(
				'name'  => 'Josefin Slab',
				'label' => "'Josefin Slab', serif",
				),
			'alegreya' => array(
				'name'  => 'Alegreya',
				'label' => "'Alegreya', serif",
				),
			'exo' => array(
				'name'  => 'Exo',
				'label' => "'Exo', sans-serif",
				),
			'signika' => array(
				'name'  => 'Signika',
				'label' => "'Signika', sans-serif",
				),
			'lobster' => array(
				'name'  => 'Lobster',
				'label' => "'Lobster', cursive",
				),
			'indie-flower' => array(
				'name'  => 'Indie Flower',
				'label' => "'Indie Flower', cursive",
				),
			'shadows-into-light' => array(
				'name'  => 'Shadows Into Light',
				'label' => "'Shadows Into Light', cursive",
				),
			'kaushan-script' => array(
				'name'  => 'Kaushan Script',
				'label' => "'Kaushan Script', cursive",
				),
			'dancing-script' => array(
				'name'  => 'Dancing Script',
				'label' => "'Dancing Script', cursive",
				),
			'fredericka-the-great' => array(
				'name'  => 'Fredericka the Great',
				'label' => "'Fredericka the Great', cursive",
				),
			'covered-by-your-grace' => array(
				'name'  => 'Covered By Your Grace',
				'label' => "'Covered By Your Grace', cursive",
				),
			);
		$choices = apply_filters( 'nature_bliss_filter_web_fonts', $choices );

		if ( ! empty( $choices ) ) {
			ksort( $choices );
		}

		return $choices;

	}

endif;

if( ! function_exists( 'nature_bliss_get_os_fonts' ) ) :

	/**
	 * Returns OS font options.
	 *
	 * @since 1.0.0
	 */
	function nature_bliss_get_os_fonts(){

		$choices = array(
			'arial' => array(
				'name'  => 'Arial',
				'label' => "'Arial', sans-serif",
				),
			'georgia' => array(
				'name'  => 'Georgia',
				'label' => "'Georgia', serif",
				),
			'cambria' => array(
				'name'  => 'Cambria',
				'label' => "'Cambria', Georgia, serif",
				),
			'tahoma' => array(
				'name'  => 'Tahoma',
				'label' => "'Tahoma', Geneva, sans-serif",
				),
			'sans-serif' => array(
				'name'  => 'Sans Serif',
				'label' => "'Sans Serif', Arial",
				),
			'verdana' => array(
				'name'  => 'Verdana',
				'label' => "'Verdana', Geneva, sans-serif",
				),
			);
		$choices = apply_filters( 'nature_bliss_filter_os_fonts', $choices );

		if ( ! empty( $choices ) ) {
			ksort( $choices );
		}
		return $choices;

	}

endif;

if( ! function_exists( 'nature_bliss_get_font_family_from_key' ) ) :

	/**
	 * Return font family from font slug.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key Font slug.
	 * @return string Font name.
	 */
	function nature_bliss_get_font_family_from_key( $key ){

		$output = '';

		$web_fonts = nature_bliss_get_web_fonts();
		$os_fonts  = nature_bliss_get_os_fonts();

		$fonts = array_merge( $web_fonts, $os_fonts );

		if ( isset( $fonts[ $key ] ) ) {
			$output = $fonts[ $key ]['label'];
		}
		return $output;

	}

endif;

if( ! function_exists( 'nature_bliss_get_font_family_theme_settings_options' ) ) :

	/**
	 * Returns font family theme settings options.
	 *
	 * @since 1.0.0
	 */
	function nature_bliss_get_font_family_theme_settings_options(){

		$choices = array(
			'font_site_title' => array(
				'label'   => __( 'Site Title', 'nature-bliss-pro' ),
				'default' => 'montserrat',
				),
			'font_site_tagline' => array(
				'label'   => __( 'Site Tagline', 'nature-bliss-pro' ),
				'default' => 'roboto',
				),
			'font_site_default' => array(
				'label'   => __( 'Default', 'nature-bliss-pro' ),
				'default' => 'roboto',
				),

			'font_content_title' => array(
				'label'   => __( 'Content Title', 'nature-bliss-pro' ),
				'default' => 'montserrat',
				),
			'font_content_body' => array(
				'label'   => __( 'Content Body', 'nature-bliss-pro' ),
				'default' => 'roboto',
				),
			'font_heading_tags' => array(
				'label'   => __( 'Heading Tags', 'nature-bliss-pro' ),
				'default' => 'montserrat',
				),
			'font_navigation' => array(
				'label'   => __( 'Navigation', 'nature-bliss-pro' ),
				'default' => 'montserrat',
				),
			);
		return $choices;

	}

endif;

if( ! function_exists( 'nature_bliss_get_default_colors' ) ) :

  /**
   * Returns default colors.
   *
   * @since 1.0.0
   *
   * @param string $scheme Color scheme.
   * @return array Color values based on scheme.
   */
	function nature_bliss_get_default_colors( $scheme = 'default' ){

		$output = array();

		switch ( $scheme ) {

			case 'default':
			default:
			$output = array(
				// Basic.
				'color_basic_text'                    => '#666666',
				'color_basic_link'                    => '#568c11',
				'color_basic_link_hover'              => '#9cc900',
				'color_basic_heading'                 => '#666666',
				'color_basic_button_background'       => '#568c11',
				'color_basic_button_text'             => '#ffffff',
				'color_basic_button_background_hover' => '#9cc900',
				'color_basic_button_text_hover'       => '#ffffff',
			
				// Top Header.
				'color_top_header_background'                    	 => '#fbfbfb',
				'color_top_header_icon'                        		 => '#9cc900',
				'color_top_header_text'                       		 => '#568c11',
				'color_top_header_link'                       		 => '#568c11',
				'color_top_header_link_hover'                        => '#568c11',

				// Header.
				'color_header_background'                     => '#ffffff',
				'color_header_title'                          => '#568c11',
				'color_header_title_hover'                    => '#9cc900',
				'color_header_tagline'                        => '#4d4d4d',

				// Primary Menu.
				'color_primary_menu_link'                => '#222222',
				'color_primary_menu_link_hover'          => '#9cc900',
				'color_primary_submenu_background'       => '#ffffff',

				// Slider.
				'color_slider_overlay'                 => '#222222',
				'color_slider_caption_text'            => '#ffffff',
				'color_slider_caption_link'            => '#ffffff',
				'color_slider_caption_link_hover'      => '#ffda44',
				'color_slider_icon'                    => '#ffffff',
				'color_slider_icon_hover'              => '#ffffff',
				'color_slider_icon_background'         => '#568c11',
				'color_slider_icon_background_hover'   => '#9cc900',
				'color_slider_pager'                   => '#ffffff',
				'color_slider_pager_active'            => '#9cc900',
				'color_slider_button'                  => '#ffffff',
				'color_slider_button_hover'            => '#ffffff',
				'color_slider_button_background'       => '#568c11',
				'color_slider_button_background_hover' => '#9cc900',

				// Content.
				'color_content_background'     			    => '#ffffff',
				'color_content_title'                       => '#666666',
				'color_content_title_link_hover'            => '#666666',
				'color_content_text'            			=> '#666666',
				'color_content_link'            			=> '#568c11',
				'color_content_link_hover'      			=> '#9cc900',
				'color_content_meta_link'       			=> '#49616b',
				'color_content_meta_link_hover' 			=> '#9cc900',
				'color_content_meta_icon'       			=> '#666666',

				// Breadcrumb.
				'color_breadcrumb_background' => '#fbfbfb',
				'color_breadcrumb_link'       => '#222222',
				'color_breadcrumb_link_hover' => '#9cc900',
				'color_breadcrumb_text'       => '#9cc900',

				// Sidebar.
				'color_sidebar_title'      			  => '#ffffff',
				'color_sidebar_title_background'      => '#9cc900',

				'color_sidebar_link'       => '#666666',
				'color_sidebar_link_hover' => '#9cc900',
				'color_sidebar_list_icon'  => '#49616b',

				// Home Page Widgets.
				'color_home_widgets_background' => '#ffffff',
				'color_home_widgets_alternate_background' => '#f5f5f5',
				'color_home_widgets_title'      => '#666666',
				'color_home_widgets_subtitle'   => '#aaaaaa',
				'color_home_widgets_separator'  => '#ffda44',
				'color_home_widgets_text'       => '#666666',
				'color_home_widgets_link'       => '#568c11',
				'color_home_widgets_link_hover' => '#9cc900',

				// Footer Widgets.
				'color_footer_widgets_background' => '#000000',
				'color_footer_widgets_title'      => '#ffffff',
				'color_footer_widgets_separator'   => '#49616b',
				'color_footer_widgets_text'       => '#dddddd',
				'color_footer_widgets_link'       => '#dddddd',
				'color_footer_widgets_link_hover' => '#ffffff',


				// Footer area.
				'color_footer_area_background' => '#171717',
				'color_footer_area_text'       => '#ffffff',
				'color_footer_area_link'       => '#9cc900',
				'color_footer_area_link_hover' => '#ffffff',

				// Go To Top.
				'color_goto_top_icon'             => '#ffffff',
				'color_goto_top_icon_hover'       => '#ffffff',
				'color_goto_top_background'       => '#568c11',
				'color_goto_top_background_hover' => '#9cc900',

				// Pagination.
				'color_pagination_link'                  => '#666666',
				'color_pagination_link_hover'            => '#ffffff',
				'color_pagination_link_background'       => '#ffffff',
				'color_pagination_link_background_hover' => '#568c11',

				// page_header.
				'color_page_header_background' => '#9cc900',
				'color_page_header_title'      => '#ffffff',



			);
			break;

		} // End switch.

	return $output;

	}

endif;


if( ! function_exists( 'nature_bliss_get_color_theme_settings_options' ) ) :

  /**
   * Returns color theme settings options.
   *
   * @since 1.0.0
   */
  function nature_bliss_get_color_theme_settings_options(){

  	$choices = array(

		// Basic.
  		'color_basic_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_heading' => array(
  			'label'   => __( 'Heading Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),

  		'color_basic_button_text' => array(
  			'label'   => __( 'Button Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_button_background' => array(
  			'label'   => __( 'Button Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_button_text_hover' => array(
  			'label'   => __( 'Button Text Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),
  		'color_basic_button_background_hover' => array(
  			'label'   => __( 'Button Background Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_basic',
  			),

  		// Top Header.
  		'color_top_header_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_top_header',
  			),

  		'color_top_header_icon' => array(
  			'label'   => __( 'Icon Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_top_header',
  		),
  		'color_top_header_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_top_header',
  		),
  		'color_top_header_link' => array(
  			'label'   => __( 'Links Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_top_header',
  		),
		'color_top_header_link_hover' => array(
  			'label'   => __( 'Links Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_top_header',
  		),

		// Header.
  		'color_header_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_header',
  			),
  		'color_header_title' => array(
  			'label'   => __( 'Site Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_header',
  			),
  		'color_header_title_hover' => array(
  			'label'   => __( 'Site Title Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_header',
  			),
  		'color_header_tagline' => array(
  			'label'   => __( 'Site Tagline Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_header',
  			),

		// Primary Menu.
  		'color_primary_menu_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_primary_menu',
  			),
  		'color_primary_menu_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_primary_menu',
  			),
  		'color_primary_submenu_background' => array(
  			'label'   => __( 'Submenu Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_primary_menu',
  			),


		// Content.
  		'color_content_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_title' => array(
  			'label'   => __( 'Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_title_link_hover' => array(
  			'label'   => __( 'Title Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_meta_link' => array(
  			'label'   => __( 'Meta Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_meta_link_hover' => array(
  			'label'   => __( 'Meta Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
  		'color_content_meta_icon' => array(
  			'label'   => __( 'Meta Icon Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_content',
  			),
		// Breadcrumb.
  		'color_breadcrumb_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_breadcrumb',
  			),
  		'color_breadcrumb_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_breadcrumb',
  			),
  		'color_breadcrumb_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_breadcrumb',
  			),
  		'color_breadcrumb_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_breadcrumb',
  			),
		// Slider.
  		'color_slider_overlay' => array(
			'label'   => __( 'Overlay Color', 'nature-bliss-pro' ),
			'section' => 'color_section_slider',
  			),
  		'color_slider_caption_text' => array(
  			'label'   => __( 'Caption Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_caption_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_caption_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_icon' => array(
  			'label'   => __( 'Icon Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_icon_hover' => array(
  			'label'   => __( 'Icon Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_icon_background' => array(
  			'label'   => __( 'Icon Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_icon_background_hover' => array(
  			'label'   => __( 'Icon Background Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_pager' => array(
  			'label'   => __( 'Pager Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_pager_active' => array(
  			'label'   => __( 'Pager Active Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_button' => array(
  			'label'   => __( 'Button Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_button_hover' => array(
  			'label'   => __( 'Button Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_button_background' => array(
  			'label'   => __( 'Button Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),
  		'color_slider_button_background_hover' => array(
  			'label'   => __( 'Button Background Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_slider',
  			),

		// Home Page Widgets.
  		'color_home_widgets_background' => array(
  			'label'   => __( 'Widget Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_alternate_background' => array(
  			'label'   => __( 'Alternate Widget Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_title' => array(
  			'label'   => __( 'Widget Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_subtitle' => array(
  			'label'   => __( 'Widget Subtitle Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_separator' => array(
  			'label'   => __( 'Separator Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),
  		'color_home_widgets_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_home_widgets',
  			),

		// Sidebar.
		'color_sidebar_title_background' => array(
  			'label'   => __( 'Title Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_sidebar',
		),
  		'color_sidebar_title' => array(
  			'label'   => __( 'Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_sidebar',
		),

  		'color_sidebar_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_sidebar',
		),
  		'color_sidebar_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_sidebar',
		),
  		'color_sidebar_list_icon' => array(
  			'label'   => __( 'List Icon Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_sidebar',
		),

		// Footer area.
  		'color_footer_area_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_area',
  			),
  		'color_footer_area_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_area',
  			),
  		'color_footer_area_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_area',
  			),
  		'color_footer_area_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_area',
  			),

		// Go To Top.
  		'color_goto_top_icon' => array(
  			'label'   => __( 'Icon Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_goto_top',
  			),
  		'color_goto_top_icon_hover' => array(
  			'label'   => __( 'Icon Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_goto_top',
  			),
  		'color_goto_top_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_goto_top',
  			),
  		'color_goto_top_background_hover' => array(
  			'label'   => __( 'Background Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_goto_top',
  			),

		// Pagination.
  		'color_pagination_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_pagination',
  			),
  		'color_pagination_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_pagination',
  			),
  		'color_pagination_link_background' => array(
  			'label'   => __( 'Link Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_pagination',
  			),
  		'color_pagination_link_background_hover' => array(
  			'label'   => __( 'Link Background Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_pagination',
  			),

		// page_header.
  		'color_page_header_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_page_header',
  			),
  		'color_page_header_title' => array(
  			'label'   => __( 'Page Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_page_header',
  			),



		// Footer widgets.
  		'color_footer_widgets_background' => array(
  			'label'   => __( 'Background Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),
  		'color_footer_widgets_title' => array(
  			'label'   => __( 'Title Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),
  		'color_footer_widgets_separator' => array(
  			'label'   => __( 'Separator Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),

  		'color_footer_widgets_text' => array(
  			'label'   => __( 'Text Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),
  		'color_footer_widgets_link' => array(
  			'label'   => __( 'Link Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),
  		'color_footer_widgets_link_hover' => array(
  			'label'   => __( 'Link Hover Color', 'nature-bliss-pro' ),
  			'section' => 'color_section_footer_widgets',
  			),

  		);

    return $choices;

  }

endif;

if ( ! function_exists( 'nature_bliss_get_color_panels_options' ) ) :

  /**
   * Returns color panels options.
   *
   * @since 1.0.0
   */
	function nature_bliss_get_color_panels_options() {

		$choices = array(
			'color_panel_main' => array(
				'label' => __( 'Color Options', 'nature-bliss-pro' ),
				),
			);

		return $choices;
	}

endif;
if ( ! function_exists( 'nature_bliss_get_color_sections_options' ) ) :

  /**
   * Returns color sections options.
   *
   * @since 1.0.0
   */
	function nature_bliss_get_color_sections_options() {

		$choices = array(
			'color_section_basic' => array(
				'label' => __( 'Basic Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_top_header' => array(
				'label' => __( 'Top Header Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_header' => array(
				'label' => __( 'Header Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_page_header' => array(
				'label' => __( 'Page Header Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_breadcrumb' => array(
				'label' => __( 'Breadcrumb Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			
			'color_section_primary_menu' => array(
				'label' => __( 'Primary Menu Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_slider' => array(
				'label' => __( 'Slider Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_home_widgets' => array(
				'label' => __( 'Home Widget Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_content' => array(
				'label' => __( 'Content Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_sidebar' => array(
				'label' => __( 'Sidebar Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_goto_top' => array(
				'label' => __( 'Go To Top Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_pagination' => array(
				'label' => __( 'Pagination Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_footer_widgets' => array(
				'label' => __( 'Footer Widgets Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_footer_contact' => array(
				'label' => __( 'Footer Contact Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			'color_section_footer_area' => array(
				'label' => __( 'Footer Area Color Options', 'nature-bliss-pro' ),
				'panel' => 'color_panel_main',
				),
			);

		return $choices;
	}

endif;

if ( ! function_exists( 'nature_bliss_get_featured_content_column_options' ) ) :

	/**
	 * Returns the featured content column.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_featured_content_column_options() {

		$choices = array(
			'3' => sprintf( esc_html__( '%d Columns', 'nature-bliss-pro' ), 3 ),
			'4' => sprintf( esc_html__( '%d Columns', 'nature-bliss-pro' ), 4 ),
		);
		$output = apply_filters( 'nature_bliss_filter_featured_content_column_options', $choices );
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_numbers_dropdown_options' ) ) :

	/**
	 * Returns numbers dropdown options.
	 *
	 * @since 1.0.0
	 *
	 * @param int $min Min.
	 * @param int $max Max.
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_numbers_dropdown_options( $min = 1, $max = 4 ) {

		$output = array();

		if ( $min <= $max ) {
			for ( $i = $min; $i <= $max; $i++ ) {
				$output[ $i ] = $i;
			}
		}

		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_widget_animation_fields' ) ) :

	/**
	 * Returns widget animation fields.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_widget_animation_fields() {

		$output = array();

		$output['animation_heading'] = array(
			'label' => __( 'ANIMATION OPTIONS', 'nature-bliss-pro' ),
			'type'  => 'heading',
			'class' => 'widefat',
			);
		$output['animation_effect'] = array(
			'label'       => __( 'Effect:', 'nature-bliss-pro' ),
			'description' => __( 'Choose an animation style.', 'nature-bliss-pro' ),
			'type'        => 'select',
			'options'     => nature_bliss_get_wow_animation_effects_options(),
			);
		$output['animation_duration'] = array(
			'label'       => __( 'Duration:', 'nature-bliss-pro' ),
			'description' => __( 'Change animation duration.', 'nature-bliss-pro' ),
			'type'        => 'select',
			'options'     => nature_bliss_get_wow_animation_duration_options(),
			);
		$output['animation_delay'] = array(
			'label'       => __( 'Delay:', 'nature-bliss-pro' ),
			'description' => __( 'Delay before the animation starts', 'nature-bliss-pro' ),
			'type'        => 'select',
			'options'     => nature_bliss_get_wow_animation_delay_options(),
			);
		$output['animation_iteration'] = array(
			'label'       => __( 'Iteration:', 'nature-bliss-pro' ),
			'description' => __( 'Number of times the animation is repeated', 'nature-bliss-pro' ),
			'type'        => 'select',
			'options'     => nature_bliss_get_wow_animation_iteration_options(),
			);
		$output['animation_offset'] = array(
			'label'       => __( 'Offset:', 'nature-bliss-pro' ),
			'description' => __( 'Distance in pixel to start the animation', 'nature-bliss-pro' ),
			'type'        => 'text',
			'css'         => 'max-width:60px;',
			);

		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_wow_animation_effects_options' ) ) :

	/**
	 * Returns WOW animation effects options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_wow_animation_effects_options() {

		$choices = array(
			'bounce'             => _x( 'bounce', 'Animation Effect', 'nature-bliss-pro' ),
			'flash'              => _x( 'flash', 'Animation Effect', 'nature-bliss-pro' ),
			'pulse'              => _x( 'pulse', 'Animation Effect', 'nature-bliss-pro' ),
			'rubberBand'         => _x( 'rubberBand', 'Animation Effect', 'nature-bliss-pro' ),
			'shake'              => _x( 'shake', 'Animation Effect', 'nature-bliss-pro' ),
			'headShake'          => _x( 'headShake', 'Animation Effect', 'nature-bliss-pro' ),
			'swing'              => _x( 'swing', 'Animation Effect', 'nature-bliss-pro' ),
			'tada'               => _x( 'tada', 'Animation Effect', 'nature-bliss-pro' ),
			'wobble'             => _x( 'wobble', 'Animation Effect', 'nature-bliss-pro' ),
			'jello'              => _x( 'jello', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceIn'           => _x( 'bounceIn', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceInDown'       => _x( 'bounceInDown', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceInLeft'       => _x( 'bounceInLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceInRight'      => _x( 'bounceInRight', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceInUp'         => _x( 'bounceInUp', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceOut'          => _x( 'bounceOut', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceOutDown'      => _x( 'bounceOutDown', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceOutLeft'      => _x( 'bounceOutLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceOutRight'     => _x( 'bounceOutRight', 'Animation Effect', 'nature-bliss-pro' ),
			'bounceOutUp'        => _x( 'bounceOutUp', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeIn'             => _x( 'fadeIn', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInDown'         => _x( 'fadeInDown', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInDownBig'      => _x( 'fadeInDownBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInLeft'         => _x( 'fadeInLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInLeftBig'      => _x( 'fadeInLeftBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInRight'        => _x( 'fadeInRight', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInRightBig'     => _x( 'fadeInRightBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInUp'           => _x( 'fadeInUp', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeInUpBig'        => _x( 'fadeInUpBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOut'            => _x( 'fadeOut', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutDown'        => _x( 'fadeOutDown', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutDownBig'     => _x( 'fadeOutDownBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutLeft'        => _x( 'fadeOutLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutLeftBig'     => _x( 'fadeOutLeftBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutRight'       => _x( 'fadeOutRight', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutRightBig'    => _x( 'fadeOutRightBig', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutUp'          => _x( 'fadeOutUp', 'Animation Effect', 'nature-bliss-pro' ),
			'fadeOutUpBig'       => _x( 'fadeOutUpBig', 'Animation Effect', 'nature-bliss-pro' ),
			'flipInX'            => _x( 'flipInX', 'Animation Effect', 'nature-bliss-pro' ),
			'flipInY'            => _x( 'flipInY', 'Animation Effect', 'nature-bliss-pro' ),
			'flipOutX'           => _x( 'flipOutX', 'Animation Effect', 'nature-bliss-pro' ),
			'flipOutY'           => _x( 'flipOutY', 'Animation Effect', 'nature-bliss-pro' ),
			'lightSpeedIn'       => _x( 'lightSpeedIn', 'Animation Effect', 'nature-bliss-pro' ),
			'lightSpeedOut'      => _x( 'lightSpeedOut', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateIn'           => _x( 'rotateIn', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateInDownLeft'   => _x( 'rotateInDownLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateInDownRight'  => _x( 'rotateInDownRight', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateInUpLeft'     => _x( 'rotateInUpLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateInUpRight'    => _x( 'rotateInUpRight', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateOut'          => _x( 'rotateOut', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateOutDownLeft'  => _x( 'rotateOutDownLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateOutDownRight' => _x( 'rotateOutDownRight', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateOutUpLeft'    => _x( 'rotateOutUpLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'rotateOutUpRight'   => _x( 'rotateOutUpRight', 'Animation Effect', 'nature-bliss-pro' ),
			'hinge'              => _x( 'hinge', 'Animation Effect', 'nature-bliss-pro' ),
			'rollIn'             => _x( 'rollIn', 'Animation Effect', 'nature-bliss-pro' ),
			'rollOut'            => _x( 'rollOut', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomIn'             => _x( 'zoomIn', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomInDown'         => _x( 'zoomInDown', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomInLeft'         => _x( 'zoomInLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomInRight'        => _x( 'zoomInRight', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomInUp'           => _x( 'zoomInUp', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomOut'            => _x( 'zoomOut', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomOutDown'        => _x( 'zoomOutDown', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomOutLeft'        => _x( 'zoomOutLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomOutRight'       => _x( 'zoomOutRight', 'Animation Effect', 'nature-bliss-pro' ),
			'zoomOutUp'          => _x( 'zoomOutUp', 'Animation Effect', 'nature-bliss-pro' ),
			'slideInDown'        => _x( 'slideInDown', 'Animation Effect', 'nature-bliss-pro' ),
			'slideInLeft'        => _x( 'slideInLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'slideInRight'       => _x( 'slideInRight', 'Animation Effect', 'nature-bliss-pro' ),
			'slideInUp'          => _x( 'slideInUp', 'Animation Effect', 'nature-bliss-pro' ),
			'slideOutDown'       => _x( 'slideOutDown', 'Animation Effect', 'nature-bliss-pro' ),
			'slideOutLeft'       => _x( 'slideOutLeft', 'Animation Effect', 'nature-bliss-pro' ),
			'slideOutRight'      => _x( 'slideOutRight', 'Animation Effect', 'nature-bliss-pro' ),
			'slideOutUp'         => _x( 'slideOutUp', 'Animation Effect', 'nature-bliss-pro' ),
		);
		$output = apply_filters( 'nature_bliss_filter_wow_animation_effects_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		$output = array( 'no-animation' => __( 'No Animation', 'nature-bliss-pro' ) ) + $output;
		return $output;

	}

endif;

if ( ! function_exists( 'nature_bliss_get_wow_animation_duration_options' ) ) :

	/**
	 * Returns WOW animation duration options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_wow_animation_duration_options() {

		$choices = array(
			'' => __( 'Default', 'nature-bliss-pro' ),
			1  => sprintf( __( '%d Second', 'nature-bliss-pro' ), 1 ),
			2  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 2 ),
			3  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 3 ),
			4  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 4 ),
			5  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 5 ),
		);
		$output = apply_filters( 'nature_bliss_filter_wow_animation_duration_options', $choices );
		return $output;
	}

endif;

if ( ! function_exists( 'nature_bliss_get_wow_animation_delay_options' ) ) :

	/**
	 * Returns WOW animation delay options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_wow_animation_delay_options() {

		$choices = array(
			'' => __( 'Default', 'nature-bliss-pro' ),
			1  => sprintf( __( '%d Second', 'nature-bliss-pro' ), 1 ),
			2  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 2 ),
			3  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 3 ),
			4  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 4 ),
			5  => sprintf( __( '%d Seconds', 'nature-bliss-pro' ), 5 ),
		);
		$output = apply_filters( 'nature_bliss_filter_wow_animation_delay_options', $choices );
		return $output;
	}

endif;

if ( ! function_exists( 'nature_bliss_get_wow_animation_iteration_options' ) ) :

	/**
	 * Returns WOW animation iteration options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function nature_bliss_get_wow_animation_iteration_options() {

		$choices = array(
			1 => __( 'Don\'t Repeat', 'nature-bliss-pro' ),
			2 => sprintf( __( 'Repeat %dx', 'nature-bliss-pro' ), 2 ),
			3 => sprintf( __( 'Repeat %dx', 'nature-bliss-pro' ), 3 ),
			4 => sprintf( __( 'Repeat %dx', 'nature-bliss-pro' ), 4 ),
			5 => sprintf( __( 'Repeat %dx', 'nature-bliss-pro' ), 5 ),
		);
		$output = apply_filters( 'nature_bliss_filter_wow_animation_iteration_options', $choices );
		return $output;
	}

endif;
