<?php
/**
 * Theme widgets.
 *
 * @package Nature_Bliss
 */

// Load widget base.
require_once get_template_directory() . '/lib/widget-base/class-widget-base.php';

if ( ! function_exists( 'nature_bliss_load_widgets' ) ) :

	/**
	 * Load widgets.
	 *
	 * @since 1.0.0
	 */
	function nature_bliss_load_widgets() {

		// Social widget.
		register_widget( 'Nature_Bliss_Social_Widget' );

		// Featured Page widget.
		register_widget( 'Nature_Bliss_Featured_Page_Widget' );

		// Latest News widget.
		register_widget( 'Nature_Bliss_Latest_News_Widget' );

		// Call To Action widget.
		register_widget( 'Nature_Bliss_Call_To_Action_Widget' );

		// Services widget.
		register_widget( 'Nature_Bliss_Services_Widget' );

		// Testimonial slider widget.
		register_widget( 'Nature_Bliss_Testimonial_Slider_Widget' );

		// Recent Posts widget.
		register_widget( 'Nature_Bliss_Recent_Posts_Widget' );

		// Featured Pages Grid widget.
		register_widget( 'Nature_Bliss_Featured_Pages_Grid_Widget' );

		// News and Events widget.
		register_widget( 'Nature_Bliss_News_Events_Widget' );

		// Latest Features widget.
		register_widget( 'Nature_Bliss_Features_Widget' );

		// Recent Works widget.
		register_widget( 'Nature_Bliss_Recent_Works_Widget' );

		if ( class_exists( 'WooCommerce' ) ) {

			// Products Grid widget.
			register_widget( 'Nature_Bliss_Products_Grid_Widget' );
		}
	}

endif;

add_action( 'widgets_init', 'nature_bliss_load_widgets' );

if ( ! class_exists( 'Nature_Bliss_Social_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Social_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_social',
				'description'                 => __( 'Displays social icons.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				);

			if ( false === has_nav_menu( 'social' ) ) {
				$fields['message'] = array(
					'label' => __( 'Social menu is not set. Please create menu and assign it to Social Menu.', 'nature-bliss-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					);
			}

			parent::__construct( 'nature-bliss-social', __( 'NB: Social', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			if ( has_nav_menu( 'social' ) ) {
				wp_nav_menu( array(
					'theme_location' => 'social',
					'container'      => false,
					'depth'          => 1,
					'link_before'    => '<span class="screen-reader-text">',
					'link_after'     => '</span>',
				) );
			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Featured_Page_Widget' ) ) :

	/**
	 * Featured page widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Featured_Page_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_featured_page',
				'description'                 => __( 'Displays single featured Page or Post.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'use_page_title' => array(
					'label'   => __( 'Use Page/Post Title as Widget Title', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'featured_page' => array(
					'label'            => __( 'Select Page:', 'nature-bliss-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'nature-bliss-pro' ),
					),
				'id_message' => array(
					'label'            => '<strong>' . _x( 'OR', 'Featured Page Widget', 'nature-bliss-pro' ) . '</strong>',
					'type'             => 'message',
					),
				'featured_post' => array(
					'label'             => __( 'Post ID:', 'nature-bliss-pro' ),
					'placeholder'       => __( 'Eg: 1234', 'nature-bliss-pro' ),
					'type'              => 'text',
					'sanitize_callback' => 'nature_bliss_widget_sanitize_post_id',
					),
				'content_type' => array(
					'label'   => __( 'Show Content:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'full',
					'options' => array(
						'excerpt' => __( 'Excerpt', 'nature-bliss-pro' ),
						'full'    => __( 'Full', 'nature-bliss-pro' ),
						),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'Applies when Excerpt is selected in Content option.', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 100,
					'min'         => 1,
					'max'         => 400,
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'options' => nature_bliss_get_image_sizes_options(),
					),
				'featured_image_alignment' => array(
					'label'   => __( 'Image Alignment:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'center',
					'options' => nature_bliss_get_image_alignment_options(),
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-featured-page', __( 'NB: Featured Page', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			$our_id = '';

			if ( absint( $params['featured_post'] ) > 0 ) {
				$our_id = absint( $params['featured_post'] );
			}

			if ( absint( $params['featured_page'] ) > 0 ) {
				$our_id = absint( $params['featured_page'] );
			}

			if ( absint( $our_id ) > 0 ) {
				$qargs = array(
					'p'             => absint( $our_id ),
					'post_type'     => 'any',
					'no_found_rows' => true,
					);

				$the_query = new WP_Query( $qargs );
				if ( $the_query->have_posts() ) {

					while ( $the_query->have_posts() ) {
						$the_query->the_post();

						echo '<div class="featured-page-widget image-align' . esc_attr( $params['featured_image_alignment'] ) . ' entry-content">';

						if ( 'disable' != $params['featured_image'] && has_post_thumbnail() ) {
							the_post_thumbnail( esc_attr( $params['featured_image'] ) );
						}

						echo '<div class="featured-page-content">';

						if ( true === $params['use_page_title'] ) {
							the_title( $args['before_title'], $args['after_title'] );
						} else {
							if ( $params['title'] ) {
								echo $args['before_title'] . $params['title'] . $args['after_title'];
							}
						}

						if ( ! empty( $params['subtitle'] ) ) {
							echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
						}

						if ( 'excerpt' === $params['content_type'] && absint( $params['excerpt_length'] ) > 0 ) {
							$excerpt = nature_bliss_the_excerpt( absint( $params['excerpt_length'] ) );
							echo wp_kses_post( wpautop( $excerpt ) );
							echo '<a href="'  . esc_url( get_permalink() ) . '" class="more-link">' . esc_html__( 'Know More', 'nature-bliss-pro' ) . '</a>';
						} else {
							the_content();
						}

						echo '</div><!-- .featured-page-content -->';
						echo '</div><!-- .featured-page-widget -->';
					}

					wp_reset_postdata();
				}

			}

			echo $args['after_widget'];
		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Latest_News_Widget' ) ) :

	/**
	 * Latest news widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Latest_News_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'nature_bliss_widget_latest_news',
				'description'                 => __( 'Displays latest posts in grid.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => __( 'Number of Columns:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => nature_bliss_get_numbers_dropdown_options( 1, 4 ),
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'nature-bliss-thumb',
					'options' => nature_bliss_get_image_sizes_options(),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'more_text' => array(
					'label'   => __( 'More Text:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => __( 'Know More', 'nature-bliss-pro' ),
					),
				'disable_date' => array(
					'label'   => __( 'Disable Date', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_more_text' => array(
					'label'   => __( 'Disable More Text', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-latest-news', __( 'NB: Latest News', 'nature-bliss-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params               = $this->get_params( $instance );
			$animation_attributes = $this->get_animation_attributes( $params );
			$animation_classes    = $this->get_animation_classes( $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) : ?>

				<?php global $post; ?>

				<div class="latest-news-widget latest-news-col-<?php echo esc_attr( $params['post_column'] ); ?>">

					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $key => $post ) : ?>
							<?php setup_postdata( $post ); ?>

							<div class="latest-news-item <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>

									<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
										<div class="latest-news-thumb">
											<div class="latest-news-meta">
												<?php if ( false === $params['disable_date'] ) : ?>
													<div class="latest-news-date"><span class="news-meta-date"><?php the_time( _x( 'd', 'date format', 'nature-bliss-pro' ) ); ?></span><span class="news-meta-months"><?php the_time( _x( 'M', 'date format', 'nature-bliss-pro' ) ); ?></div>
												<?php endif; ?>
											</div><!-- .latest-news-meta -->
											<a href="<?php the_permalink(); ?>">
												<?php
												$img_attributes = array( 'class' => 'aligncenter' );
												the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
												?>
											</a>
										</div><!-- .latest-news-thumb -->
									<?php endif; ?>
									<div class="latest-news-text-wrap">

										<div class="latest-news-text-content">
											<h3 class="latest-news-title">
												<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
											</h3><!-- .latest-news-title -->

											<?php if ( false === $params['disable_excerpt'] ) :  ?>
												<div class="latest-news-summary">
												<?php
												$summary = nature_bliss_the_excerpt( esc_attr( $params['excerpt_length'] ), $post );
												echo wp_kses_post( wpautop( $summary ) );
												?>
												</div><!-- .latest-news-summary -->
											<?php endif; ?>
										</div><!-- .latest-news-text-content -->

										<?php if ( false === $params['disable_date'] || false === $params['disable_more_text'] ) : ?>

													<?php if ( false === $params['disable_more_text'] ) :  ?>
												<a href="<?php the_permalink(); ?>" class="custom-button"><?php echo esc_html( $params['more_text'] ); ?><span class="screen-reader-text">"<?php the_title(); ?>"</span>
														</a>
													<?php endif; ?>

										<?php endif; ?>

									</div><!-- .latest-news-text-wrap -->

							</div><!-- .latest-news-item -->

						<?php endforeach; ?>
					</div><!-- .row -->

				</div><!-- .latest-news-widget -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Call_To_Action_Widget' ) ) :

	/**
	 * Call to action widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Call_To_Action_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_call_to_action',
				'description'                 => __( 'Call To Action Widget.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'description' => array(
					'label' => __( 'Description:', 'nature-bliss-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
					),
				'primary_button_text' => array(
					'label'   => __( 'Primary Button Text:', 'nature-bliss-pro' ),
					'default' => __( 'Learn more', 'nature-bliss-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'primary_button_url' => array(
					'label' => __( 'Primary Button URL:', 'nature-bliss-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				'secondary_button_text' => array(
					'label'   => __( 'Secondary Button Text:', 'nature-bliss-pro' ),
					'default' => '',
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'secondary_button_url' => array(
					'label' => __( 'Secondary Button URL:', 'nature-bliss-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				'background_image' => array(
					'label'   => __( 'Background Image:', 'nature-bliss-pro' ),
					'type'    => 'image',
					'default' => '',
					),
				'enable_background_overlay' => array(
					'label'   => __( 'Enable Background Overlay', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-call-to-action', __( 'NB: Call To Action', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			// Add background image.
			if ( ! empty( $params['background_image'] ) ) {
				$background_style = '';
				$background_style .= ' style="background-image:url(' . esc_url( $params['background_image'] ) . ');" ';
				$args['before_widget'] = implode( $background_style . ' ' . 'class="', explode( 'class="', $args['before_widget'], 2 ) );
			}

			// Add overlay class.
			$overlay_class = ( true === $params['enable_background_overlay'] ) ? 'overlay-enabled' : 'overlay-disabled' ;
			$args['before_widget'] = implode( 'class="' . $overlay_class . ' ', explode( 'class="', $args['before_widget'], 2 ) );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			?>
			<div class="call-to-action-content">
				<?php if ( ! empty( $params['description'] ) ) : ?>
				    <div class="call-to-action-description">
				        <?php echo wp_kses_post( wpautop( $params['description'] ) ); ?>
				    </div><!-- .call-to-action-description -->
				<?php endif; ?>
				<?php if ( ! empty( $params['primary_button_text'] ) || ! empty( $params['secondary_button_text'] ) ) : ?>
					<div class="call-to-action-buttons">
						<?php if ( ! empty( $params['primary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['primary_button_url'] ); ?>" class="custom-button btn-call-to-action button-primary"><?php echo esc_html( $params['primary_button_text'] ); ?></a>
						<?php endif; ?>
						<?php if ( ! empty( $params['secondary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['secondary_button_url'] ); ?>" class="custom-button btn-call-to-action button-secondary"><?php echo esc_html( $params['secondary_button_text'] ); ?></a>
						<?php endif; ?>
					</div><!-- .call-to-action-buttons -->
				<?php endif; ?>
			</div><!-- .call-to-action-content -->
			<?php

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Services_Widget' ) ) :

	/**
	 * Services widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Services_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_services',
				'description'                 => __( 'Show your services with icon and read more link.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'more_text' => array(
					'label'   => __( 'Read More Text:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => __( 'Know More', 'nature-bliss-pro' ),
					),
				'disable_more_text' => array(
					'label'   => __( 'Disable Read More', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			for( $i = 1; $i <= 6; $i++ ) {
				$fields[ 'block_heading_' . $i ] = array(
					'label' => __( 'Block', 'nature-bliss-pro' ) . ' #' . $i,
					'type'  => 'heading',
					'class' => 'widefat',
					);
				$fields[ 'block_page_' . $i ] = array(
					'label'            => __( 'Select Page:', 'nature-bliss-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'nature-bliss-pro' ),
					);
				$fields[ 'block_icon_' . $i ] = array(
					'label'       => __( 'Icon:', 'nature-bliss-pro' ),
					'description' => __( 'Eg: fa-cogs', 'nature-bliss-pro' ),
					'type'        => 'text',
					'default'     => 'fa-cogs',
					);
			}

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-services', __( 'NB: Services', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$service_arr = array();
			for ( $i = 0; $i < 6 ; $i++ ) {
				$block = ( $i + 1 );
				$service_arr[ $i ] = array(
					'page' => $params[ 'block_page_' . $block ],
					'icon' => $params[ 'block_icon_' . $block ],
					);
			}
			$refined_arr = array();
			if ( ! empty( $service_arr ) ) {
				foreach ( $service_arr as $item ) {
					if ( ! empty( $item['page'] ) ) {
						$refined_arr[ $item['page'] ] = $item;
					}
				}
			}

			if ( ! empty( $refined_arr ) ) {
				$this->render_widget_content( $refined_arr, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Render services content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $service_arr Services array.
		 * @param array $params      Parameters array.
		 */
		function render_widget_content( $service_arr, $params ) {

			$column = count( $service_arr );

			$page_ids = array_keys( $service_arr );

			$qargs = array(
				'posts_per_page' => count( $page_ids ),
				'post__in'       => $page_ids,
				'post_type'      => 'page',
				'orderby'        => 'post__in',
				'no_found_rows'  => true,
			);

			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) : ?>
				<?php
				$animation_attributes = $this->get_animation_attributes( $params );
				$animation_classes    = $this->get_animation_classes( $params );
				?>

				<?php global $post; ?>

				<div class="service-block-list service-col-<?php echo esc_attr( $column ); ?>">
					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $post ) : ?>
							<?php setup_postdata( $post ); ?>
							<div class="service-block-item <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
								<div class="service-block-inner">
									<?php if ( isset( $service_arr[ $post->ID ]['icon'] ) && ! empty( $service_arr[ $post->ID ]['icon'] ) ) : ?>
										<a class="service-icon" href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>"><i class="<?php echo 'fa ' . esc_attr( $service_arr[ $post->ID ]['icon'] ); ?>"></i></a>
									<?php endif; ?>
									<div class="service-block-inner-content">
										<h3 class="service-item-title">
											<a href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>">
												<?php echo get_the_title( $post->ID ); ?>
											</a>
										</h3>
										<?php if ( true !== $params['disable_excerpt'] ) :  ?>
											<div class="service-block-item-excerpt">
												<?php
												$excerpt = nature_bliss_the_excerpt( $params['excerpt_length'], $post );
												echo wp_kses_post( wpautop( $excerpt ) );
												?>
											</div><!-- .service-block-item-excerpt -->
										<?php endif; ?>

										<?php if ( true !== $params['disable_more_text'] ) :  ?>
											<a href="<?php echo esc_url( get_permalink( $post -> ID ) ); ?>" class="more-link"><?php echo esc_html( $params['more_text'] ); ?></a>
										<?php endif; ?>
									</div><!-- .service-block-inner-content -->
								</div><!-- .service-block-inner -->
							</div><!-- .service-block-item -->
						<?php endforeach; ?>

					</div><!-- .inner-wrapper -->

				</div><!-- .service-block-list -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
		}

	}
endif;

if ( ! class_exists( 'Nature_Bliss_Testimonial_Slider_Widget' ) ) :

	/**
	 * Testimonial Slider widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Testimonial_Slider_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'nature_bliss_widget_testimonial_slider',
				'description' => __( 'Displays Testimonials as a Slider.', 'nature-bliss-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 30,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'background_image' => array(
					'label'   => __( 'Background Image:', 'nature-bliss-pro' ),
					'type'    => 'image',
					'default' => '',
					),
				'enable_background_overlay' => array(
					'label'   => __( 'Enable Background Overlay', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'slider_heading' => array(
					'label'   => __( 'SLIDER OPTIONS', 'nature-bliss-pro' ),
					'type'    => 'heading',
					),
				'transition_delay' => array(
					'label'       => __( 'Transition Delay:', 'nature-bliss-pro' ),
					'description' => __( 'in seconds', 'nature-bliss-pro' ),
					'type'        => 'number',
					'default'     => 3,
					'css'         => 'max-width:50px;',
					'min'         => 1,
					'max'         => 10,
					'adjacent'    => true,
					),
				'enable_autoplay' => array(
					'label'   => __( 'Enable Autoplay', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-testimonial-slider', __( 'NB: Testimonial Slider', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			// Add background image.
			if ( ! empty( $params['background_image'] ) ) {
				$background_style = '';
				$background_style .= ' style="background-image:url(' . esc_url( $params['background_image'] ) . ');" ';
				$args['before_widget'] = implode( $background_style . ' ' . 'class="', explode( 'class="', $args['before_widget'], 2 ) );
			}

			// Add overlay class.
			$overlay_class = ( true === $params['enable_background_overlay'] ) ? 'overlay-enabled' : 'overlay-disabled' ;
			$args['before_widget'] = implode( 'class="' . $overlay_class . ' ', explode( 'class="', $args['before_widget'], 2 ) );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$testimonial_posts = $this->get_testimonials( $params );

			if ( ! empty( $testimonial_posts ) ) {
				$this->render_testimonials( $testimonial_posts, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Return testimonial posts detail.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return array Posts details.
		 */
		function get_testimonials( $params ) {

			$output = array();

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$all_posts = get_posts( $qargs );

			if ( ! empty( $all_posts ) ) {
				$cnt = 0;
				global $post;
				foreach ( $all_posts as $key => $post ) {

					setup_postdata( $post );

					$item = array();
					$item['title']   = get_the_title( $post->ID );
					$item['excerpt'] = nature_bliss_the_excerpt( absint( $params['excerpt_length'] ), $post );
					$item['image']   = null;
					if ( has_post_thumbnail( $post->ID ) ) {
						$image_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'thumbnail' );
						if ( ! empty( $image_detail ) ) {
							$item['image'] = $image_detail;
						}
					}

					$output[ $cnt ] = $item;
					$cnt++;

				}

				wp_reset_postdata();
			}

			return $output;

		}

		/**
		 * Render testimonial slider.
		 *
		 * @since 1.0.0
		 *
		 * @param array $testimonials Testimonials.
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_testimonials( $testimonials, $params ) {

			$timeout = 0;
			if ( true === $params['enable_autoplay'] ) {
				$timeout = 1000 * absint( $params['transition_delay'] );
			}
			?>
			<div class="cycle-slideshow"
				data-cycle-slides="> article"
				data-cycle-auto-height="container"
				data-cycle-timeout="<?php echo absint( $timeout ); ?>"
				data-pager-template="<span></span>"
				>
				<div class="cycle-pager"></div>
				<?php foreach ( $testimonials as $testimonial ) : ?>
					<article>
						<?php if ( ! empty( $testimonial['image'] ) ) : ?>
							<div class="testimonial-thumb"><img src="<?php echo esc_url( $testimonial['image'][0] ); ?>" /></div> <!-- .testimonial-thumb -->
						<?php endif; ?>
						<div class="testimonial-content-area">
						<?php if ( ! empty( $testimonial['excerpt'] ) ) : ?>
							<div class="testimonial-excerpt">
								<?php echo wp_kses_post( wpautop( $testimonial['excerpt'] ) ); ?>
							</div><!-- .testimonial-excerpt -->
						<?php endif; ?>
						<h4><?php echo esc_html( $testimonial['title'] ); ?></h4>
						</div> <!-- .testimonial-content-area -->
					</article>

				<?php endforeach; ?>

			</div><!-- .cycle-slideshow -->
			<?php

		}

	}
endif;

if ( ! class_exists( 'Nature_Bliss_Recent_Posts_Widget' ) ) :

	/**
	 * Recent posts widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Recent_Posts_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_recent_posts',
				'description'                 => __( 'Displays recent posts.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'thumbnail',
					'options' => nature_bliss_get_image_sizes_options( true, array( 'disable', 'thumbnail' ), false ),
					),
				'image_width' => array(
					'label'       => __( 'Image Width:', 'nature-bliss-pro' ),
					'type'        => 'number',
					'description' => __( 'px', 'nature-bliss-pro' ),
					'css'         => 'max-width:60px;',
					'adjacent'    => true,
					'default'     => 70,
					'min'         => 1,
					'max'         => 150,
					),
				'disable_date' => array(
					'label'   => __( 'Disable Date', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			parent::__construct( 'nature-bliss-recent-posts', __( 'NB: Recent Posts', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = $params['post_category'];
			}
			$all_posts = get_posts( $qargs );

			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>

				<?php global $post; ?>

				<div class="recent-posts-wrapper">

					<?php foreach ( $all_posts as $key => $post ) : ?>
						<?php setup_postdata( $post ); ?>

						<div class="recent-posts-item">

							<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
								<div class="recent-posts-thumb">
									<a href="<?php the_permalink(); ?>">
										<?php
										$img_attributes = array(
											'class' => 'alignleft',
											'style' => 'max-width:' . esc_attr( $params['image_width'] ). 'px;',
											);
										the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
										?>
									</a>
								</div><!-- .recent-posts-thumb -->
							<?php endif ?>
							<div class="recent-posts-text-wrap">
								<h3 class="recent-posts-title">
									<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
								</h3><!-- .recent-posts-title -->

								<?php if ( false === $params['disable_date'] ) :  ?>
									<div class="recent-posts-meta">

										<?php if ( false === $params['disable_date'] ) :  ?>
											<span class="recent-posts-date"><?php the_time( get_option( 'date_format' ) ); ?></span><!-- .recent-posts-date -->
										<?php endif; ?>

									</div><!-- .recent-posts-meta -->
								<?php endif; ?>

							</div><!-- .recent-posts-text-wrap -->

						</div><!-- .recent-posts-item -->

					<?php endforeach; ?>

				</div><!-- .recent-posts-wrapper -->

				<?php wp_reset_postdata(); // Reset. ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Featured_Pages_Grid_Widget' ) ) :

	/**
	 * Featured pages grid widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Featured_Pages_Grid_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_featured_pages_grid',
				'description'                 => __( 'Displays featured pages in grid.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'nature-bliss-thumb',
					'options' => nature_bliss_get_image_sizes_options(),
					),
				'post_column' => array(
					'label'   => __( 'Number of Columns:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => nature_bliss_get_numbers_dropdown_options( 3, 4 ),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 20,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'more_text' => array(
					'label'   => __( 'Read More Text:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => __( 'Know More', 'nature-bliss-pro' ),
					),
				);

			for( $i = 1; $i <= 8; $i++ ) {
				$fields[ 'block_page_' . $i ] = array(
					'label'            => sprintf( __( 'Page #%d:', 'nature-bliss-pro' ), $i ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'nature-bliss-pro' ),
					);
			}

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-featured-pages-grid', __( 'NB: Featured Pages Grid', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$pages_arr = array();
			for ( $i = 0; $i < 8; $i++ ) {
				$block = ( $i + 1 );
				if ( absint( $params[ 'block_page_' . $block ] ) > 0 ) {
					$pages_arr[] = absint( $params[ 'block_page_' . $block ] );
				}
			}

			if ( ! empty( $pages_arr ) ) {
				$qargs = array(
					'no_found_rows'  => true,
					'orderby'        => 'post__in',
					'post__in'       => $pages_arr,
					'post_type'      => 'page',
					'posts_per_page' => count( $pages_arr ),
				);

				$the_query = new WP_Query( $qargs );

				if ( $the_query->have_posts() ) {

					$animation_attributes = $this->get_animation_attributes( $params );
					$animation_classes    = $this->get_animation_classes( $params );

					echo '<div class="featured-pages-grid grid-column-' . absint( $params['post_column'] ) . '">';
					echo '<div class="inner-wrapper">';

					while ( $the_query->have_posts() ) {
						$the_query->the_post();
						?>
						<div class="grid-item <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
							<div class="grid-item-inner">
								<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
									<div class="grid-item-thumb">
										<a href="<?php the_permalink(); ?>">
											<?php
											$img_attributes = array( 'class' => 'aligncenter' );
											the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
											?>
										</a>
										<h3 class="item-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
									</div><!-- .grid-item-thumb -->
								<?php endif; ?>
								<div class="grid-text-content">

									<?php if ( false === $params['disable_excerpt'] ) :  ?>
										<div class="grid-summary">
										<?php
										$excerpt = nature_bliss_the_excerpt( esc_attr( $params['excerpt_length'] ) );
										echo wp_kses_post( wpautop( $excerpt ) );
										?>
										</div><!-- .grid-summary -->
									<?php endif; ?>

									<?php if ( ! empty( $params['more_text'] ) ) : ?>
										<a href="<?php the_permalink(); ?>" class="custom-button"><?php echo esc_html( $params['more_text'] ); ?><span class="screen-reader-text">"<?php the_title(); ?>"</span>
										</a>
									<?php endif; ?>

								</div><!-- .grid-text-content -->
							</div><!-- .grid-item-inner -->
						</div><!-- .grid-item -->
						<?php
					}

					echo '</div><!-- .inner-wrapper -->';
					echo '</div><!-- .featured-pages-grid -->';

					wp_reset_postdata();
				}

			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_News_Events_Widget' ) ) :

	/**
	 * News events widget class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_News_Events_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'nature_bliss_widget_news_events',
				'description'                 => __( 'Displays news and events.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'news_heading' => array(
					'label' => __( 'News', 'nature-bliss-pro' ),
					'type'  => 'heading',
					),
				'news_section_title' => array(
					'label'   => __( 'Section Title:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
					'default' => __( 'News', 'nature-bliss-pro' ),
					),
				'news_post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'news_post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 2,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'news_featured_image' => array(
					'label'   => __( 'Featured Image:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 'nature-bliss-thumb',
					'options' => nature_bliss_get_image_sizes_options(),
					),
				'news_excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'news_more_text' => array(
					'label'   => __( 'More Text:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => __( 'Read More', 'nature-bliss-pro' ),
					),
				'events_heading' => array(
					'label' => __( 'Events', 'nature-bliss-pro' ),
					'type'  => 'heading',
					),
				'events_section_title' => array(
					'label'   => __( 'Section Title:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
					'default' => __( 'Events', 'nature-bliss-pro' ),
					),
				'events_post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'events_post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'events_excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 10,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-news-events', __( 'NB: News and Events', 'nature-bliss-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params               = $this->get_params( $instance );
			$animation_attributes = $this->get_animation_attributes( $params );
			$animation_classes    = $this->get_animation_classes( $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			?>

			<div class="featured-news-events">
					<div class="inner-wrapper">

						<div class="news-events-column recent-news <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
							<h3><?php echo esc_html( $params['news_section_title'] ); ?></h3>
							<?php
								$qargs = array(
									'posts_per_page'      => absint( $params['news_post_number'] ),
									'no_found_rows'       => true,
									'ignore_sticky_posts' => true,
								);

								if ( absint( $params['news_post_category'] ) > 0 ) {
									$qargs['cat'] = absint( $params['news_post_category'] );
								}
								$the_query = new WP_Query( $qargs );
							?>
							<?php if ( $the_query->have_posts() ) : ?>
								<div class="new-inner-wrapper">
									<div class="inner-wrapper">
										<?php while ( $the_query->have_posts() ) : $the_query->the_post(); ?>
											<div class="news-post">
											<div class="news-post-wrapper">
												<?php if ( 'disable' !== $params['news_featured_image'] && has_post_thumbnail() ) : ?>
													<div class="news-post-thumb">
														<a href="<?php the_permalink(); ?>">
															<?php
															$img_attributes = array( 'class' => 'aligncenter' );
															the_post_thumbnail( esc_attr( $params['news_featured_image'] ), $img_attributes );
															?>
															</a>
													</div> <!-- .news-post-thumb -->
												<?php endif; ?>
												<div class="news-content">
													<h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
													<div class="news-meta">
														<span class="news-meta-date"><?php the_time( get_option( 'date_format') ); ?></span>
													</div>
													<?php
													$excerpt = nature_bliss_the_excerpt( absint( $params['news_excerpt_length'] ) );
													echo wp_kses_post( wpautop( $excerpt ) );
													?>
													<a href="<?php the_permalink(); ?>" class="more-link"><?php echo esc_html( $params['news_more_text'] ) ?></a>
												</div> <!-- .news-content -->
												</div> <!-- .news-post-wrapper -->
											</div><!-- .news-post -->
										<?php endwhile; ?>

										<?php wp_reset_postdata(); ?>
									</div><!-- .inner-wrapper -->
								</div><!-- .new-inner-wrapper -->

							<?php endif; ?>

						</div> <!-- .recent-news -->

						<div class="news-events-column recent-events <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
							<h3><?php echo esc_html( $params['events_section_title'] ); ?></h3>
							<?php
								$qargs = array(
									'posts_per_page'      => absint( $params['events_post_number'] ),
									'no_found_rows'       => true,
									'ignore_sticky_posts' => true,
								);

								if ( absint( $params['events_post_category'] ) > 0 ) {
									$qargs['cat'] = absint( $params['events_post_category'] );
								}
								$the_query = new WP_Query( $qargs );
							?>
							<?php if ( $the_query->have_posts() ) : ?>
								<div class="events-inner-wrapper">
										<?php while ( $the_query->have_posts() ) : $the_query->the_post(); ?>
											<div class="event-post">
												<div class="custom-entry-date">
													<span class="entry-day"><?php the_time( _x( 'd', 'date format', 'nature-bliss-pro' ) ); ?></span>
													<span class="entry-month"><?php the_time( _x( 'M', 'date format', 'nature-bliss-pro' ) ); ?></span>

												</div>
												<h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
												<?php
												$excerpt = nature_bliss_the_excerpt( absint( $params['events_excerpt_length'] ) );
												echo wp_kses_post( wpautop( $excerpt ) );
												?>
											</div> <!-- .event-post -->
										<?php endwhile; ?>

										<?php wp_reset_postdata(); ?>
								</div><!-- .new-inner-wrapper -->

							<?php endif; ?>

						</div> <!-- .recent-events -->
					</div> <!-- .inner-wrapper -->
			</div> <!-- .featured-news-events -->


			<?php echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Features_Widget' ) ) :

	/**
	 * Features widget Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Features_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'nature_bliss_widget_features',
				'description'                 => __( 'Displays features.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'main_image' => array(
					'label'       => __( 'Main Image:', 'nature-bliss-pro' ),
					'description' => __( 'Square image is recommended.', 'nature-bliss-pro' ),
					'type'        => 'image',
					),
				);

			for( $i = 1; $i <= 4; $i++ ) {
				$fields[ 'block_heading_' . $i ] = array(
					'label' => __( 'Block', 'nature-bliss-pro' ) . ' #' . $i,
					'type'  => 'heading',
					'class' => 'widefat',
					);
				$fields[ 'block_page_' . $i ] = array(
					'label'            => __( 'Select Page:', 'nature-bliss-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'nature-bliss-pro' ),
					);
				$fields[ 'block_icon_' . $i ] = array(
					'label'       => __( 'Icon:', 'nature-bliss-pro' ),
					'description' => __( 'Eg: fa-cogs', 'nature-bliss-pro' ),
					'type'        => 'text',
					'default'     => 'fa-cogs',
					);
			}

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-features', __( 'NB: Features', 'nature-bliss-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$feature_arr = array();
			for ( $i = 0; $i < 4 ; $i++ ) {
				$block = ( $i + 1 );
				$feature_arr[ $i ] = array(
					'page' => $params[ 'block_page_' . $block ],
					'icon' => $params[ 'block_icon_' . $block ],
					);
			}
			$refined_arr = array();
			if ( ! empty( $feature_arr ) ) {
				foreach ( $feature_arr as $item ) {
					if ( ! empty( $item['page'] ) ) {
						$refined_arr[] = $item;
					}
				}
			}

			if ( ! empty( $refined_arr ) ) {
				$this->render_widget_content( $refined_arr, $params );
			}

			echo $args['after_widget'];

		}

		function get_details( $items ) {
			$output = array();

			$page_ids = wp_list_pluck( $items, 'page' );
			$qargs = array(
				'post__in'      => $page_ids,
				'post_type'     => 'page',
				'orderby'       => 'post__in',
				'no_found_rows' => true,
			);
			$all_posts = get_posts( $qargs );
			if ( ! empty( $all_posts ) ) {
				$cnt = 0;
				foreach ( $all_posts as $post ) {
					$item = array();
					$item['title']   = apply_filters( 'the_title', $post->post_title );
					$item['url']     = get_permalink( $post->ID );
					$item['excerpt'] = nature_bliss_the_excerpt( 15, $post );
					$item['icon']    = '';
					if ( isset( $items[ $cnt ]['icon'] ) ) {
						$item['icon'] = $items[ $cnt ]['icon'];
					}
					$output[] = $item;
					$cnt++;
				}

			}

			return $output;
		}

		/**
		 * Render feature content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $feature_arr Features array.
		 * @param array $params      Parameters array.
		 */
		function render_widget_content( $feature_arr, $params ) {

			$animation_attributes = $this->get_animation_attributes( $params );
			$animation_classes    = $this->get_animation_classes( $params );

			$item_details = $this->get_details( $feature_arr );

			if ( empty( $item_details ) ) {
				return;
			}

			$first_part  = array();
			$second_part = array();
			$first_part  = array_slice( $item_details, 0, 2 );
			$second_part = array_slice( $item_details, 2, 2 );
			?>
			<div class="features-widget">
				<?php if ( ! empty( $first_part ) ) : ?>
					<div class="features-column features-main-content features-main-content-left <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
						<div class="features-page">
							<div class="features-block-list">
								<div class="inner-widget-wrapper">
									<?php foreach ( $first_part as $item ) : ?>
										<div class="features-block-item">
											<div class="features-block-inner">
												<?php if ( ! empty( $item['icon'] ) ) : ?>
													<a class="features-block-icon" href="<?php echo esc_url( $item['url'] ); ?>"><i class="fa <?php echo esc_attr( $item['icon'] ); ?>"></i></a>
												<?php endif; ?>
												<div class="features-block-inner-content">
													<h3 class="features-item-title">
														<a href="<?php echo esc_url( $item['url'] ); ?>"><?php echo esc_html( $item['title'] ); ?></a>
													</h3>
													<?php if ( ! empty( $item['excerpt'] ) ) : ?>
														<div class="features-item-excerpt">
															<?php echo wp_kses_post( wpautop( $item['excerpt'] ) ); ?>
														</div><!-- .features-item-excerpt -->
													<?php endif; ?>
												</div><!-- .features-block-inner-content -->
											</div><!-- .features-block-inner -->
										</div><!-- .features-block-item -->
									<?php endforeach; ?>
								</div><!-- .inner-widget-wrapper -->
							</div><!-- .features-block-list -->
						</div><!-- .features-page -->
					</div><!-- .features-main-content -->
				<?php endif; ?>

				<?php if ( ! empty( $params['main_image'] ) ) : ?>
					<div class="features-column features-thumb-main <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
						<img src="<?php echo esc_url( $params['main_image'] ); ?>" alt="">
					</div><!-- .features-thumb-main -->
				<?php endif; ?>

				<?php if ( ! empty( $second_part ) ) : ?>
					<div class="features-column features-main-content features-main-content-right <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
						<div class="features-page">
							<div class="features-block-list">
								<div class="inner-widget-wrapper">
									<?php foreach ( $second_part as $item ) : ?>
										<div class="features-block-item">
											<div class="features-block-inner">
												<?php if ( ! empty( $item['icon'] ) ) : ?>
													<a class="features-block-icon" href="<?php echo esc_url( $item['url'] ); ?>"><i class="fa <?php echo esc_attr( $item['icon'] ); ?>"></i></a>
												<?php endif; ?>
												<div class="features-block-inner-content">
													<h3 class="features-item-title">
														<a href="<?php echo esc_url( $item['url'] ); ?>"><?php echo esc_html( $item['title'] ); ?></a>
													</h3>
													<?php if ( ! empty( $item['excerpt'] ) ) : ?>
														<div class="features-item-excerpt">
															<?php echo wp_kses_post( wpautop( $item['excerpt'] ) ); ?>
														</div><!-- .features-item-excerpt -->
													<?php endif; ?>
												</div><!-- .features-block-inner-content -->
											</div><!-- .features-block-inner -->
										</div><!-- .features-block-item -->
									<?php endforeach; ?>
								</div><!-- .inner-widget-wrapper -->
							</div><!-- .features-block-list -->
						</div><!-- .features-page -->
					</div><!-- .features-main-content -->
				<?php endif; ?>
			</div><!-- .features-widget -->

			<?php
		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Recent_Works_Widget' ) ) :

	/**
	 * Recent works widget class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Recent_Works_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'nature_bliss_widget_recent_works',
				'description'                 => __( 'Displays recent works in grid.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'nature-bliss-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'disable_featured' => array(
					'label'   => __( 'Disable Featured', 'nature-bliss-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'nature-bliss-pro' ),
					'description' => __( 'in words', 'nature-bliss-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'more_text' => array(
					'label'   => __( 'More Text:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => __( 'Explore More', 'nature-bliss-pro' ),
					),
				'more_url' => array(
					'label'   => __( 'More URL:', 'nature-bliss-pro' ),
					'type'    => 'text',
					'default' => '#',
					),
				);

			$animation_fields = nature_bliss_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'nature-bliss-recent-works', __( 'NB: Recent Works', 'nature-bliss-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params               = $this->get_params( $instance );
			$animation_attributes = $this->get_animation_attributes( $params );
			$animation_classes    = $this->get_animation_classes( $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$qargs = array(
				'posts_per_page'      => absint( $params['post_number'] ),
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
				'meta_key'            => '_thumbnail_id',
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );

			?>
			<?php if ( $the_query->have_posts() ) : ?>

				<div class="inner-wrapper">
					<div class="portfolio-section">
						<?php while ( $the_query->have_posts() ) : $the_query->the_post(); ?>

							<?php $featured_class = ( 1 === $the_query->current_post + 1 && false === $params['disable_featured'] ) ? 'main-block' : ''; ?>

							<div class="portfolio-block <?php echo esc_attr( $featured_class ); ?> <?php echo $animation_classes; ?>" <?php echo $animation_attributes; ?>>
								<div class="portfolio-inner">
									<div class="block-container">
										<div class="thumb-overlay"></div>
										<div class="thumb-holder">
											<?php the_post_thumbnail( 'nature-bliss-works' ); ?>
										</div>
										<div class="portfolio-content">
											<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a> </h3>
										</div>
										<div class="portfolio-button">
											<div class="vmiddle-holder">
												<div class="vmiddle">
													<?php
													$excerpt = nature_bliss_the_excerpt( absint( $params['excerpt_length'] ) );
													echo wp_kses_post( wpautop( $excerpt ) );
													?>
													<a class="custom-button" href="<?php the_permalink(); ?>"><?php esc_html_e( 'Know More', 'nature-bliss-pro' ); ?></a>
												</div>
											</div>
										</div>
									</div> <!-- .block-container -->
								</div>
							</div> <!-- .portfolio-block -->

						<?php endwhile; ?>

						<?php if ( ! empty( $params['more_url'] ) && ! empty( $params['more_text'] ) ) : ?>
							<div class="load-more-wrapper">
								<a href="<?php echo esc_url( $params['more_url'] ) ?>" class="custom-button"><?php echo esc_html( $params['more_text'] ) ?></a>
							</div> <!-- .load-more-wrapper -->
						<?php endif; ?>

					</div><!-- .portfolio-section -->
				</div><!-- .inner-wrapper -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Nature_Bliss_Products_Grid_Widget' ) ) :

	/**
	 * Products Grid Class.
	 *
	 * @since 1.0.0
	 */
	class Nature_Bliss_Products_Grid_Widget extends Nature_Bliss_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'nature_bliss_widget_products_grid',
				'description'                 => esc_html__( 'Displays products in grid.', 'nature-bliss-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'nature-bliss-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'product_category' => array(
					'label'           => esc_html__( 'Select Product Category:', 'nature-bliss-pro' ),
					'type'            => 'dropdown-taxonomies',
					'taxonomy'        => 'product_cat',
					'show_option_all' => esc_html__( 'All Product Categories', 'nature-bliss-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Products:', 'nature-bliss-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => esc_html__( 'Number of Columns:', 'nature-bliss-pro' ),
					'type'    => 'select',
					'default' => 4,
					'options' => nature_bliss_get_numbers_dropdown_options( 3, 4 ),
					),
				);

			parent::__construct( 'nature-bliss-products-grid', esc_html__( 'NB: Products Grid', 'nature-bliss-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			// Render now.
			$this->render_products( $params );

			echo $args['after_widget'];

		}

		/**
		 * Render products.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_products( $params ) {

			$query_args = array(
				'post_type'           => 'product',
				'post_status'         => 'publish',
				'ignore_sticky_posts' => 1,
				'posts_per_page'      => esc_attr( $params['post_number'] ),
				'no_found_rows'       => true,
			);
			if ( absint( $params['product_category'] ) > 0 ) {
				$query_args['tax_query'] = array(
					array(
						'taxonomy' => 'product_cat',
						'field'    => 'term_id',
						'terms'    => absint( $params['product_category'] ),
						),
					);
			}

			global $woocommerce_loop;
			$products = new WP_Query( $query_args );

			if ( $products->have_posts() ) {
				?>
				<div class="inner-wrapper">
					<div class="nature-bliss-woocommerce nature-bliss-woocommerce-product-grid-<?php echo absint( $params['post_column'] ); ?>">

						<ul class="products">

							<?php while ( $products->have_posts() ) : $products->the_post(); ?>

								<?php wc_get_template_part( 'content', 'product' ); ?>

							<?php endwhile; ?>

						</ul><!-- .products -->

					</div><!-- .woocommerce -->
				</div> <!-- .inner-wrapper -->
				<?php
			}
			woocommerce_reset_loop();
			wp_reset_postdata();

		}
	}
endif;

