<?php
/**
 * Theme Palace functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Theme Palace
 * @subpackage Teach Kiddo Pro
 * @since Teach Kiddo Pro 1.0.0
 */


if ( ! function_exists( 'teach_kiddo_pro_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function teach_kiddo_pro_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Theme Palace, use a find and replace
		 * to change 'teach-kiddo-pro' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'teach-kiddo-pro' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		add_theme_support( "responsive-embeds" );

		add_theme_support( 'register_block_pattern' ); 

		add_theme_support( 'register_block_style' ); 

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Load Footer Widget Support.
		require_if_theme_supports( 'footer-widgets', get_template_directory() . '/inc/footer-widgets.php' );

		add_theme_support( 'woocommerce' );
		if ( class_exists( 'WooCommerce' ) ) {
	    	global $woocommerce;

	    	if( version_compare( $woocommerce->version, '3.0.0', ">=" ) ) {
	      		add_theme_support( 'wc-product-gallery-zoom' );
				add_theme_support( 'wc-product-gallery-lightbox' );
				add_theme_support( 'wc-product-gallery-slider' );
			}
	  	}
		
		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 600, 450, true );

		// Set the default content width.
		$GLOBALS['content_width'] = 525;
		
		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' 	=> esc_html__( 'Primary', 'teach-kiddo-pro' ),
			'social' 	=> esc_html__( 'Social', 'teach-kiddo-pro' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'teach_kiddo_pro_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// This setup supports logo, site-title & site-description
		add_theme_support( 'custom-logo', array(
			'height'      => 70,
			'width'       => 120,
			'flex-height' => true,
			'flex-width'  => true,
			'header-text' => array( 'site-title', 'site-description' ),
		) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		/*
		 * This theme styles the visual editor to resemble the theme style,
		 * specifically font, colors, icons, and column width.
		 */
		add_editor_style( array( '/assets/css/editor-style' . teach_kiddo_pro_min() . '.css', teach_kiddo_pro_fonts_url() ) );

		// Gutenberg support
		add_theme_support( 'editor-color-palette', array(
	       	array(
				'name' => esc_html__( 'Blue', 'teach-kiddo-pro' ),
				'slug' => 'blue',
				'color' => '#2c7dfa',
	       	),
	       	array(
	           	'name' => esc_html__( 'Green', 'teach-kiddo-pro' ),
	           	'slug' => 'green',
	           	'color' => '#07d79c',
	       	),
	       	array(
	           	'name' => esc_html__( 'Orange', 'teach-kiddo-pro' ),
	           	'slug' => 'orange',
	           	'color' => '#ff8737',
	       	),
	       	array(
	           	'name' => esc_html__( 'Black', 'teach-kiddo-pro' ),
	           	'slug' => 'black',
	           	'color' => '#2f3633',
	       	),
	       	array(
	           	'name' => esc_html__( 'Grey', 'teach-kiddo-pro' ),
	           	'slug' => 'grey',
	           	'color' => '#82868b',
	       	),
	   	));

		add_theme_support( 'align-wide' );
		add_theme_support( 'editor-font-sizes', array(
		   	array(
		       	'name' => esc_html__( 'small', 'teach-kiddo-pro' ),
		       	'shortName' => esc_html__( 'S', 'teach-kiddo-pro' ),
		       	'size' => 12,
		       	'slug' => 'small'
		   	),
		   	array(
		       	'name' => esc_html__( 'regular', 'teach-kiddo-pro' ),
		       	'shortName' => esc_html__( 'M', 'teach-kiddo-pro' ),
		       	'size' => 16,
		       	'slug' => 'regular'
		   	),
		   	array(
		       	'name' => esc_html__( 'larger', 'teach-kiddo-pro' ),
		       	'shortName' => esc_html__( 'L', 'teach-kiddo-pro' ),
		       	'size' => 36,
		       	'slug' => 'larger'
		   	),
		   	array(
		       	'name' => esc_html__( 'huge', 'teach-kiddo-pro' ),
		       	'shortName' => esc_html__( 'XL', 'teach-kiddo-pro' ),
		       	'size' => 48,
		       	'slug' => 'huge'
		   	)
		));
		add_theme_support('editor-styles');
		add_theme_support( 'wp-block-styles' );
	}
endif;
add_action( 'after_setup_theme', 'teach_kiddo_pro_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function teach_kiddo_pro_content_width() {

	$content_width = $GLOBALS['content_width'];


	$sidebar_position = teach_kiddo_pro_layout();
	switch ( $sidebar_position ) {

	  case 'no-sidebar':
	    $content_width = 1170;
	    break;

	  case 'left-sidebar':
	  case 'right-sidebar':
	    $content_width = 819;
	    break;

	  default:
	    break;
	}

	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$content_width = 1170;
	}

	/**
	 * Filter Teach Kiddo Pro content width of the theme.
	 *
	 * @since Teach Kiddo Pro 1.0.0
	 *
	 * @param int $content_width Content width in pixels.
	 */
	$GLOBALS['content_width'] = apply_filters( 'teach_kiddo_pro_content_width', $content_width );
}
add_action( 'template_redirect', 'teach_kiddo_pro_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function teach_kiddo_pro_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'teach-kiddo-pro' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'teach-kiddo-pro' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'teach_kiddo_pro_widgets_init' );


if ( ! function_exists( 'teach_kiddo_pro_fonts_url' ) ) :
/**
 * Register Google fonts
 *
 * @return string Google fonts URL for the theme.
 */
function teach_kiddo_pro_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';
	$options   = teach_kiddo_pro_get_theme_options();

	/* translators: If there are characters in your language that are not supported by Mulish, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Mulish font: on or off', 'teach-kiddo-pro' ) ) {
		$fonts[] = 'Mulish:400,500,600,700';
	}

	/* translators: If there are characters in your language that are not supported by Raleway, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Raleway font: on or off', 'teach-kiddo-pro' ) ) {
		$fonts[] = 'Raleway:400,500,600,700,800,900';
	}

	/* translators: If there are characters in your language that are not supported by Lora, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Lora font: on or off', 'teach-kiddo-pro' ) ) {
		$fonts[] = 'Lora';
	}

	if(!empty( $options['theme_head_typography'] )){
		$fonts[] =  str_replace("+", " ", $options['theme_head_typography']).':400,700';
	}

	if(!empty( $options['theme_menu_typography'] )){
		$fonts[] =  str_replace("+", " ", $options['theme_menu_typography']).':400,700';
	}

	if(!empty( $options['theme_site_title_typography'] )){
		$fonts[] =  str_replace("+", " ", $options['theme_site_title_typography']).':400,700';
	}

	if(!empty( $options['theme_site_description_typography'] )){
		$fonts[] =  str_replace("+", " ", $options['theme_site_description_typography']).':400,700';
	}
	
	if(!empty( $options['theme_body_typography'] )){
		$fonts[] = str_replace("+", " ", $options['theme_body_typography']).':400,700';
	}

	if(!empty( $options['theme_btn_label_typography'] )){
		$fonts[] = str_replace("+", " ", $options['theme_btn_label_typography']).':400,700';
	}

	if(!empty( $options['theme_sub_head_typography'] )){
		$fonts[] = str_replace("+", " ", $options['theme_sub_head_typography']).':400,700';
	}

	
	$query_args = array(
		'family' => urlencode( implode( '|', $fonts ) ),
		'subset' => urlencode( $subsets ),
	);

	if ( $fonts ) {
		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return esc_url_raw( $fonts_url );
}
endif;

/**
 * Add preconnect for Google Fonts.
 *
 * @since Teach Kiddo Pro 1.0.0
 *
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function teach_kiddo_pro_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'teach-kiddo-pro-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'teach_kiddo_pro_resource_hints', 10, 2 );

/**
 * Enqueue scripts and styles.
 */
function teach_kiddo_pro_scripts() {
	$options = teach_kiddo_pro_get_theme_options();
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'teach-kiddo-pro-fonts', wptt_get_webfont_url( teach_kiddo_pro_fonts_url() ), array(), null );

	wp_enqueue_style( 'teach-kiddo-pro-woocommerce-layout', get_template_directory_uri() . '/assets/css/woocommerce-layout' . teach_kiddo_pro_min() . '.css' );

	wp_enqueue_style( 'teach-kiddo-pro-woocommerce', get_template_directory_uri() . '/assets/css/woocommerce' . teach_kiddo_pro_min() . '.css' );

	// slick
	wp_enqueue_style( 'slick', get_template_directory_uri() . '/assets/css/slick' . teach_kiddo_pro_min() . '.css' );

	// slick theme
	wp_enqueue_style( 'slick-theme', get_template_directory_uri() . '/assets/css/slick-theme' . teach_kiddo_pro_min() . '.css' );

	// font awesome
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/css/font-awesome' . teach_kiddo_pro_min() . '.css' );

	// blocks
	wp_enqueue_style( 'teach-kiddo-pro-blocks', get_template_directory_uri() . '/assets/css/blocks' . teach_kiddo_pro_min() . '.css' );

	wp_enqueue_style( 'teach-kiddo-pro-style', get_stylesheet_uri() );

	// imagesloaded
	wp_enqueue_script( 'imagesloaded' );

	// Load the colorscheme.
	$color_scheme = $options['colorscheme'];
	if ( ! in_array( $color_scheme, array( 'default', 'custom' ) ) ) {
		wp_enqueue_style( 'teach-kiddo-pro-colors', get_template_directory_uri() . '/assets/css/' . esc_attr( $color_scheme ) . teach_kiddo_pro_min() .'.css', array( 'teach-kiddo-pro-style' ), '1.0' );
	}
	
	// Load the html5 shiv.
	wp_enqueue_script( 'teach-kiddo-pro-html5', get_template_directory_uri() . '/assets/js/html5' . teach_kiddo_pro_min() . '.js', array(), '3.7.3' );
	wp_script_add_data( 'teach-kiddo-pro-html5', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'teach-kiddo-pro-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix' . teach_kiddo_pro_min() . '.js', array(), '20160412', true );

	wp_enqueue_script( 'teach-kiddo-pro-navigation', get_template_directory_uri() . '/assets/js/navigation' . teach_kiddo_pro_min() . '.js', array(), '20151215', true );
	
	$teach_kiddo_pro_l10n = array(
		'quote'          => teach_kiddo_pro_get_svg( array( 'icon' => 'quote-right' ) ),
		'expand'         => esc_html__( 'Expand child menu', 'teach-kiddo-pro' ),
		'collapse'       => esc_html__( 'Collapse child menu', 'teach-kiddo-pro' ),
		'icon'           => teach_kiddo_pro_get_svg( array( 'icon' => 'down', 'fallback' => true ) ),
	);
	
	wp_localize_script( 'teach-kiddo-pro-navigation', 'teach_kiddo_pro_l10n', $teach_kiddo_pro_l10n );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}


	wp_enqueue_script( 'jquery-slick', get_template_directory_uri() . '/assets/js/slick.js', array( 'jquery' ), '', true );

	wp_enqueue_script( 'jquery-packery', get_template_directory_uri() . '/assets/js/packery.pkgd' . teach_kiddo_pro_min() . '.js', array( 'jquery' ), '', true );

	wp_enqueue_script( 'teach-kiddo-pro-custom', get_template_directory_uri() . '/assets/js/custom' . teach_kiddo_pro_min() . '.js', array( 'jquery' ), '20151215', true );

	if ( 'infinite' == $options['pagination_type'] ) {
		// infinite scroll js
		wp_enqueue_script( 'teach-kiddo-pro-infinite-scroll', get_template_directory_uri() . '/assets/js/infinite-scroll' . teach_kiddo_pro_min() . '.js', array( 'jquery' ), '', true );
	}
}
add_action( 'wp_enqueue_scripts', 'teach_kiddo_pro_scripts' );

/**
 * Enqueue editor styles for Gutenberg
 *
 * @since Teach Kiddo Pro 1.0.0
 */
function teach_kiddo_pro_block_editor_styles() {
	// Block styles.
	wp_enqueue_style( 'teach-kiddo-pro-block-editor-style', get_theme_file_uri( '/assets/css/editor-blocks' . teach_kiddo_pro_min() . '.css' ) );
	// Add custom fonts.
	wp_enqueue_style( 'teach-kiddo-pro-fonts', wptt_get_webfont_url( teach_kiddo_pro_fonts_url() ), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'teach_kiddo_pro_block_editor_styles' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Load core file
 */
require get_template_directory() . '/inc/core.php';



