<?php
/**
 * Basic theme functions.
 *
 * This file contains hook functions attached to core hooks.
 *
 * @package Clean_Commerce
 */

if ( ! function_exists( 'clean_commerce_customize_search_form' ) ) :
	/**
	 * Customize search form.
	 *
	 * @since 1.0.0
	 *
	 * @return string The search form HTML output.
	 */
	function clean_commerce_customize_search_form() {

		$search_placeholder = clean_commerce_get_option( 'search_placeholder' );

		$form = '<form role="search" method="get" class="search-form" action="' . esc_url( home_url( '/' ) ) . '">
			<label>
			<span class="screen-reader-text">' . esc_html_x( 'Search for:', 'label', 'clean-commerce-pro' ) . '</span>
			<input type="search" class="search-field" placeholder="' . esc_attr( $search_placeholder ) . '" value="' . get_search_query() . '" name="s" title="' . esc_attr_x( 'Search for:', 'label', 'clean-commerce-pro' ) . '" />
			</label>
			<input type="submit" class="search-submit" value="' . esc_attr_x( 'Search', 'search button', 'clean-commerce-pro' ) . '" /></form>';
		return $form;

	}

endif;

add_filter( 'get_search_form', 'clean_commerce_customize_search_form', 15 );

if ( ! function_exists( 'clean_commerce_implement_excerpt_length' ) ) :

	/**
	 * Implement excerpt length.
	 *
	 * @since 1.0.0
	 *
	 * @param int $length The number of words.
	 * @return int Excerpt length.
	 */
	function clean_commerce_implement_excerpt_length( $length ) {

		$excerpt_length = clean_commerce_get_option( 'excerpt_length' );
		$excerpt_length = apply_filters( 'clean_commerce_filter_excerpt_length', $excerpt_length );

		if ( absint( $excerpt_length ) > 0 ) {
			$length = absint( $excerpt_length );
		}

		return $length;

	}

endif;

if ( ! function_exists( 'clean_commerce_implement_read_more' ) ) :

	/**
	 * Implement read more in excerpt.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more The string shown within the more link.
	 * @return string The excerpt.
	 */
	function clean_commerce_implement_read_more( $more ) {

		$flag_apply_excerpt_read_more = apply_filters( 'clean_commerce_filter_excerpt_read_more', true );
		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more;
		}

		$output = $more;
		$read_more_text = clean_commerce_get_option( 'read_more_text' );
		if ( ! empty( $read_more_text ) ) {
			$output = ' <a href="'. esc_url( get_permalink() ) . '" class="read-more">' . esc_html( $read_more_text ) . '</a>';
			$output = apply_filters( 'clean_commerce_filter_read_more_link' , $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'clean_commerce_content_more_link' ) ) :

	/**
	 * Implement read more in content.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more_link Read More link element.
	 * @param string $more_link_text Read More text.
	 * @return string Link.
	 */
	function clean_commerce_content_more_link( $more_link, $more_link_text ) {

		$flag_apply_excerpt_read_more = apply_filters( 'clean_commerce_filter_excerpt_read_more', true );
		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more_link;
		}

		$read_more_text = clean_commerce_get_option( 'read_more_text' );
		if ( ! empty( $read_more_text ) ) {
			$more_link = str_replace( $more_link_text, esc_html( $read_more_text ), $more_link );
		}
		return $more_link;

	}

endif;

if ( ! function_exists( 'clean_commerce_custom_body_class' ) ) :

	/**
	 * Custom body class.
	 *
	 * @since 1.0.0
	 *
	 * @param string|array $input One or more classes to add to the class list.
	 * @return array Array of classes.
	 */
	function clean_commerce_custom_body_class( $input ) {

		// Adds a class of group-blog to blogs with more than 1 published author.
		if ( is_multi_author() ) {
			$input[] = 'group-blog';
		}

		$home_content_status = clean_commerce_get_option( 'home_content_status' );
		if( true !== $home_content_status ) {
			$input[] = 'home-content-not-enabled';
		}

		// Global layout.
		global $post;
		$global_layout = clean_commerce_get_option( 'global_layout' );
		$global_layout = apply_filters( 'clean_commerce_filter_theme_global_layout', $global_layout );

		// Check if single.
		if ( $post  && is_singular() ) {
			$post_options = get_post_meta( $post->ID, 'clean_commerce_theme_settings', true );
			if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
				$global_layout = $post_options['post_layout'];
			}
		}

		$input[] = 'global-layout-' . esc_attr( $global_layout );

		// Common class for three columns.
		switch ( $global_layout ) {
			case 'three-columns':
			case 'three-columns-pcs':
			case 'three-columns-cps':
			case 'three-columns-psc':
			case 'three-columns-pcs-equal':
			case 'three-columns-scp-equal':
				$input[] = 'three-columns-enabled';
				break;

			default:
				break;
		}

		// Sticky menu.
		$enable_sticky_primary_menu = clean_commerce_get_option( 'enable_sticky_primary_menu' );

		if ( true === $enable_sticky_primary_menu) {
			$input[] = 'enabled-sticky-primary-menu';
		}

		//WooCommerce Button Hover
		$woo_button = clean_commerce_get_option( 'show_button_hover' );

		if ( true === $woo_button) {
			$input[] = 'woobuttonhover';
		}

		return $input;

	}
endif;

add_filter( 'body_class', 'clean_commerce_custom_body_class' );

if ( ! function_exists( 'clean_commerce_custom_content_width' ) ) :

	/**
	 * Custom content width.
	 *
	 * @since 1.0.0
	 */
	function clean_commerce_custom_content_width() {

		global $post, $wp_query, $content_width;

		$global_layout = clean_commerce_get_option( 'global_layout' );
		$global_layout = apply_filters( 'clean_commerce_filter_theme_global_layout', $global_layout );

		// Check if single.
		if ( $post  && is_singular() ) {
		  $post_options = get_post_meta( $post->ID, 'clean_commerce_theme_settings', true );
		  if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
		    $global_layout = esc_attr( $post_options['post_layout'] );
		  }
		}
		switch ( $global_layout ) {

			case 'no-sidebar':
				$content_width = 1220;
				break;

			case 'three-columns':
			case 'three-columns-pcs':
			case 'three-columns-cps':
			case 'three-columns-psc':
			case 'no-sidebar-centered':
				$content_width = 570;
				break;

			case 'three-columns-pcs-equal':
			case 'three-columns-scp-equal':
				$content_width = 363;
				break;

			case 'left-sidebar':
			case 'right-sidebar':
				$content_width = 895;
				break;

			default:
				break;
		}

	}
endif;

add_filter( 'template_redirect', 'clean_commerce_custom_content_width' );

if ( ! function_exists( 'clean_commerce_hook_read_more_filters' ) ) :

	/**
	 * Hook read more filters.
	 *
	 * @since 1.0.0
	 */
	function clean_commerce_hook_read_more_filters() {
		if ( is_home() || is_category() || is_tag() || is_author() || is_date() ) {

			add_filter( 'excerpt_length', 'clean_commerce_implement_excerpt_length', 999 );
			add_filter( 'the_content_more_link', 'clean_commerce_content_more_link', 10, 2 );
			add_filter( 'excerpt_more', 'clean_commerce_implement_read_more' );

		}
	}
endif;

add_action( 'wp', 'clean_commerce_hook_read_more_filters' );

/**
 * Load theme options from free theme.
 *
 * Checks if there are options already present from free version and adds it to the Pro theme options.
 *
 * @since 1.0.0
 */
function clean_commerce_import_free_options() {

	// Perform action only if theme_mods_XXX[theme_options] does not exist.
	if( ! get_theme_mod( 'theme_options' ) ) {

		// Perform action only if theme_mods_XXX free version exists.
		if ( $free_options = get_option ( 'theme_mods_clean-commerce' ) ) {
			if ( isset( $free_options['theme_options'] ) ) {
				$pro_default_options = clean_commerce_get_default_theme_options();

				$pro_theme_options = $free_options;

				$pro_theme_options['theme_options'] = array_merge( $pro_default_options , $free_options['theme_options'] );

				// WP default fields.
				$fields = array(
					'custom_logo',
					'header_image',
					'header_image_data',
					'background_color',
					'background_image',
					'background_repeat',
					'background_position_x',
					'background_attachment',
				);

				foreach ( $fields as $key ) {
					if ( isset( $free_options[ $key ] ) && ! empty( $free_options[ $key ] ) ) {
						$pro_theme_options[ $key ] = $free_options[ $key ];
					}
				}

				update_option( 'theme_mods_clean-commerce-pro', $pro_theme_options );
			}
		}
	}
}

add_action( 'after_switch_theme', 'clean_commerce_import_free_options' );

if ( ! function_exists( 'clean_commerce_customizer_reset_callback' ) ) :

	/**
	 * Callback for reset in Customizer.
	 *
	 * @since 1.0.0
	 */
	function clean_commerce_customizer_reset_callback() {

		$reset_all_settings = clean_commerce_get_option( 'reset_all_settings' );

		if ( true === $reset_all_settings ) {

			// Reset custom theme options.
			set_theme_mod( 'theme_options', array() );

			// Reset custom header, logo and backgrounds.
			remove_theme_mod( 'custom_logo' );
			remove_theme_mod( 'header_image' );
			remove_theme_mod( 'header_image_data' );
			remove_theme_mod( 'background_image' );
			remove_theme_mod( 'background_color' );
		}

		// Reset color options.
		$reset_color_settings = clean_commerce_get_option( 'reset_color_settings' );
		if ( true === $reset_color_settings ) {
			$options = clean_commerce_get_options();
			$options['reset_color_settings'] = false;
			$color_fields = clean_commerce_get_color_theme_settings_options();
			$default = clean_commerce_get_default_theme_options();
			if ( ! empty( $color_fields ) ) {
				foreach ( $color_fields as $key => $val ) {
					$options[ $key ] = $default[ $key ];
				}
			}
			remove_theme_mod( 'background_color' );
			set_theme_mod( 'theme_options', $options );
		}

		// Reset font options.
		$reset_font_settings = clean_commerce_get_option( 'reset_font_settings' );
		if ( true === $reset_font_settings ) {
			$options = clean_commerce_get_options();
			$options['reset_font_settings'] = false;
			$font_settings = clean_commerce_get_font_family_theme_settings_options();
			if ( ! empty( $font_settings ) ) {
				foreach ( $font_settings as $key => $val ) {
					$options[ $key ] = $val['default'];
				}
			}
			set_theme_mod( 'theme_options', $options );
		}

		// Reset footer content.
		$reset_footer_content = clean_commerce_get_option( 'reset_footer_content' );
		if ( true === $reset_footer_content ) {
			$options = clean_commerce_get_options();
			$options['reset_footer_content'] = false;
	  		$default = clean_commerce_get_default_theme_options();
	  		$footer_fields = array(
				'copyright_text',
				'powered_by_text',
			);
	  		foreach ( $footer_fields as $field ) {
		  		if ( isset( $default[ $field ] ) ) {
		  			$options[ $field ] = $default[ $field ];
		  		}
	  		}
			set_theme_mod( 'theme_options', $options );
		}

	}
endif;

add_action( 'customize_save_after', 'clean_commerce_customizer_reset_callback' );
