<?php
/**
 * Theme widgets.
 *
 * @package Clean_Commerce
 */

// Load widget base.
require_once get_template_directory() . '/lib/widget-base/class-widget-base.php';

if ( ! function_exists( 'clean_commerce_load_widgets' ) ) :

	/**
	 * Load widgets.
	 *
	 * @since 1.0.0
	 */
	function clean_commerce_load_widgets() {

		// Social widget.
		register_widget( 'Clean_Commerce_Social_Widget' );

		// Image widget.
		register_widget( 'Clean_Commerce_Image_Widget' );

		// Featured Page widget.
		register_widget( 'Clean_Commerce_Featured_Page_Widget' );

		// Latest News widget.
		register_widget( 'Clean_Commerce_Latest_News_Widget' );

		// Call To Action widget.
		register_widget( 'Clean_Commerce_Call_To_Action_Widget' );

		// Services widget.
		register_widget( 'Clean_Commerce_Services_Widget' );

		// Testimonial slider widget.
		register_widget( 'Clean_Commerce_Testimonial_Slider_Widget' );

		// Recent Posts widget.
		register_widget( 'Clean_Commerce_Recent_Posts_Widget' );

		if ( class_exists( 'WooCommerce' ) ) {

			// Featured Products Carousel widget.
			register_widget( 'Clean_Commerce_Featured_Products_Carousel_Widget' );

			// Products Grid widget.
			register_widget( 'Clean_Commerce_Products_Grid_Widget' );

			// Products Slider widget.
			register_widget( 'Clean_Commerce_Products_Slider_Widget' );

			// Products Category Grid widget.
			register_widget( 'Clean_Commerce_Products_Category_Grid_Widget' );

		}

	}

endif;

add_action( 'widgets_init', 'clean_commerce_load_widgets' );

if ( ! class_exists( 'Clean_Commerce_Social_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Social_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_social',
				'description'                 => esc_html__( 'Displays social icons.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				);

			if ( false === has_nav_menu( 'social' ) ) {
				$fields['message'] = array(
					'label' => esc_html__( 'Social menu is not set. Please create menu and assign it to Social Menu.', 'clean-commerce-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					);
			}

			parent::__construct( 'clean-commerce-social', esc_html__( 'CC: Social', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			if ( has_nav_menu( 'social' ) ) {
				wp_nav_menu( array(
					'theme_location' => 'social',
					'container'      => false,
					'depth'          => 1,
					'link_before'    => '<span class="screen-reader-text">',
					'link_after'     => '</span>',
				) );
			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Image_Widget' ) ) :

	/**
	 * Image widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Image_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_image',
				'description'                 => esc_html__( 'Displays a image with link.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'image_url' => array(
					'label' => esc_html__( 'Image:', 'clean-commerce-pro' ),
					'type'  => 'image',
					),
				'link_url' => array(
					'label' => esc_html__( 'URL:', 'clean-commerce-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				);

			parent::__construct( 'clean-commerce-image', esc_html__( 'CC: Image', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['image_url'] ) ) {
				$link_open  = '';
				$link_close = '';
				if ( ! empty( $params['link_url'] ) ) {
					$link_open  = '<a href="' . esc_url( $params['link_url'] ) . '">';
					$link_close = '</a>';
				}
				echo $link_open;
				echo '<img src="' . esc_url( $params['image_url'] ) . '" />';
				echo $link_close;
			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Featured_Page_Widget' ) ) :

	/**
	 * Featured page widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Featured_Page_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_featured_page',
				'description'                 => esc_html__( 'Displays single featured Page or Post.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'use_page_title' => array(
					'label'   => esc_html__( 'Use Page/Post Title as Widget Title', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'featured_page' => array(
					'label'            => esc_html__( 'Select Page:', 'clean-commerce-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => esc_html__( '&mdash; Select &mdash;', 'clean-commerce-pro' ),
					),
				'id_message' => array(
					'label'            => '<strong>' . _x( 'OR', 'Featured Page Widget', 'clean-commerce-pro' ) . '</strong>',
					'type'             => 'message',
					),
				'featured_post' => array(
					'label'             => esc_html__( 'Post ID:', 'clean-commerce-pro' ),
					'placeholder'       => esc_html__( 'Eg: 1234', 'clean-commerce-pro' ),
					'type'              => 'text',
					'sanitize_callback' => 'clean_commerce_widget_sanitize_post_id',
					),
				'content_type' => array(
					'label'   => esc_html__( 'Show Content:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'full',
					'options' => array(
						'excerpt' => esc_html__( 'Excerpt', 'clean-commerce-pro' ),
						'full'    => esc_html__( 'Full', 'clean-commerce-pro' ),
						),
					),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'Applies when Excerpt is selected in Content option.', 'clean-commerce-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 40,
					'min'         => 1,
					'max'         => 400,
					),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'options' => clean_commerce_get_image_sizes_options(),
					),
				'featured_image_alignment' => array(
					'label'   => esc_html__( 'Image Alignment:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'center',
					'options' => clean_commerce_get_image_alignment_options(),
					),
				);

			parent::__construct( 'clean-commerce-featured-page', esc_html__( 'CC: Featured Page', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			// ID validation.
			$our_post_object = null;
			$our_id = '';
			if ( absint( $params['featured_post'] ) > 0 ) {
				$our_id = absint( $params['featured_post'] );
			}
			if ( absint( $params['featured_page'] ) > 0 ) {
				$our_id = absint( $params['featured_page'] );
			}
			if ( absint( $our_id ) > 0 ) {
				$raw_object = get_post( $our_id );
				if ( ! in_array( $raw_object->post_type, array( 'attachment', 'nav_menu_item', 'revision' ) ) ) {
					$our_post_object = $raw_object;
				}
			}
			if ( ! $our_post_object ) {
				// No valid object; bail now!
				return;
			}

			echo $args['before_widget'];

			global $post;
			// Setup global post.
			$post = $our_post_object;
			setup_postdata( $post );

			// Override title if checkbox is selected.
			if ( true === $params['use_page_title'] ) {
				$params['title'] = get_the_title( $post );
			}

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			?>
			<div class="featured-page-widget entry-content">
				<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
					<?php the_post_thumbnail( esc_attr( $params['featured_image'] ), array( 'class' => 'align' . esc_attr( $params['featured_image_alignment'] ) ) ); ?>
				<?php endif; ?>
				<?php if ( 'excerpt' === $params['content_type'] ) : ?>
					<?php
						$excerpt = clean_commerce_the_excerpt( absint( $params['excerpt_length'] ) );
						echo wp_kses_post( wpautop( $excerpt ) );
						?>
				<?php else : ?>
					<?php the_content(); ?>
				<?php endif; ?>

			</div><!-- .featured-page-widget -->
			<?php
			// Reset.
			wp_reset_postdata();

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Latest_News_Widget' ) ) :

	/**
	 * Latest news widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Latest_News_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'clean_commerce_widget_latest_news',
				'description'                 => esc_html__( 'Displays latest posts in grid.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => esc_html__( 'Select Category:', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'clean-commerce-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Posts:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => esc_html__( 'Number of Columns:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => clean_commerce_get_numbers_dropdown_options( 1, 4 ),
					),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'medium',
					'options' => clean_commerce_get_image_sizes_options(),
					),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'in words', 'clean-commerce-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'more_text' => array(
					'label'   => esc_html__( 'More Text:', 'clean-commerce-pro' ),
					'type'    => 'text',
					'default' => esc_html__( 'Read more', 'clean-commerce-pro' ),
					),
				'disable_date' => array(
					'label'   => esc_html__( 'Disable Date', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_excerpt' => array(
					'label'   => esc_html__( 'Disable Excerpt', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_more_text' => array(
					'label'   => esc_html__( 'Disable More Text', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			parent::__construct( 'clean-commerce-latest-news', esc_html__( 'CC: Latest News', 'clean-commerce-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) : ?>

				<?php global $post; ?>

				<div class="latest-news-widget latest-news-col-<?php echo esc_attr( $params['post_column'] ); ?>">

					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $key => $post ) : ?>
							<?php setup_postdata( $post ); ?>

							<div class="latest-news-item">

									<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
										<div class="latest-news-thumb">
											<a href="<?php the_permalink(); ?>">
												<?php
												$img_attributes = array( 'class' => 'aligncenter' );
												the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
												?>
											</a>
										</div><!-- .latest-news-thumb -->
									<?php endif; ?>
									<div class="latest-news-text-wrap">

										<div class="latest-news-text-content">
											<h3 class="latest-news-title">
												<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
											</h3><!-- .latest-news-title -->

											<?php if ( false === $params['disable_excerpt'] ) :  ?>
												<div class="latest-news-summary">
												<?php
												$summary = clean_commerce_the_excerpt( esc_attr( $params['excerpt_length'] ), $post );
												echo wp_kses_post( wpautop( $summary ) );
												?>
												</div><!-- .latest-news-summary -->
											<?php endif; ?>
										</div><!-- .latest-news-text-content -->

										<?php if ( false === $params['disable_date'] || false === $params['disable_more_text'] ) : ?>
											<div class="latest-news-meta">
												<ul>
													<?php if ( false === $params['disable_date'] ) :  ?>
														<li class="news-date-meta"><span class="latest-news-date"><?php the_time( 'j M Y' ); ?></span></li>
													<?php endif; ?>
													<?php if ( false === $params['disable_more_text'] ) :  ?>
														<li class="news-read-more"><a href="<?php the_permalink(); ?>"><?php echo esc_html( $params['more_text'] ); ?><span class="screen-reader-text">"<?php the_title(); ?>"</span>
														</a></li>
													<?php endif; ?>
												</ul>
											</div><!-- .latest-news-meta -->
										<?php endif; ?>

									</div><!-- .latest-news-text-wrap -->

							</div><!-- .latest-news-item -->

						<?php endforeach; ?>

					</div><!-- .row -->

				</div><!-- .latest-news-widget -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php echo $args['after_widget'];

		}
	}
endif;


if ( ! class_exists( 'Clean_Commerce_Call_To_Action_Widget' ) ) :

	/**
	 * Call to action widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Call_To_Action_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_call_to_action',
				'description'                 => esc_html__( 'Call To Action Widget.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'description' => array(
					'label' => esc_html__( 'Description:', 'clean-commerce-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
					),
				'primary_button_text' => array(
					'label'   => esc_html__( 'Primary Button Text:', 'clean-commerce-pro' ),
					'default' => esc_html__( 'Learn more', 'clean-commerce-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'primary_button_url' => array(
					'label' => esc_html__( 'Primary Button URL:', 'clean-commerce-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				'secondary_button_text' => array(
					'label'   => esc_html__( 'Secondary Button Text:', 'clean-commerce-pro' ),
					'default' => '',
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'secondary_button_url' => array(
					'label' => esc_html__( 'Secondary Button URL:', 'clean-commerce-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				'background_image' => array(
					'label'   => esc_html__( 'Background Image:', 'clean-commerce-pro' ),
					'type'    => 'image',
					'default' => '',
					),
				'layout' => array(
					'label'   => esc_html__( 'Layout:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 1,
					'options' => clean_commerce_get_numbers_dropdown_options( 1, 2, esc_html__( 'Layout', 'clean-commerce-pro' ) . ' ' ),
					),
				);

			parent::__construct( 'clean-commerce-call-to-action', esc_html__( 'CC: Call To Action', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			// Add background image for layout 1.
			if ( 1 === absint( $params['layout'] ) && ! empty( $params['background_image'] ) ) {
				$background_style = '';
				$background_style .= ' style="background-image:url(' . esc_url( $params['background_image'] ) . ');" ';
				$args['before_widget'] = implode( $background_style . ' ' . 'class="', explode( 'class="', $args['before_widget'], 2 ) );
			}

			// Add layout class.
			$layout_class = 'cta-layout-' . absint( $params['layout'] );
			$args['before_widget'] = implode( 'class="' . $layout_class . ' ', explode( 'class="', $args['before_widget'], 2 ) );

			echo $args['before_widget'];

			echo '<div class="cta-wrapper">';

			// Add background image for layout 2.
			if ( 2 === absint( $params['layout'] ) && ! empty( $params['background_image'] ) ) {
				echo '<img src="' . esc_url( $params['background_image'] ) . '" alt="" />';
			}
			?>
			<div class="call-to-action-content">
				<?php
				if ( ! empty( $params['title'] ) ) {
					echo $args['before_title'] . $params['title'] . $args['after_title'];
				}
				?>
				<?php if ( ! empty( $params['description'] ) ) : ?>
				    <div class="call-to-action-description">
				        <?php echo wp_kses_post( wpautop( $params['description'] ) ); ?>
				    </div><!-- .call-to-action-description -->
				<?php endif; ?>
				<?php if ( ! empty( $params['primary_button_text'] ) || ! empty( $params['secondary_button_text'] ) ) : ?>
					<div class="call-to-action-buttons">
						<?php if ( ! empty( $params['primary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['primary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-primary"><?php echo esc_html( $params['primary_button_text'] ); ?></a>
						<?php endif; ?>
						<?php if ( ! empty( $params['secondary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['secondary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-secondary"><?php echo esc_html( $params['secondary_button_text'] ); ?></a>
						<?php endif; ?>
					</div><!-- .call-to-action-buttons -->
				<?php endif; ?>
			</div><!-- .call-to-action-content -->
			<?php
			echo '</div><!-- .cta-wrapper -->';

			echo $args['after_widget'];

		}
	}
endif;


if ( ! class_exists( 'Clean_Commerce_Services_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Services_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_services',
				'description'                 => esc_html__( 'Show your services with icon and read more link.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'in words', 'clean-commerce-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'disable_excerpt' => array(
					'label'   => esc_html__( 'Disable Excerpt', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'post_column' => array(
					'label'   => esc_html__( 'Number of Columns:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => clean_commerce_get_numbers_dropdown_options( 2, 4 ),
					),
				);

			for( $i = 1; $i <= 8; $i++ ) {
				$fields[ 'block_heading_' . $i ] = array(
					'label' => esc_html__( 'Block', 'clean-commerce-pro' ) . ' #' . $i,
					'type'  => 'heading',
					'class' => 'widefat',
					);
				$fields[ 'block_page_' . $i ] = array(
					'label'            => esc_html__( 'Select Page:', 'clean-commerce-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => esc_html__( '&mdash; Select &mdash;', 'clean-commerce-pro' ),
					);
				$fields[ 'block_icon_' . $i ] = array(
					'label'       => esc_html__( 'Icon:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'Eg: fa-cogs', 'clean-commerce-pro' ),
					'type'        => 'text',
					'default'     => 'fa-cogs',
					);
			}

			parent::__construct( 'clean-commerce-services', esc_html__( 'CC: Services', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$service_arr = array();
			for ( $i = 0; $i < 8 ; $i++ ) {
				$block = ( $i + 1 );
				$service_arr[ $i ] = array(
					'page' => $params[ 'block_page_' . $block ],
					'icon' => $params[ 'block_icon_' . $block ],
					);
			}
			$refined_arr = array();
			if ( ! empty( $service_arr ) ) {
				foreach ( $service_arr as $item ) {
					if ( ! empty( $item['page'] ) ) {
						$refined_arr[ $item['page'] ] = $item;
					}
				}
			}


			if ( ! empty( $refined_arr ) ) {
				$this->render_widget_content( $refined_arr, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Render services content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $service_arr Services array.
		 * @param array $params      Parameters array.
		 */
		function render_widget_content( $service_arr, $params ) {

			$column = $params['post_column'];

			$page_ids = array_keys( $service_arr );

			$qargs = array(
				'post__in'       => $page_ids,
				'posts_per_page' => count( $page_ids ),
				'post_type'      => 'page',
				'orderby'        => 'post__in',
				'no_found_rows'  => true,
			);

			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) : ?>

				<?php global $post; ?>

				<div class="service-block-list service-col-<?php echo esc_attr( $column ); ?>">
					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $post ) :  ?>
							<?php setup_postdata( $post ); ?>
							<div class="service-block-item">
								<div class="service-block-inner">
									<?php if ( isset( $service_arr[ $post->ID ]['icon'] ) && ! empty( $service_arr[ $post->ID ]['icon'] ) ) : ?>
										<a href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>"><i class="<?php echo 'fa ' . esc_attr( $service_arr[ $post->ID ]['icon'] ); ?>"></i></a>
									<?php endif; ?>
									<div class="service-block-inner-content">
										<h3 class="service-item-title">
											<a href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>">
												<?php echo get_the_title( $post->ID ); ?>
											</a>
										</h3>
										<?php if ( true !== $params['disable_excerpt'] ) :  ?>
											<div class="service-block-item-excerpt">
												<?php
												$excerpt = clean_commerce_the_excerpt( $params['excerpt_length'], $post );
												echo wp_kses_post( wpautop( $excerpt ) );
												?>
											</div><!-- .service-block-item-excerpt -->
										<?php endif; ?>

									</div><!-- .service-block-inner-content -->
								</div><!-- .service-block-inner -->
							</div><!-- .service-block-item -->
						<?php endforeach; ?>

					</div><!-- .inner-wrapper -->

				</div><!-- .service-block-list -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
		}

	}
endif;

if ( ! class_exists( 'Clean_Commerce_Testimonial_Slider_Widget' ) ) :

	/**
	 * Testimonial Slider widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Testimonial_Slider_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'clean_commerce_widget_testimonial_slider',
				'description' => esc_html__( 'Displays Testimonials as a Slider.', 'clean-commerce-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => esc_html__( 'Select Category:', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'clean-commerce-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Posts:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'in words', 'clean-commerce-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 30,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'slider_heading' => array(
					'label'   => esc_html__( 'SLIDER OPTIONS', 'clean-commerce-pro' ),
					'type'    => 'heading',
					),
				'transition_delay' => array(
					'label'       => esc_html__( 'Transition Delay:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'in seconds', 'clean-commerce-pro' ),
					'type'        => 'number',
					'default'     => 3,
					'css'         => 'max-width:50px;',
					'min'         => 1,
					'max'         => 10,
					'adjacent'         => true,
					),
				'enable_autoplay' => array(
					'label'   => esc_html__( 'Enable Autoplay', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				);

			parent::__construct( 'clean-commerce-testimonial-slider', esc_html__( 'CC: Testimonial Slider', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$testimonial_posts = $this->get_testimonials( $params );

			if ( ! empty( $testimonial_posts ) ) {
				$this->render_testimonials( $testimonial_posts, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Return testimonial posts detail.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return array Posts details.
		 */
		function get_testimonials( $params ) {

			$output = array();

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$all_posts = get_posts( $qargs );

			if ( ! empty( $all_posts ) ) {
				$cnt = 0;
				global $post;
				foreach ( $all_posts as $key => $post ) {

					setup_postdata( $post );

					$item = array();
					$item['title']   = get_the_title( $post->ID );
					$item['excerpt'] = clean_commerce_the_excerpt( absint( $params['excerpt_length'] ), $post );
					$item['image']   = null;
					if ( has_post_thumbnail( $post->ID ) ) {
						$image_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'thumbnail' );
						if ( ! empty( $image_detail ) ) {
							$item['image'] = $image_detail;
						}
					}
					$output[ $cnt ] = $item;
					$cnt++;

				}
				wp_reset_postdata();
			}

			return $output;

		}

		/**
		 * Render testimonial slider.
		 *
		 * @since 1.0.0
		 *
		 * @param array $testimonials Testimonials.
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_testimonials( $testimonials, $params ) {

			$timeout = 0;
			if ( true === $params['enable_autoplay'] ) {
				$timeout = 1000 * absint( $params['transition_delay'] );
			}
			?>
			<div class="cycle-slideshow"
				data-cycle-slides="> article"
				data-cycle-auto-height="container"
				data-cycle-timeout="<?php echo absint( $timeout ); ?>"
				>
				<div class="cycle-pager"></div>
				<?php foreach ( $testimonials as $testimonial ) : ?>
					<article>
						<div class="testimonial-wrapper">
							<?php if ( ! empty( $testimonial['image'] ) ) : ?>
								<div class="testimonial-thumb">
								<img src="<?php echo esc_url( $testimonial['image'][0] ); ?>" alt="<?php echo esc_attr( $testimonial['title'] ); ?>" />
								</div>
							<?php endif; ?>
							<div class="testimonial-content-area">
							<?php if ( ! empty( $testimonial['excerpt'] ) ) : ?>
								<div class="testimonial-excerpt">
									<?php echo wp_kses_post( wpautop( $testimonial['excerpt'] ) ); ?>
								</div><!-- .testimonial-excerpt -->
							<?php endif; ?>
							</div> <!-- .testimonial-content-area -->

							<h4><?php echo esc_html( $testimonial['title'] ); ?></h4>
						</div>
					</article>

				<?php endforeach; ?>

			</div><!-- .cycle-slideshow -->
			<?php

		}

	}
endif;

if ( ! class_exists( 'Clean_Commerce_Recent_Posts_Widget' ) ) :

	/**
	 * Recent posts widget Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Recent_Posts_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_recent_posts',
				'description'                 => esc_html__( 'Displays recent posts.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => esc_html__( 'Select Category:', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'clean-commerce-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Posts:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'thumbnail',
					'options' => clean_commerce_get_image_sizes_options( true, array( 'disable', 'thumbnail' ), false ),
					),
				'image_width' => array(
					'label'       => esc_html__( 'Image Width:', 'clean-commerce-pro' ),
					'type'        => 'number',
					'description' => esc_html__( 'px', 'clean-commerce-pro' ),
					'css'         => 'max-width:60px;',
					'adjacent'    => true,
					'default'     => 90,
					'min'         => 1,
					'max'         => 150,
					),
				'disable_date' => array(
					'label'   => esc_html__( 'Disable Date', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			parent::__construct( 'clean-commerce-recent-posts', esc_html__( 'CC: Recent Posts', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = $params['post_category'];
			}
			$all_posts = get_posts( $qargs );

			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>

				<?php global $post; ?>

				<div class="recent-posts-wrapper">

					<?php foreach ( $all_posts as $key => $post ) :  ?>
						<?php setup_postdata( $post ); ?>

						<div class="recent-posts-item">

							<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) :  ?>
								<div class="recent-posts-thumb">
									<a href="<?php the_permalink(); ?>">
										<?php
										$img_attributes = array(
											'class' => 'alignleft',
											'style' => 'max-width:' . esc_attr( $params['image_width'] ). 'px;',
											);
										the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
										?>
									</a>
								</div><!-- .recent-posts-thumb -->
							<?php endif ?>
							<div class="recent-posts-text-wrap">
								<h3 class="recent-posts-title">
									<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
								</h3><!-- .recent-posts-title -->

								<?php if ( false === $params['disable_date'] ) :  ?>
									<div class="recent-posts-meta">

										<?php if ( false === $params['disable_date'] ) :  ?>
											<span class="recent-posts-date"><?php the_time( get_option( 'date_format' ) ); ?></span><!-- .recent-posts-date -->
										<?php endif; ?>

									</div><!-- .recent-posts-meta -->
								<?php endif; ?>

							</div><!-- .recent-posts-text-wrap -->

						</div><!-- .recent-posts-item -->

					<?php endforeach; ?>

				</div><!-- .recent-posts-wrapper -->

				<?php wp_reset_postdata(); // Reset. ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Featured_Products_Carousel_Widget' ) ) :

	/**
	 * Featured Products Carousel Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Featured_Products_Carousel_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'clean_commerce_widget_featured_products_carousel',
				'description' => esc_html__( 'Displays featured products as carousel.', 'clean-commerce-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Products:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				);

			parent::__construct( 'clean-commerce-featured-products-carousel', esc_html__( 'CC: Featured Products Carousel', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			// Render now.
			$this->render_products( $params );

			echo $args['after_widget'];

		}

		/**
		 * Render products.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_products( $params ) {

			$carousel_args = array(
				'slidesToShow' => 4,
				'dots'         => false,
				'prevArrow'    => '<span data-role="none" class="slick-prev" tabindex="0"><i class="fa fa-angle-left" aria-hidden="true"></i></span>',
				'nextArrow'    => '<span data-role="none" class="slick-next" tabindex="0"><i class="fa fa-angle-right" aria-hidden="true"></i></span>',
				'responsive'   => array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow' => 4,
							),
						),
					array(
						'breakpoint' => 800,
						'settings'   => array(
							'slidesToShow' => 3,
							),
						),
					array(
						'breakpoint' => 659,
						'settings'   => array(
							'slidesToShow' => 2,
							),
						),
					array(
						'breakpoint' => 479,
						'settings'   => array(
							'slidesToShow' => 1,
							),
						),
					),
				);

			$carousel_args_encoded = wp_json_encode( $carousel_args );

			$meta_query = WC()->query->get_meta_query();
			$tax_query  = WC()->query->get_tax_query();

			$tax_query[] = array(
				'taxonomy' => 'product_visibility',
				'field'    => 'name',
				'terms'    => 'featured',
				'operator' => 'IN',
			);

			$query_args = array(
				'post_type'           => 'product',
				'post_status'         => 'publish',
				'ignore_sticky_posts' => 1,
				'posts_per_page'      => absint( $params['post_number'] ),
				'meta_query'          => $meta_query,
				'tax_query'           => $tax_query,
				'no_found_rows'       => true,
			);

			global $woocommerce_loop;
			$products = new WP_Query( $query_args );

			if ( $products->have_posts() ) {
				?>
				<div class="clean-commerce-woocommerce">
				<div class="clean-commerce-woocommmerce-wrapper">
				<ul class="products clean-commerce-featured-products-carousel" data-slick='<?php echo $carousel_args_encoded; ?>'>

				<?php while ( $products->have_posts() ) : $products->the_post(); ?>

					<?php wc_get_template_part( 'content', 'product' ); ?>

				<?php endwhile; // end of the loop. ?>
				</ul><!-- .products -->
				</div>
				</div><!-- .woocommerce -->
				<?php
			}
			woocommerce_reset_loop();
			wp_reset_postdata();

		}
	}
endif;


if ( ! class_exists( 'Clean_Commerce_Products_Grid_Widget' ) ) :

	/**
	 * Products Grid Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Products_Grid_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_products_grid',
				'description'                 => esc_html__( 'Displays products in grid.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'product_category' => array(
					'label'           => esc_html__( 'Select Product Category:', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'taxonomy'        => 'product_cat',
					'show_option_all' => esc_html__( 'All Product Categories', 'clean-commerce-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Products:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => esc_html__( 'Number of Columns:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 4,
					'options' => clean_commerce_get_numbers_dropdown_options( 3, 4 ),
					),
				);

			parent::__construct( 'clean-commerce-products-grid', esc_html__( 'CC: Products Grid', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			// Render now.
			$this->render_products( $params );

			echo $args['after_widget'];

		}

		/**
		 * Render products.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_products( $params ) {

			$query_args = array(
				'post_type'           => 'product',
				'post_status'         => 'publish',
				'ignore_sticky_posts' => 1,
				'posts_per_page'      => esc_attr( $params['post_number'] ),
				'no_found_rows'       => true,
			);
			if ( absint( $params['product_category'] ) > 0 ) {
				$query_args['tax_query'] = array(
					array(
						'taxonomy' => 'product_cat',
						'field'    => 'term_id',
						'terms'    => absint( $params['product_category'] ),
						),
					);
			}

			global $woocommerce_loop;
			$products = new WP_Query( $query_args );

			if ( $products->have_posts() ) {
				?>
				<div class="inner-wrapper">
					<div class="clean-commerce-woocommerce clean-commerce-woocommerce-product-grid-<?php echo esc_attr( $params['post_column'] ); ?>">

						<ul class="products">

							<?php while ( $products->have_posts() ) : $products->the_post(); ?>

								<?php wc_get_template_part( 'content', 'product' ); ?>

							<?php endwhile; ?>

						</ul><!-- .products -->

					</div><!-- .woocommerce -->
				</div> <!-- .inner-wrapper -->
				<?php
			}
			woocommerce_reset_loop();
			wp_reset_postdata();

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Products_Slider_Widget' ) ) :

	/**
	 * Products Slider Class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Products_Slider_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'clean_commerce_widget_products_slider',
				'description' => esc_html__( 'Displays products slider.', 'clean-commerce-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'product_category' => array(
					'label'           => esc_html__( 'Select Product Category:', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'taxonomy'        => 'product_cat',
					'show_option_all' => esc_html__( 'All Product Categories', 'clean-commerce-pro' ),
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Products:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'featured_image' => array(
					'label'   => esc_html__( 'Select Image:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'large',
					'options' => clean_commerce_get_image_sizes_options( false ),
					),
				'featured_image_message' => array(
					'label' => sprintf( esc_html__( 'Recommended size: %1$dpx X %2$dpx', 'clean-commerce-pro' ), 820, 440 ),
					'type'  => 'message',
					),
				'transition_delay' => array(
					'label'       => esc_html__( 'Transition Delay:', 'clean-commerce-pro' ),
					'description' => esc_html__( 'in seconds', 'clean-commerce-pro' ),
					'type'        => 'number',
					'default'     => 3,
					'css'         => 'max-width:50px;',
					'min'         => 1,
					'max'         => 10,
					'adjacent'         => true,
					),
				'enable_autoplay' => array(
					'label'   => esc_html__( 'Enable Autoplay', 'clean-commerce-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'side_banner_heading' => array(
					'label'   => esc_html__( 'SIDE BANNER', 'clean-commerce-pro' ),
					'type'    => 'heading',
					),
				'side_banner_image_url' => array(
					'label'       => esc_html__( 'Banner Image:', 'clean-commerce-pro' ),
					'description' => sprintf( esc_html__( 'Recommended size: %1$dpx X %2$dpx', 'clean-commerce-pro' ), 335, 440 ),
					'type'        => 'image',
					),
				'side_banner_link_url' => array(
					'label' => esc_html__( 'Link:', 'clean-commerce-pro' ),
					'type'  => 'url',
					),
				);

			parent::__construct( 'clean-commerce-products-slider', esc_html__( 'CC: Products Slider', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			// Render now.
			$this->render_products( $params );

			echo $args['after_widget'];

		}

		/**
		 * Render products.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_products( $params ) {

			$query_args = array(
				'post_type'           => 'product',
				'post_status'         => 'publish',
				'ignore_sticky_posts' => true,
				'posts_per_page'      => esc_attr( $params['post_number'] ),
				'no_found_rows'       => true,
				'meta_query'          => array(
					array( 'key' => '_thumbnail_id' ),
				),
			);

			if ( absint( $params['product_category'] ) > 0 ) {
				$query_args['tax_query'] = array(
					array(
						'taxonomy' => 'product_cat',
						'field'    => 'term_id',
						'terms'    => absint( $params['product_category'] ),
						),
					);
			}

			$products = get_posts( $query_args );

			if ( ! empty( $products ) ) {
				$timeout = 0;

				if ( true === $params['enable_autoplay'] ) {
					$timeout = 1000 * absint( $params['transition_delay'] );
				}

				$banner_status_class = 'side-banner-disabled';

				if ( ! empty( $params['side_banner_image_url'] ) ) {
					$banner_status_class = 'side-banner-enabled';
				}
				?>
					<div class="cycle-slideshow <?php echo esc_attr( $banner_status_class ); ?>"
						data-cycle-slides="> article"
						data-cycle-auto-height="container"
						data-cycle-timeout="<?php echo absint( $timeout ); ?>"
						>
						<div class="cycle-prev"><i class="fa fa-angle-left" aria-hidden="true"></i></div>
						<div class="cycle-next"><i class="fa fa-angle-right" aria-hidden="true"></i></div>
						<?php foreach ( $products as $product ) : ?>
							<article>
								<div class="product-wrapper">
									<?php
									$image_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $product->ID ), esc_attr( $params['featured_image'] ) );

									if ( ! empty( $image_detail ) ) {
										echo '<img src="' . esc_url( $image_detail[0] ) . '" alt="" />';
									}
									?>
									<div class="product-caption">
										<h4 class="product-title"><a href="<?php echo esc_url( get_permalink( $product->ID) ); ?>"><?php echo get_the_title( $product->ID ); ?></a></h4>
										<?php
											$excerpt = clean_commerce_the_excerpt( 10, $product );
											echo wp_kses_post( wpautop( $excerpt ) );
											?>
										<a class="button" href="<?php echo esc_url( get_permalink( $product->ID) ); ?>" class="product-read-more"><?php esc_html_e( 'View details', 'clean-commerce-pro' ); ?></a>
									</div><!-- .product-caption -->
								</div><!-- .product-wrapper -->
							</article>

						<?php endforeach; ?>

					</div><!-- .cycle-slideshow -->
				<?php
			}
			?>
			<?php if ( ! empty( $params['side_banner_image_url'] ) ) : ?>
				<div class="side-banner">
					<?php
						$link_open  = '';
						$link_close = '';

						if ( ! empty( $params['side_banner_link_url'] ) ) {
							$link_open  = '<a href="' . esc_url( $params['side_banner_link_url'] ) . '">';
							$link_close = '</a>';
						}

						echo $link_open;
						echo '<img src="' . esc_url( $params['side_banner_image_url'] ) . '" alt="" />';
						echo $link_close;
					?>
				</div> <!-- .side-banner -->
			<?php endif; ?>
			<?php

		}
	}
endif;

if ( ! class_exists( 'Clean_Commerce_Products_Category_Grid_Widget' ) ) :

	/**
	 * Products category grid widget class.
	 *
	 * @since 1.0.0
	 */
	class Clean_Commerce_Products_Category_Grid_Widget extends Clean_Commerce_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'clean_commerce_widget_products_category_grid',
				'description'                 => esc_html__( 'Displays products category grid.', 'clean-commerce-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => esc_html__( 'Subtitle:', 'clean-commerce-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_number' => array(
					'label'   => esc_html__( 'Number of Categories:', 'clean-commerce-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => esc_html__( 'Number of Columns:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 4,
					'options' => clean_commerce_get_numbers_dropdown_options( 3, 4 ),
					),
				'product_category' => array(
					'label'           => esc_html__( 'Select Product Category:', 'clean-commerce-pro' ),
					'description'     => esc_html__( 'Hold down the Ctrl (Windows) / Command (Mac) button to select multiple options.', 'clean-commerce-pro' ),
					'type'            => 'dropdown-taxonomies',
					'multiple'        => true,
					'taxonomy'        => 'product_cat',
					'show_option_all' => esc_html__( 'All Product Categories', 'clean-commerce-pro' ),
					'default'         => array(),
					),
				'featured_image' => array(
					'label'   => esc_html__( 'Select Image Size:', 'clean-commerce-pro' ),
					'type'    => 'select',
					'default' => 'medium',
					'options' => clean_commerce_get_image_sizes_options(),
					),
				);

			parent::__construct( 'clean-commerce-products-category-grid', esc_html__( 'CC: Products Category Grid', 'clean-commerce-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			// Render now.
			$this->render_categories( $params );

			echo $args['after_widget'];

		}

		/**
		 * Render categories.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_categories( $params ) {

			$query_args = array();

			$query_args['number']       = absint( $params['post_number'] );
			$query_args['hierarchical'] = false;

			$product_category = (array) $params['product_category'];
			$product_category = array_filter( $product_category );

			if ( ! empty( $product_category ) ) {
				$query_args['include'] = $product_category;
			}

			$taxonomies = get_terms( 'product_cat', $query_args );

			if ( ! is_wp_error( $taxonomies) && ! empty( $taxonomies ) ) {
				?>
				<div class="inner-wrapper">
					<div class="clean-commerce-products-category products-category-grid-<?php echo absint( $params['post_column'] ); ?>">

						<div class="categories">
							<?php foreach ( $taxonomies as $tax ) : ?>

								<div class="category-item">
									<a href="<?php echo esc_url( get_term_link( $tax ) ); ?>">
										<?php
										$image_url = get_template_directory_uri() . '/images/no-image-category.png';
										$thumbnail_id = get_woocommerce_term_meta( $tax->term_id, 'thumbnail_id', true );
										if ( $thumbnail_id ) {
											$image_array = wp_get_attachment_image_src( $thumbnail_id, esc_attr( $params['featured_image'] ) );
											if ( ! empty( $image_array ) ) {
												$image_url = $image_array[0];
											}
										}
										?>
										<img src="<?php echo esc_url( $image_url ); ?>" alt="" />
									</a>
									<h4 class="category-title"><a href="<?php echo esc_url( get_term_link( $tax ) ); ?>"><?php echo esc_html( $tax->name ); ?>&nbsp;(<?php echo absint( $tax->count );  ?>)</a></h4>
								</div><!-- .category-item -->

							<?php endforeach; ?>

						</div><!-- .categories -->

					</div><!-- .clean-commerce-products-category -->
				</div> <!-- .inner-wrapper -->
				<?php
			}

		}

	}
endif;
