<?php
/**
 * Gallery slider Widget
 *
 * @package Theme Palace
 * @subpackage Travel Insight Pro
 * @since Travel Insight Pro 1.0
 */

if ( ! class_exists( 'Travel_Insight_Pro_Gallery_slider_Widget' ) ) :

/**
 * Gallery slider Class
 *
 * @since 1.0
 *
 * @see WP_Widget
 */
class Travel_Insight_Pro_Gallery_slider_Widget extends WP_Widget {

	/**
	 * Sets up a new gallery slider widget instance.
	 *
	 * @since 1.0
	 * @access public
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'                   => 'widget_gallery_slider',
			'description'                 => esc_html__( 'Shows Your site&#8217;s gallery post format images.','travel-insight-pro' ),
			'customize_selective_refresh' => true,
		);
		parent::__construct( 'tp-widget-gallery-slider', esc_html__( 'TP: Gallery Slider Widget', 'travel-insight-pro' ), $widget_ops );
		$this->alt_option_name = 'widget_gallery_slider';
	}

	/**
	 * Outputs the content for the gallery slider widget instance.
	 *
	 * @since 1.0
	 * @access public
	 *
	 * @param array $args     Display arguments including 'before_title', 'after_title',
	 *                        'before_widget', and 'after_widget'.
	 * @param array $instance Settings for the gallery slider widget instance.
	 */
	public function widget( $args, $instance ) {
		$post_id = isset( $instance['post_id'] ) ? $instance['post_id'] : 0;
		$show_image_title = isset( $instance['show_image_title'] ) ? $instance['show_image_title'] : true;
		if ( ! has_post_format( 'gallery', $post_id ) ) return;

		$attachments    = get_post_gallery( $post_id, false );

		if( empty($attachments['ids'] ) ) return;
		
		$attachment_ids = explode( ',', $attachments['ids'] );

		if ( empty( $attachments ) ) return;

		if ( ! isset( $args['widget_id'] ) )
		$args['widget_id'] = $this->id;

		$title             = ( !empty( $instance['title'] ) ) ? $instance['title'] : esc_html__( 'Gallery Slider', 'travel-insight-pro' );

		/** This filter is documented in wp-includes/widgets/class-wp-widget-pages.php */
		$title             = apply_filters( 'widget_title', $title, $instance, $this->id_base );

		echo $args['before_widget'];
		if ( ! empty( $title ) ) {
			echo $args['before_title'] . esc_html( $title ) . $args['after_title'];
		}
		?>
		<div class="widget-wrap gallery-widget">
            <div class="regular" data-effect="cubic-bezier(0.680, 0, 0.265, 1)" data-slick='{"slidesToShow": 1, "slidesToScroll": 1, "speed": "800", "dots": false, "arrows": true, "infinite": true, "autoplay": true}'>
            	<?php foreach ( $attachment_ids as $attachment_id ) : ?>
                <div class="slider-item">
                  	<?php echo wp_get_attachment_link( $attachment_id, 'medium_large', true ); 
                  	if( $show_image_title == true ) : ?>
                  	<div class="slider-title">
                    	<a href="<?php echo esc_url( get_the_permalink( $post_id ) );  ?>"><h6><?php echo esc_html( get_the_title( $attachment_id ) ); ?></h6></a>
                  	</div>
                  	<?php endif; ?>
                </div><!-- .slider-item -->
                <?php endforeach; ?>
            </div><!-- .entry-content-->
        </div><!-- .widget-wrap -->
		<?php 
		echo $args['after_widget'];
	}

	/**
	 * Handles updating settings for the gallery slider widget instance.
	 *
	 * @since 1.0
	 * @access public
	 *
	 * @param array $new_instance New settings for this instance as input by the user via
	 *                            WP_Widget::form().
	 * @param array $old_instance Old settings for this instance.
	 * @return array Updated settings to save.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance            = $old_instance;
		$instance['title']   = sanitize_text_field( $new_instance['title'] );
		$instance['post_id'] = absint( $new_instance['post_id'] );
		$instance['show_image_title']       = travel_insight_pro_sanitize_checkbox( $new_instance['show_image_title'] );
		return $instance;
	}

	/**
	 * Outputs the settings form for the gallery slider widget.
	 *
	 * @since 1.0
	 * @access public
	 *
	 * @param array $instance Current settings.
	 */
	public function form( $instance ) {

		$defaults = array( 'title' => '', 'post_id' => 0, 'show_image_title' => false );
		$instance = wp_parse_args ( $instance , $defaults );
		extract( $instance );

		$gallery_posts = get_posts( array(
			'numberposts'	=>	-1,
			'tax_query'		=>	array(
				array(
					'taxonomy'	=>	'post_format',
					'field'		=>	'slug',
					'terms'		=>	array( 'post-format-gallery' )
				)
			)
		) );

		if ( empty( $gallery_posts ) ) {
			echo '<p class="description">'. sprintf( esc_html__( 'No galleries have been created yet. %1$s%2$s%3$s.', 'travel-insight-pro' ), '<a href="' . esc_url( admin_url( 'post-new.php' ) ) .'">',esc_html__( 'Create Some','travel-insight-pro' ), '</a>' ) . '</p>';
			return;
		}
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'travel-insight-pro' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'post_id' ) ); ?>"><?php esc_html_e( 'Select Gallery:', 'travel-insight-pro' ); ?></label>
			<select name="<?php echo esc_attr( $this->get_field_name( 'post_id' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'post_id' ) ); ?>" class="widefat">
				<?php foreach ( $gallery_posts as $gallery_post ) {
					echo '<option value="' . esc_attr( $gallery_post->ID ) . '"' . selected( $post_id, $gallery_post->ID ) .'>' . esc_html( $gallery_post->post_title ). '</option>';
				} ?>
			</select>
			<span><b><?php esc_html_e( 'Create gallery in selected post and select the similar sized images.', 'travel-insight-pro' ); ?></b></span>
		</p>

		<p>			
			<label for="<?php echo esc_attr( $this->get_field_id("show_image_title") ); ?>">	
				<input type="checkbox" class="checkbox" id="<?php echo esc_attr( $this->get_field_id("show_image_title") ); ?>" name="<?php echo esc_attr( $this->get_field_name("show_image_title") ); ?>" value="1" <?php if ( isset( $show_image_title ) ) { checked( 1, $show_image_title, true ); } ?> />	
				<?php esc_html_e( 'Show Image Title', 'travel-insight-pro'); ?>	
			</label>	
		</p>
		<?php
	}
}
endif;
/*
 * Function to register counter widget
 *
 * Since 1.0
 */
function travel_insight_pro_register_gallery_slider_widget(){
	register_widget( 'Travel_Insight_Pro_Gallery_slider_Widget' );
}
add_action( 'widgets_init', 'travel_insight_pro_register_gallery_slider_widget' );