<?php
/**
 * Basic theme functions.
 *
 * This file contains hook functions attached to core hooks.
 *
 * @package Magazine_Plus
 */

if ( ! function_exists( 'magazine_plus_custom_body_class' ) ) :
	/**
	 * Custom body class.
	 *
	 * @since 1.0.0
	 *
	 * @param string|array $input One or more classes to add to the class list.
	 * @return array Array of classes.
	 */
	function magazine_plus_custom_body_class( $input ) {

		// Adds a class of group-blog to blogs with more than 1 published author.
		if ( is_multi_author() ) {
			$input[] = 'group-blog';
		}

		// Site Design.
		$site_design = magazine_plus_get_option( 'site_design' );
		$input[]     = 'site-design-' . esc_attr( $site_design );

		// Site layout.
		$site_layout = magazine_plus_get_option( 'site_layout' );
		$input[] = 'site-layout-' . esc_attr( $site_layout );

		// Global layout.
		global $post;
		$global_layout = magazine_plus_get_option( 'global_layout' );
		$global_layout = apply_filters( 'magazine_plus_filter_theme_global_layout', $global_layout );

		// Check if single.
		if ( $post  && is_singular() ) {
			$post_options = get_post_meta( $post->ID, 'magazine_plus_theme_settings', true );
			if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
				$global_layout = $post_options['post_layout'];
			}
		}

		$input[] = 'global-layout-' . esc_attr( $global_layout );

		// Common class for three columns.
		switch ( $global_layout ) {
			case 'three-columns':
			case 'three-columns-pcs':
			case 'three-columns-cps':
			case 'three-columns-psc':
			case 'three-columns-pcs-equal':
			case 'three-columns-scp-equal':
				$input[] = 'three-columns-enabled';
				break;

			default:
				break;
		}

		// Sticky menu.
		$enable_sticky_primary_menu = magazine_plus_get_option( 'enable_sticky_primary_menu' );

		if ( true === $enable_sticky_primary_menu) {
			$input[] = 'enabled-sticky-primary-menu';
		}

		if ( true === magazine_plus_get_option( 'show_social_in_header' ) && has_nav_menu( 'social' ) ) {
			$input[] = 'header-social-active';
		}

		return $input;

	}
endif;

add_filter( 'body_class', 'magazine_plus_custom_body_class' );

if ( ! function_exists( 'magazine_plus_custom_content_width' ) ) :

	/**
	 * Custom content width.
	 *
	 * @since 1.0.0
	 */
	function magazine_plus_custom_content_width() {

		global $content_width;

		$global_layout = magazine_plus_get_option( 'global_layout' );
		$global_layout = apply_filters( 'magazine_plus_filter_theme_global_layout', $global_layout );

		switch ( $global_layout ) {

			case 'no-sidebar':
				$content_width = 1220;
				break;

			case 'three-columns':
			case 'three-columns-pcs':
			case 'three-columns-cps':
			case 'three-columns-psc':
			case 'no-sidebar-centered':
				$content_width = 570;
				break;

			case 'three-columns-pcs-equal':
			case 'three-columns-scp-equal':
				$content_width = 363;
				break;

			case 'left-sidebar':
			case 'right-sidebar':
				$content_width = 895;
				break;

			default:
				break;
		}

	}
endif;

add_filter( 'template_redirect', 'magazine_plus_custom_content_width' );

if ( ! function_exists( 'magazine_plus_implement_excerpt_length' ) ) :

	/**
	 * Implement excerpt length.
	 *
	 * @since 1.0.0
	 *
	 * @param int $length The number of words.
	 * @return int Excerpt length.
	 */
	function magazine_plus_implement_excerpt_length( $length ) {

		$excerpt_length = magazine_plus_get_option( 'excerpt_length' );
		if ( empty( $excerpt_length ) ) {
			$excerpt_length = $length;
		}
		return apply_filters( 'magazine_plus_filter_excerpt_length', esc_attr( $excerpt_length ) );

	}

endif;

if ( ! function_exists( 'magazine_plus_implement_read_more' ) ) :

	/**
	 * Implement read more in excerpt.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more The string shown within the more link.
	 * @return string The excerpt.
	 */
	function magazine_plus_implement_read_more( $more ) {

		$flag_apply_excerpt_read_more = apply_filters( 'magazine_plus_filter_excerpt_read_more', true );

		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more;
		}

		$output = $more;
		$read_more_text = magazine_plus_get_option( 'read_more_text' );

		if ( ! empty( $read_more_text ) ) {
			$output = ' <a href="'. esc_url( get_permalink() ) . '" class="read-more">' . esc_html( $read_more_text ) . '</a>';
			$output = apply_filters( 'magazine_plus_filter_read_more_link' , $output );
		}

		return $output;

	}

endif;

if ( ! function_exists( 'magazine_plus_content_more_link' ) ) :

	/**
	 * Implement read more in content.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more_link Read More link element.
	 * @param string $more_link_text Read More text.
	 * @return string Link.
	 */
	function magazine_plus_content_more_link( $more_link, $more_link_text ) {

		$flag_apply_excerpt_read_more = apply_filters( 'magazine_plus_filter_excerpt_read_more', true );

		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more_link;
		}

		$read_more_text = magazine_plus_get_option( 'read_more_text' );

		if ( ! empty( $read_more_text ) ) {
			$more_link = str_replace( $more_link_text, esc_html( $read_more_text ), $more_link );
		}

		return $more_link;

	}

endif;

if ( ! function_exists( 'magazine_plus_hook_read_more_filters' ) ) :

	/**
	 * Hook read more filters.
	 *
	 * @since 1.0.0
	 */
	function magazine_plus_hook_read_more_filters() {

		if ( is_home() || is_category() || is_tag() || is_author() || is_date() ) {
			add_filter( 'excerpt_length', 'magazine_plus_implement_excerpt_length', 999 );
			add_filter( 'the_content_more_link', 'magazine_plus_content_more_link', 10, 2 );
			add_filter( 'excerpt_more', 'magazine_plus_implement_read_more' );
		}

	}
endif;

add_action( 'wp', 'magazine_plus_hook_read_more_filters' );

if ( ! function_exists( 'magazine_plus_customizer_reset_callback' ) ) :

	/**
	 * Callback for reset in Customizer.
	 *
	 * @since 1.0.0
	 */
	function magazine_plus_customizer_reset_callback() {

		$reset_all_settings = magazine_plus_get_option( 'reset_all_settings' );

		if ( true === $reset_all_settings ) {

			// Reset custom theme options.
			set_theme_mod( 'theme_options', array() );

			// Reset custom header, logo and backgrounds.
			remove_theme_mod( 'custom_logo' );
			remove_theme_mod( 'header_image' );
			remove_theme_mod( 'header_image_data' );
			remove_theme_mod( 'background_image' );
			remove_theme_mod( 'background_color' );
		}

		// Reset color options.
		$reset_color_settings = magazine_plus_get_option( 'reset_color_settings' );
		if ( true === $reset_color_settings ) {
			$options = magazine_plus_get_options();
			$options['reset_color_settings'] = false;
			$color_fields = magazine_plus_get_color_theme_settings_options();
			$default = magazine_plus_get_default_theme_options();
			if ( ! empty( $color_fields ) ) {
				foreach ( $color_fields as $key => $val ) {
					$options[ $key ] = $default[ $key ];
				}
			}
			set_theme_mod( 'theme_options', $options );
			remove_theme_mod( 'background_color' );
		}

		// Reset font options.
		$reset_font_settings = magazine_plus_get_option( 'reset_font_settings' );
		if ( true === $reset_font_settings ) {
			$options = magazine_plus_get_options();
			$options['reset_font_settings'] = false;
			$font_settings = magazine_plus_get_font_family_theme_settings_options();
			if ( ! empty( $font_settings ) ) {
				foreach ( $font_settings as $key => $val ) {
					$options[ $key ] = $val['default'];
				}
			}
			set_theme_mod( 'theme_options', $options );
		}

		// Reset footer content.
		$reset_footer_content = magazine_plus_get_option( 'reset_footer_content' );
		if ( true === $reset_footer_content ) {
			$options = magazine_plus_get_options();
			$options['reset_footer_content'] = false;
	  		$default = magazine_plus_get_default_theme_options();
	  		$footer_fields = array(
				'copyright_text',
				'powered_by_text',
			);
	  		foreach ( $footer_fields as $field ) {
		  		if ( isset( $default[ $field ] ) ) {
		  			$options[ $field ] = $default[ $field ];
		  		}
	  		}
			set_theme_mod( 'theme_options', $options );
		}

	}
endif;

add_action( 'customize_save_after', 'magazine_plus_customizer_reset_callback' );

/**
 * Load theme options from free theme.
 *
 * Checks if there are options already present from free version and adds it to the Pro theme options.
 *
 * @since 1.4
 */
function magazine_plus_import_free_options( $old_theme_name ) {
	$free_version_slug = array(
		'magazine-plus',
		'moto-news',
	);

	$old_theme_slug = sanitize_title( $old_theme_name );

	if ( ! in_array( $old_theme_slug, $free_version_slug ) ) {
		// Bail if free theme s not of this pro theme.
		return;
	}

	// Perform action only if theme_mods_XXX[theme_options] does not exist.
	if( ! get_theme_mod( 'theme_options' ) ) {
		$free_options = get_option( 'theme_mods_' . $old_theme_slug );

		// Perform action only if theme_mods_XXX free version exists.

		if ( $free_options ) {
			if ( isset( $free_options['theme_options'] ) ) {
				if ( 'moto-news' === $old_theme_slug ) {
					$free_options['theme_options']['site_design'] = 'moto-news';
				}

				$pro_default_options = magazine_plus_get_default_theme_options();

				$pro_theme_options = $free_options;

				$pro_theme_options['theme_options'] = array_merge( $pro_default_options , $free_options['theme_options'] );

				// WP default fields.
				$fields = array(
					'custom_logo',
					'header_image',
					'header_image_data',
					'background_image',
					'background_repeat',
					'background_position_x',
					'background_attachment',
				);

				foreach ( $fields as $key ) {
					if ( isset( $free_options[ $key ] ) && ! empty( $free_options[ $key ] ) ) {
						$pro_theme_options[ $key ] = $free_options[ $key ];
					}
				}

				update_option( 'theme_mods_magazine-plus-pro', $pro_theme_options );
			}
		}
	}
}
add_action( 'after_switch_theme', 'magazine_plus_import_free_options' );

if ( ! function_exists( 'magazine_plus_custom_links_in_navigation' ) ) :

	/**
	 * Add custom links in navigation.
	 *
	 * @since 1.0.0
	 *
	 * @param string $items The HTML list content for the menu items.
	 * @param object $args  An object containing wp_nav_menu() arguments.
	 * @return string Modified HTML list content for the menu items.
	 */
	function magazine_plus_custom_links_in_navigation( $items, $args ) {

		if ( 'primary' === $args->theme_location ) {
			$classes = 'home-menu';
			if ( is_front_page() ) {
				$classes .= ' current-menu-item';
			}
			$items = '<li class="' . $classes . '"><a href="' . esc_url( home_url( '/' ) ) . '"><span class="screen-reader-text">' . __( 'Home', 'magazine-plus-pro' ) . '</span></a></li>' . $items;
		}

		return $items;

	}
endif;

add_filter( 'wp_nav_menu_items', 'magazine_plus_custom_links_in_navigation', 10, 2 );
