<?php
/**
 * Theme Palace functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Theme Palace
 * @subpackage Travel Insight Pro
 * @since Travel Insight Pro 1.0
 */

if ( ! function_exists( 'travel_insight_pro_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function travel_insight_pro_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Theme Palace, use a find and replace
		 * to change 'travel-insight-pro' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'travel-insight-pro', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		add_theme_support( 'register_block_pattern' ); 

		add_theme_support( 'register_block_style' ); 

		add_theme_support( 'responsive-embeds' );

		// add woocommerce support
		add_theme_support( 'woocommerce' );
		if ( class_exists( 'WooCommerce' ) ) {
			global $woocommerce;

			if( version_compare( $woocommerce->version, '3.0.0', ">=" ) ) {
				add_theme_support( 'wc-product-gallery-zoom' );
				add_theme_support( 'wc-product-gallery-lightbox' );
				add_theme_support( 'wc-product-gallery-slider' );
			}
		}

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );
		add_theme_support( 'post-formats', array( 'gallery' ) );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 500, 500, true );
		add_image_size( 'travel-insight-pro-blog', 145, 345, true );
		add_image_size( 'travel-insight-pro-portrait', 381, 458, true );
		add_image_size( 'travel-insight-pro-package', 271, 548, true );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' 	=> esc_html__( 'Primary', 'travel-insight-pro' ),
			'social' 	=> esc_html__( 'Social', 'travel-insight-pro' ),
			'footer' 	=> esc_html__( 'Footer', 'travel-insight-pro' ),
		) );
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'travel_insight_pro_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// This setup supports logo, site-title & site-description
		add_theme_support( 'custom-logo', array(
			'height'      => 70,
			'width'       => 120,
			'flex-height' => true,
			'flex-width'  => true,
			'header-text' => array( 'site-title', 'site-description' ),
		) );

		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		/*
		 * This theme styles the visual editor to resemble the theme style,
		 * specifically font, colors, icons, and column width.
		 */
		add_editor_style( array( 'assets/css/editor-style' . $min . '.css', travel_insight_pro_fonts_url() ) );

		// Indicate widget sidebars can use selective refresh in the Customizer.
		add_theme_support( 'customize-selective-refresh-widgets' );

		if ( is_admin() ) {
			/**
			 * Load updater file
			 */
			require get_template_directory() . '/updater/theme-updater.php';
		}
		// Gutenberg support
		add_theme_support( 'editor-color-palette', array(
	       	array(
				'name' => esc_html__( 'Blue', 'travel-insight-pro' ),
				'slug' => 'blue',
				'color' => '#20aec8',
	       	),
	       	array(
	           	'name' => esc_html__( 'Black', 'travel-insight-pro' ),
	           	'slug' => 'black',
	           	'color' => '#303c48',
	       	),
	       	array(
	           	'name' => esc_html__( 'Grey', 'travel-insight-pro' ),
	           	'slug' => 'grey',
	           	'color' => '#404040',
	       	),
	   	));

		add_theme_support( 'align-wide' );
		add_theme_support( 'editor-font-sizes', array(
		   	array(
		       	'name' => esc_html__( 'small', 'travel-insight-pro' ),
		       	'shortName' => esc_html__( 'S', 'travel-insight-pro' ),
		       	'size' => 12,
		       	'slug' => 'small'
		   	),
		   	array(
		       	'name' => esc_html__( 'regular', 'travel-insight-pro' ),
		       	'shortName' => esc_html__( 'M', 'travel-insight-pro' ),
		       	'size' => 16,
		       	'slug' => 'regular'
		   	),
		   	array(
		       	'name' => esc_html__( 'larger', 'travel-insight-pro' ),
		       	'shortName' => esc_html__( 'L', 'travel-insight-pro' ),
		       	'size' => 36,
		       	'slug' => 'larger'
		   	),
		   	array(
		       	'name' => esc_html__( 'huge', 'travel-insight-pro' ),
		       	'shortName' => esc_html__( 'XL', 'travel-insight-pro' ),
		       	'size' => 48,
		       	'slug' => 'huge'
		   	)
		));
		add_theme_support('editor-styles');
		add_theme_support( 'wp-block-styles' );
		
	}
endif;
add_action( 'after_setup_theme', 'travel_insight_pro_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function travel_insight_pro_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'travel_insight_pro_content_width', 640 );
}
add_action( 'after_setup_theme', 'travel_insight_pro_content_width', 0 );

if ( ! function_exists( 'travel_insight_pro_fonts_url' ) ) :
	/**
	 * Register Google fonts
	 *
	 * @return string Google fonts URL for the theme.
	 */
	function travel_insight_pro_fonts_url() {
		$fonts_url = '';
		$fonts     = array();
		$subsets   = 'latin,latin-ext';

		// Header Font
		
		/* translators: If there are characters in your language that are not supported by Playfair Display, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Playfair Display font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Playfair Display:400,700,900';
		}

		/* translators: If there are characters in your language that are not supported by Rajdhani, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Rajdhani font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Rajdhani';
		}

		/* translators: If there are characters in your language that are not supported by Cherry Swash, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Cherry Swash font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Cherry Swash';
		}

		/* translators: If there are characters in your language that are not supported by Philosopher, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Philosopher font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Philosopher';
		}

		/* translators: If there are characters in your language that are not supported by Slabo 27px, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Slabo 27px font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Slabo 27px';
		}

		/* translators: If there are characters in your language that are not supported by Dosis, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Dosis font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Dosis';
		}

		// Body Fonts

		/* translators: If there are characters in your language that are not supported by Oxygen, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Oxygen font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Oxygen:300,400,700';
		}

		/* translators: If there are characters in your language that are not supported by |News Cycle, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'News Cycle font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'News Cycle';
		}

		/* translators: If there are characters in your language that are not supported by Pontano Sans, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Pontano Sans font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Pontano Sans';
		}

		/* translators: If there are characters in your language that are not supported by Gudea, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Gudea font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Gudea';
		}

		/* translators: If there are characters in your language that are not supported by Quattrocento, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Quattrocento font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Quattrocento';
		}

		/* translators: If there are characters in your language that are not supported by Khand, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Khand font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Khand:300,400,500,600,700';
		}

		/* translators: If there are characters in your language that are not supported by Poppins, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Poppins font: on or off', 'travel-insight-pro' ) ) {
			$fonts[] = 'Poppins';
		}

		if ( $fonts ) {
			$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
			), 'https://fonts.googleapis.com/css' );
		}

		return $fonts_url;
	}
endif;
/**
 * Enqueue scripts and styles.
 */
function travel_insight_pro_scripts() {
	$options = travel_insight_pro_get_theme_options();
	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'travel-insight-pro-fonts', wptt_get_webfont_url( travel_insight_pro_fonts_url() ), array(), null );

	// magnific-popup
	wp_enqueue_style( 'magnific-popup', get_template_directory_uri() . '/assets/css/magnific-popup' . $min . '.css' );

	// blocks
	wp_enqueue_style( 'travel-insight-pro-blocks', get_template_directory_uri() . '/assets/css/blocks' . $min . '.css' );

	// woocommerce-layout
	wp_enqueue_style( 'woocommerce-layout', get_template_directory_uri() . '/assets/css/woocommerce-layout.css' );

	// woocommerce
	wp_enqueue_style( 'woocommerce', get_template_directory_uri() . '/assets/css/woocommerce.css' );
	
	// FontAwesome
	wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/assets/css/font-awesome.css' );


	// slick
	wp_enqueue_style( 'slick', get_template_directory_uri() . '/assets/css/slick' . $min . '.css' );

	// slick theme
	wp_enqueue_style( 'slick-theme', get_template_directory_uri() . '/assets/css/slick-theme' . $min . '.css' );
	
	wp_enqueue_style( 'travel-insight-pro-style', get_stylesheet_uri() );

	$color = ! empty( $options['theme_color'] ) ? $options['theme_color'] : 'blue';
	// color style
	wp_enqueue_style( 'travel-insight-pro-color', get_template_directory_uri() . '/assets/css/' . esc_attr( $color ) . $min . '.css' );

	// jquery slick
	wp_enqueue_script( 'jquery-slick', get_template_directory_uri() . '/assets/js/slick' . $min . '.js', array( 'jquery' ), '', true );

	// jquery magnific popup
	wp_enqueue_script( 'jquery-magnific-popup', get_template_directory_uri() . '/assets/js/jquery-magnific-popup' . $min . '.js', array( 'jquery' ), '', true );

	// jquery isotope pkgd
	wp_enqueue_script( 'jquery-isotope-pkgd', get_template_directory_uri() . '/assets/js/isotope-pkgd' . $min . '.js', array( 'jquery' ), '', true );

	// jquery packery-mode pkgd
	wp_enqueue_script( 'jquery-packery-mode', get_template_directory_uri() . '/assets/js/packery-mode-pkgd' . $min . '.js', array( 'jquery' ), '', true );

	// jquery packery-mode pkgd
	wp_enqueue_script( 'packery-pkgd', get_template_directory_uri() . '/assets/js/packery.pkgd' . $min . '.js', array( 'jquery' ), '', true );

	// jquery parallax
	wp_enqueue_script( 'jquery-parallax', get_template_directory_uri() . '/assets/js/jquery-parallax' . $min . '.js', array( 'jquery' ), '', true );

	// jquery conter
	wp_enqueue_script( 'jquery-conter', get_template_directory_uri() . '/assets/js/conter' . $min . '.js', array( 'jquery' ), '', true );

	// Load the html5 shiv.
	wp_enqueue_script( 'travel-insight-pro-html5', get_template_directory_uri() . '/assets/js/html5' . $min . '.js', array(), '3.7.3' );

	wp_script_add_data( 'travel-insight-pro-html5', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'travel-insight-pro-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix' . $min . '.js', array(), '20160412', true );

	wp_enqueue_script( 'travel-insight-pro-navigation', get_template_directory_uri() . '/assets/js/navigation' . $min . '.js', array(), '20151215', true );

	wp_localize_script( 'travel-insight-pro-navigation', 'travelInsightProScreenReaderText', array(
        'expand'   => esc_html__( 'expand child menu', 'travel-insight-pro' ),
        'collapse' => esc_html__( 'collapse child menu', 'travel-insight-pro' ),
        'icon'     => travel_insight_pro_get_svg( array( 'icon' => 'angle-down', 'fallback' => true ) )
    ) );

	// Custom Script
	wp_enqueue_script( 'travel-insight-pro-custom', get_template_directory_uri() . '/assets/js/custom' . $min . '.js', array( 'jquery' ), '', true );

	$message = esc_html__( 'OOPS..! No post found with featured image.', 'travel-insight-pro' );

	if ( is_front_page() ) {
		// Custom Frontpage Script
		wp_register_script( 'travel-insight-pro-custom-frontpage', get_template_directory_uri() . '/assets/js/custom-frontpage' . $min . '.js', array( 'jquery' ), '', true );
		wp_localize_script( 'travel-insight-pro-custom-frontpage', 'travel_insight_pro_message', array( 'message' => $message ) );
		wp_enqueue_script( 'travel-insight-pro-custom-frontpage' );
	}

	if ( is_page_template( 'custom-template/gallery-template.php' ) ) {
		// Custom Gallery Script
		wp_register_script( 'travel-insight-pro-custom-gallery', get_template_directory_uri() . '/assets/js/custom-gallery' . $min . '.js', array( 'jquery' ), '', true );
		wp_localize_script( 'travel-insight-pro-custom-gallery', 'travel_insight_pro_message', array( 'message' => $message ) );
		wp_enqueue_script( 'travel-insight-pro-custom-gallery' );
	}

	if ( 'infinite' == $options['pagination_type'] ) {
		// infinite scroll js
		wp_enqueue_script( 'travel-insight-pro-infinite-scroll', get_template_directory_uri() . '/assets/js/infinite-scroll' . $min . '.js', array( 'jquery' ), '', true );

	}

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'travel_insight_pro_scripts' );

/**
 * Enqueue editor styles for Gutenberg
 *
 * @since Travel Insight Pro 1.0.0
 */
function travel_insight_pro_block_editor_styles() {
	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	// Block styles.
	wp_enqueue_style( 'travel-insight-pro-block-editor-style', get_theme_file_uri( '/assets/css/editor-blocks' . $min . '.css' ) );
	// Add custom fonts.
	wp_enqueue_style( 'travel-insight-pro-fonts', wptt_get_webfont_url( travel_insight_pro_fonts_url() ), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'travel_insight_pro_block_editor_styles' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Load core file
 */
require get_template_directory() . '/inc/core.php';

function travel_insight_pro_wptravel_get_template_part( $attr1, $attr2 ){
	if ( function_exists( 'wptravel_get_template_part' ) ) {
		return wptravel_get_template_part( $attr1, $attr2 );
	} elseif ( function_exists( 'wp_travel_get_template_part' ) ) {
		return wp_travel_get_template_part( $attr1, $attr2 );
	}
}

function travel_insight_pro_wptravel_get_formated_price_currency( $attr ){
	if ( function_exists( 'wptravel_get_formated_price_currency' ) ) {
		return wptravel_get_formated_price_currency( $attr );
	} elseif ( function_exists( 'wp_travel_get_formated_price_currency' ) ) {
		return wp_travel_get_formated_price_currency( $attr );
	}
}


function travel_insight_pro_wptravel_get_settings( ){
	if ( function_exists( 'wptravel_get_settings' ) ) {
		return wptravel_get_settings( );
	} elseif ( function_exists( 'wp_travel_get_settings' ) ) {
		return wp_travel_get_settings( );
	}
}

function travel_insight_pro_wptravel_get_currency_symbol( $currency_code ){
	if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
		return wptravel_get_currency_symbol( $currency_code );
	} elseif ( function_exists( 'wp_travel_get_currency_symbol' ) ) {
		return wp_travel_get_currency_symbol( $currency_code );
	}
}

function travel_insight_pro_wptravel_get_trip_duration( $trip_id ){
	if ( function_exists( 'wptravel_get_trip_duration' ) ) {
		return wptravel_get_trip_duration( $trip_id );
	} elseif ( function_exists( 'wp_travel_get_trip_duration' ) ) {
		return wp_travel_get_trip_duration( $trip_id );
	}
}

function travel_insight_pro_wptravel_get_average_rating( $attr = '' ){
	if ( function_exists( 'wptravel_get_average_rating' ) ) {
		return wptravel_get_average_rating( $attr );
	} elseif ( function_exists( 'wp_travel_get_average_rating' ) ) {
		return wp_travel_get_average_rating( $attr );
	}
}