<?php
/**
 * Photo Perfect functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function photo_perfect_setup() {
		/*
		 * Make theme available for translation.
		 */
		load_theme_textdomain( 'photo-perfect-pro', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );
		add_image_size( 'photo-perfect-small-banner', 1600, 350, true ); // Hardcropped image.

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in four location.
		register_nav_menus( array(
			'primary'  => esc_html__( 'Primary Menu', 'photo-perfect-pro' ),
			'footer'   => esc_html__( 'Footer Menu', 'photo-perfect-pro' ),
			'social'   => esc_html__( 'Social Menu', 'photo-perfect-pro' ),
			'notfound' => esc_html__( '404 Menu', 'photo-perfect-pro' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Editor style.
		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		add_editor_style( array( 'css/editor-style' . $min . '.css' ) );

		/*
		 * Enable support for Post Formats.
		 * See https://developer.wordpress.org/themes/functionality/post-formats/
		 */
		add_theme_support( 'post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'photo_perfect_custom_background_args', array(
			'default-color' => 'f9f9f9',
			'default-image' => '',
		) ) );

		/*
		 * Enable support for custom logo.
		 */
		add_theme_support( 'custom-logo' );

		// Load default block styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		/**
		* Enable support for footer widgets
		*/
		add_theme_support( 'footer-widgets', 4 );

		/**
		 * Enable support for WooCommerce
		 */
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-lightbox' );

		// Enable support for selective refresh of widgets in Customizer.
		add_theme_support( 'customize-selective-refresh-widgets' );

		/**
		* Load Supports.
		*/
		require get_template_directory() . '/inc/support.php';

		global $photo_perfect_default_options;
		$photo_perfect_default_options = photo_perfect_get_default_theme_options();

		global $photo_perfect_post_count;
		$photo_perfect_post_count = 1;

	}
endif;

add_action( 'after_setup_theme', 'photo_perfect_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function photo_perfect_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'photo_perfect_content_width', 1220 );
}
add_action( 'after_setup_theme', 'photo_perfect_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function photo_perfect_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'photo-perfect-pro' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'photo-perfect-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar', 'photo-perfect-pro' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar.', 'photo-perfect-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Front Page Widget Area', 'photo-perfect-pro' ),
		'id'            => 'sidebar-front-page-widget-area',
		'description'   => esc_html__( 'Add widgets here to appear in your Front Page Widget Area.', 'photo-perfect-pro' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
	  'name'          => sprintf( __( 'Extra Sidebar %d', 'photo-perfect-pro' ), 1 ),
	  'id'            => 'extra-sidebar-1',
	  'before_widget' => '<aside id="%1$s" class="widget %2$s">',
	  'after_widget'  => '</aside>',
	  'before_title'  => '<h2 class="widget-title">',
	  'after_title'   => '</h2>',
	) );
	register_sidebar( array(
	  'name'          => sprintf( __( 'Extra Sidebar %d', 'photo-perfect-pro' ), 2 ),
	  'id'            => 'extra-sidebar-2',
	  'before_widget' => '<aside id="%1$s" class="widget %2$s">',
	  'after_widget'  => '</aside>',
	  'before_title'  => '<h2 class="widget-title">',
	  'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'photo-perfect-pro' ), 3 ),
	  'id'            => 'extra-sidebar-3',
	  'before_widget' => '<aside id="%1$s" class="widget %2$s">',
	  'after_widget'  => '</aside>',
	  'before_title'  => '<h2 class="widget-title">',
	  'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => sprintf( __( 'Extra Sidebar %d', 'photo-perfect-pro' ), 4 ),
	  'id'            => 'extra-sidebar-4',
	  'before_widget' => '<aside id="%1$s" class="widget %2$s">',
	  'after_widget'  => '</aside>',
	  'before_title'  => '<h2 class="widget-title">',
	  'after_title'   => '</h2>',
	) );

}
add_action( 'widgets_init', 'photo_perfect_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function photo_perfect_scripts() {

	$theme_version = wp_get_theme()->get( 'Version' );

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	$fonts_url = photo_perfect_fonts_url();

	if ( ! empty( $fonts_url ) ) {
		wp_enqueue_style( 'photo-perfect-google-fonts', photo_perfect_fonts_url(), array(), null );
	}

	wp_enqueue_style( 'photo-perfect-style', get_stylesheet_uri(), array(), $theme_version );

	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '4.7.0' );

	wp_enqueue_style( 'photo-perfect-photobox-css', get_template_directory_uri() . '/third-party/photobox/photobox' . $min . '.css', array(), '1.6.3' );

	wp_enqueue_script( 'photo-perfect-cycle2-script', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2' . $min . '.js', array( 'jquery' ), '2.1.6', true );

	wp_enqueue_script( 'photo-perfect-navigation', get_template_directory_uri() . '/js/navigation' . $min . '.js', array(), '20120206', true );

	wp_enqueue_script( 'photo-perfect-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix' . $min . '.js', array(), '20130115', true );

	wp_enqueue_script( 'photo-perfect-imageloaded', get_template_directory_uri() . '/third-party/imageloaded/imagesloaded.pkgd' . $min . '.js', array( 'jquery' ), '1.0.0', true );

	wp_enqueue_script( 'photo-perfect-photobox', get_template_directory_uri() . '/third-party/photobox/jquery.photobox' . $min . '.js', array( 'jquery' ), '1.6.3', true );

	wp_enqueue_script( 'photo-perfect-custom', get_template_directory_uri() . '/js/custom' . $min . '.js', array( 'jquery', 'jquery-masonry', 'photo-perfect-imageloaded', 'photo-perfect-photobox' ), '2.3.0', true );
	wp_localize_script( 'photo-perfect-custom', 'screenReaderText', array(
		'expand'   => '<span class="screen-reader-text">' . __( 'expand child menu', 'photo-perfect-pro' ) . '</span>',
		'collapse' => '<span class="screen-reader-text">' . __( 'collapse child menu', 'photo-perfect-pro' ) . '</span>',
	) );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'photo_perfect_scripts' );

if ( ! function_exists( 'photo_perfect_blocks_support' ) ) :
	/**
	 * Create add default blocks support
	 */
	function photo_perfect_blocks_support() {
		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );

		// Add support for editor styles.
		add_theme_support( 'editor-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__( 'Small', 'photo-perfect-pro' ),
					'shortName' => esc_html__( 'S', 'photo-perfect-pro' ),
					'size'      => 14,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__( 'Normal', 'photo-perfect-pro' ),
					'shortName' => esc_html__( 'M', 'photo-perfect-pro' ),
					'size'      => 18,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__( 'Large', 'photo-perfect-pro' ),
					'shortName' => esc_html__( 'L', 'photo-perfect-pro' ),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__( 'Huge', 'photo-perfect-pro' ),
					'shortName' => esc_html__( 'XL', 'photo-perfect-pro' ),
					'size'      => 54,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support( 'editor-color-palette', array(
			array(
				'name'  => esc_html__( 'White', 'photo-perfect-pro' ),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__( 'Black', 'photo-perfect-pro' ),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__( 'Gray', 'photo-perfect-pro' ),
				'slug'  => 'gray',
				'color' => '#f7f7f7',
			),
			array(
				'name'  => esc_html__( 'Blue', 'photo-perfect-pro' ),
				'slug'  => 'blue',
				'color' => '#1b8be0',
			),
			array(
				'name'  => esc_html__( 'Dark Blue', 'photo-perfect-pro' ),
				'slug'  => 'dark-blue',
				'color' => '#191970',
			),
			array(
				'name'  => esc_html__( 'Orange', 'photo-perfect-pro' ),
				'slug'  => 'orange',
				'color' => '#ffc300',
			),
		) );
	}
	add_action( 'after_setup_theme', 'photo_perfect_blocks_support', 20 );
endif; //photo_perfect_blocks_support

if ( ! function_exists( 'photo_perfect_add_blocks_style' ) ) :
	/**
	 * Add Blocks Style
	 */
	function photo_perfect_add_blocks_style() {
		// Theme block stylesheet.
		wp_enqueue_style( 'photo-perfect-block-style', get_theme_file_uri( '/css/blocks.css' ), array( 'photo-perfect-style' ), date( 'Ymd-Gis', filemtime( get_template_directory() . '/css/blocks.css' ) ) );
	}
	add_action( 'wp_enqueue_scripts', 'photo_perfect_add_blocks_style' );
endif; //photo_perfect_add_blocks_style

if ( ! function_exists( 'photo_perfect_block_editor_styles' ) ) :
	/**
	 * Enqueue editor styles for Blocks
	 */
	function photo_perfect_block_editor_styles() {
		// Block styles.
		wp_enqueue_style( 'photo-perfect-block-editor-style', get_theme_file_uri( '/css/editor-blocks.css' ), null, date( 'Ymd-Gis', filemtime( get_template_directory() . '/css/editor-blocks.css' ) ) );

		// Add custom fonts.
		wp_enqueue_style( 'photo-perfect-fonts', photo_perfect_fonts_url(), array(), null );
	}
	add_action( 'enqueue_block_editor_assets', 'photo_perfect_block_editor_styles' );
endif; //photo_perfect_block_editor_styles

/**
 * Load init.
 */
require get_template_directory() . '/inc/init.php';

/**
 * Load theme updater functions.
 */
function photo_perfect_theme_updater() {
	if ( is_admin() ) {
		require_once get_template_directory() . '/updater/theme-updater.php';
	}
}
add_action( 'after_setup_theme', 'photo_perfect_theme_updater' );
