<?php
/**
 * Common helper functions.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_the_excerpt' ) ) :

	/**
	 * Generate excerpt.
	 *
	 * @since 1.0.0
	 *
	 * @param int     $length Excerpt length in words (Default: 40).
	 * @param WP_Post $post_obj WP_Post instance (Optional).
	 * @return string Excerpt.
	 */
	function photo_perfect_the_excerpt( $length = 40, $post_obj = null ) {

		global $post;
		if ( is_null( $post_obj ) ) {
			$post_obj = $post;
		}
		$length = absint( $length );
		if ( $length < 1 ) {
			$length = 40;
		}
		$source_content = $post_obj->post_content;
		if ( ! empty( $post_obj->post_excerpt ) ) {
			$source_content = $post_obj->post_excerpt;
		}
		$source_content = preg_replace( '`\[[^\]]*\]`', '', $source_content );
		$trimmed_content = wp_trim_words( $source_content, $length, '...' );
		return $trimmed_content;

	}

endif;

if ( ! function_exists( 'photo_perfect_apply_theme_shortcode' ) ) :

	/**
	 * Apply theme shortcode.
	 *
	 * @since 1.0.0
	 *
	 * @param string $string Content.
	 * @return string Modified content.
	 */
	function photo_perfect_apply_theme_shortcode( $string ) {

		if ( empty( $string ) ) {
			return $string;
		}

		$search = array( '[the-year]', '[the-site-link]' );

		$replace = array(
			date( 'Y' ),
			'<a href="'. esc_url( home_url( '/' ) ) .'">'. esc_html( get_bloginfo( 'name', 'display' ) ) . '</a>',
		);
		$string = str_replace( $search, $replace, $string );

		return $string;

	}

endif;

if ( ! function_exists( 'photo_perfect_render_category_menu_listing' ) ) :
	/**
	 * Render category menu listing.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_render_category_menu_listing(){

		$header_category_text    = photo_perfect_get_option( 'header_category_text' );
		$header_category_number  = photo_perfect_get_option( 'header_category_number' );
		$header_category_depth   = photo_perfect_get_option( 'header_category_depth' );
		$header_category_exclude = photo_perfect_get_option( 'header_category_exclude' );
		$cat_args = array(
			'title_li' => '',
			'depth'    => esc_attr( $header_category_depth ),
			'number'   => absint( $header_category_number ),
			'exclude'  => esc_attr( $header_category_exclude ),
		);
		?>
    <button class="nav-list-btn"><i class="fa fa-list"></i><span><?php echo esc_html( $header_category_text ); ?></span></button>
    <div class="category-list-wrapper">
      <ul>
				<?php wp_list_categories( $cat_args ); ?>
      </ul>
    </div><!-- .category-list-wrapper -->
		<?php


	}
endif;

if ( ! function_exists( 'photo_perfect_render_custom_banner' ) ) :

	/**
	 * Render custom banner.
	 *
	 * @since 1.0.0
	 *
	 * @param array $detail Custom banner detail.
	 */
	function photo_perfect_render_custom_banner( $detail ) {

		if ( empty( $detail ) ) {
			return;
		}
		$image_attr = array();
		if ( isset( $detail['image_url'] ) && ! empty( $detail['image_url'] ) ) {
			$image_attr['src'] = esc_url( $detail['image_url'] );
		}
		if ( isset( $detail['image_width'] ) && ! empty( $detail['image_width'] ) ) {
			$image_attr['width'] = absint( $detail['image_width'] );
		}
		if ( isset( $detail['image_height'] ) && ! empty( $detail['image_height'] ) ) {
			$image_attr['height'] = absint( $detail['image_height'] );
		}
		if ( isset( $detail['image_alt'] ) && ! empty( $detail['image_alt'] ) ) {
			$image_attr['alt'] = esc_attr( $detail['image_alt'] );
		}
		$image_attr_text = '';
		if ( ! empty( $image_attr ) ) {
			foreach ( $image_attr as $key => $attr ) {
				$image_attr_text .= ' ' . $key . '="' . $attr .'"';
			}
		}

		$link_open = '';
		$link_close = '';
		if ( isset( $detail['link_url'] ) && ! empty( $detail['link_url'] ) ) {
			$target_text = '';
			if ( isset( $detail['link_url_open'] ) && true == $detail['link_url_open'] ) {
				$target_text = ' target="_blank" ';
			}
			$link_open = '<a href="' . esc_url( $detail['link_url'] ) . '" ' . $target_text . '>';
			$link_close = '</a>';
		}
		$image_tag = sprintf( '<img %s />', $image_attr_text );
		echo '<div id="featured-banner" class="featured-banner">';
		printf('%s%s%s',
			$link_open,
			$image_tag,
			$link_close
		);
		echo '</div>';

	}

endif;

if ( ! function_exists( 'photo_perfect_fonts_url' ) ) :

	/**
	 * Return fonts URL.
	 *
	 * @since 1.0.0
	 * @return string Font URL.
	 */
	function photo_perfect_fonts_url() {

		$fonts_url = '';
		$fonts     = array();
		$subsets   = 'latin,latin-ext';

		$font_settings = photo_perfect_get_font_family_theme_settings_options();
		$web_fonts = photo_perfect_get_web_fonts();

		$theme_options = array();
		if ( ! empty( $font_settings ) ) {
			foreach ( $font_settings as $k => $v ) {
				$theme_options[] = photo_perfect_get_option( $k );
			}
		}
		$theme_options = array_unique( $theme_options );

		$required_fonts = array();

		if ( ! empty( $theme_options ) ) {
			foreach ( $theme_options as $key => $value ) {
				if ( array_key_exists( $value, $web_fonts ) ) {
					$required_fonts[] = $value;
				}
			}
		}

		if ( ! empty( $required_fonts ) ) {
			foreach ( $required_fonts as $key => $f ) {
				$fonts[] = $web_fonts[ $f ]['name'] . ':300,300i,400,400i,600,600i,700,700i,800,800i';
			}
		}

		if ( $fonts ) {
			$fonts_url = add_query_arg( array(
				'family' => urlencode( implode( '|', $fonts ) ),
				'subset' => urlencode( $subsets ),
			), 'https://fonts.googleapis.com/css' );
		}

		$font_load = photo_perfect_get_option( 'load_font_cdn' );

		if ( ! $font_load ) {
			require_once get_theme_file_path( 'inc/wptt-webfont-loader.php' );
		
			$fonts_url = wptt_get_webfont_url( $fonts_url );
		}

		return $fonts_url;

	}

endif;

if( ! function_exists( 'photo_perfect_get_sidebar_options' ) ) :

  /**
   * Get sidebar options.
   *
   * @since 1.0.0
   * @return array Sidebar options.
   */
  function photo_perfect_get_sidebar_options(){

    global $wp_registered_sidebars;

    $output = array();

    if ( ! empty( $wp_registered_sidebars ) && is_array( $wp_registered_sidebars ) ) {
      foreach ( $wp_registered_sidebars as $key => $sidebar ) {
        $output[$key] = $sidebar['name'];
      }
    }

    return $output;

  }

endif;


if ( ! function_exists( 'photo_perfect_hex2rgba' ) ) :

	/**
	 * Convert HEX color to rgba.
	 *
	 * @since 1.0.0
	 *
	 * @param  string  $color   Color value in HEX
	 * @param  boolean $opacity Whether opacity is required
	 * @return string           Color value in rgba
	 */
	function photo_perfect_hex2rgba( $color, $opacity = false ) {

	 $default = 'rgb(0,0,0)';

	 // Return default if no color provided.
	 if(empty($color))
     return $default;

	  // Sanitize $color if "#" is provided.
    if ($color[0] == '#' ) {
     $color = substr( $color, 1 );
    }

    // Check if color has 6 or 3 characters and get values.
    if (strlen($color) == 6) {
      $hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
    } elseif ( strlen( $color ) == 3 ) {
			$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
    } else {
			return $default;
    }

    // Convert hexadec to rgb.
    $rgb =  array_map( 'hexdec', $hex );

    // Check if opacity is set(rgba or rgb).
    if( $opacity ){
     if( abs( $opacity ) > 1 )
     $opacity = 1.0;
     $output = 'rgba('.implode(",",$rgb).','.$opacity.')';
    } else {
     $output = 'rgb('.implode(",",$rgb).')';
    }

    // Return rgb(a) color string.
    return $output;
	}

endif;

if ( ! function_exists( 'photo_perfect_simple_breadcrumb' ) ) :

	/**
	 * Simple breadcrumb.
	 *
	 * @since 1.0.0
	 *
	 * @link: https://gist.github.com/melissacabral/4032941
	 *
	 * @param array $args Arguments.
	 */
	function photo_perfect_simple_breadcrumb( $args = array() ) {

		$args = wp_parse_args( (array) $args, array(
			'separator' => '',
		) );

		/* === OPTIONS === */
		$text['home']     = get_bloginfo( 'name', 'display' ); // text for the 'Home' link
		$text['category'] = __( 'Archive for <em>%s</em>', 'photo-perfect-pro' ); // text for a category page
		$text['tax']      = __( 'Archive for <em>%s</em>', 'photo-perfect-pro' ); // text for a taxonomy page
		$text['search']   = __( 'Search results for: <em>%s</em>', 'photo-perfect-pro' ); // text for a search results page
		$text['tag']      = __( 'Posts tagged <em>%s</em>', 'photo-perfect-pro' ); // text for a tag page
		$text['author']   = __( 'View all posts by <em>%s</em>', 'photo-perfect-pro' ); // text for an author page
		$text['404']      = __( 'Error 404', 'photo-perfect-pro' ); // text for the 404 page

		$showCurrent = 1; // 1 - show current post/page title in breadcrumbs, 0 - don't show
		$showOnHome  = 0; // 1 - show breadcrumbs on the homepage, 0 - don't show
		$delimiter   = ' ' . $args['separator'] . ' '; // delimiter between crumbs
		$before      = '<span class="current">'; // tag before the current crumb
		$after       = '</span>'; // tag after the current crumb
		/* === END OF OPTIONS === */

		global $post;
		$homeLink   = esc_url( home_url( '/' ) );
		$linkBefore = '<span typeof="v:Breadcrumb">';
		$linkAfter  = '</span>';
		$linkAttr   = ' rel="v:url" property="v:title"';
		$link       = $linkBefore . '<a' . $linkAttr . ' href="%1$s">%2$s</a>' . $linkAfter;

		if ( is_home() || is_front_page() ) {

			if ( $showOnHome == 1 ) { echo '<div id="crumbs"><a href="' . $homeLink . '">' . $text['home'] . '</a></div>'; }
		} else {

			echo '<div id="crumbs" xmlns:v="http://rdf.data-vocabulary.org/#">' . sprintf( $link, $homeLink, $text['home'] ) . $delimiter;

			if ( is_category() ) {
				$thisCat = get_category( get_query_var( 'cat' ), false );
				if ( $thisCat->parent != 0 ) {
					$cats = get_category_parents( $thisCat->parent, true, $delimiter );
					$cats = str_replace( '<a', $linkBefore . '<a' . $linkAttr, $cats );
					$cats = str_replace( '</a>', '</a>' . $linkAfter, $cats );
					echo $cats;
				}
				echo $before . sprintf( $text['category'], single_cat_title( '', false ) ) . $after;

			} elseif ( is_tax() ) {
				$thisCat = get_category( get_query_var( 'cat' ), false );
				if ( $thisCat->parent != 0 ) {
					$cats = get_category_parents( $thisCat->parent, true, $delimiter );
					$cats = str_replace( '<a', $linkBefore . '<a' . $linkAttr, $cats );
					$cats = str_replace( '</a>', '</a>' . $linkAfter, $cats );
					echo $cats;
				}
				echo $before . sprintf( $text['tax'], single_cat_title( '', false ) ) . $after;

			} elseif ( is_search() ) {
				echo $before . sprintf( $text['search'], get_search_query() ) . $after;

			} elseif ( is_day() ) {
				echo sprintf( $link, get_year_link( get_the_time( 'Y' ) ), get_the_time( 'Y' ) ) . $delimiter;
				echo sprintf( $link, get_month_link( get_the_time( 'Y' ),get_the_time( 'm' ) ), get_the_time( 'F' ) ) . $delimiter;
				echo $before . get_the_time( 'd' ) . $after;

			} elseif ( is_month() ) {
				echo sprintf( $link, get_year_link( get_the_time( 'Y' ) ), get_the_time( 'Y' ) ) . $delimiter;
				echo $before . get_the_time( 'F' ) . $after;

			} elseif ( is_year() ) {
				echo $before . get_the_time( 'Y' ) . $after;

			} elseif ( is_single() && ! is_attachment() ) {
				if ( get_post_type() != 'post' ) {
					$post_type = get_post_type_object( get_post_type() );
					$slug = $post_type->rewrite;
					printf( $link, $homeLink . '/' . $slug['slug'] . '/', $post_type->labels->singular_name );
					if ( $showCurrent == 1 ) { echo $delimiter . $before . get_the_title() . $after; }
				} else {
					$cat = get_the_category();
					$cat = $cat[0];
					$cats = get_category_parents( $cat, true, $delimiter );
					if ( $showCurrent == 0 ) { $cats = preg_replace( "#^(.+)$delimiter$#", '$1', $cats ); }
					$cats = str_replace( '<a', $linkBefore . '<a' . $linkAttr, $cats );
					$cats = str_replace( '</a>', '</a>' . $linkAfter, $cats );
					echo $cats;
					if ( $showCurrent == 1 ) { echo $before . get_the_title() . $after; }
				}
			} elseif ( ! is_single() && ! is_page() && get_post_type() != 'post' && ! is_404() ) {
				$post_type = get_post_type_object( get_post_type() );
				echo $before . $post_type->labels->singular_name . $after;

			} elseif ( is_attachment() ) {
				$parent = get_post( $post->post_parent );
				$cat = get_the_category( $parent->ID );
				$cat = isset( $cat[0] ) ? $cat[0] : '';
				$cats = get_category_parents( $cat, true, $delimiter );
				$cats = str_replace( '<a', $linkBefore . '<a' . $linkAttr, $cats );
				$cats = str_replace( '</a>', '</a>' . $linkAfter, $cats );
				echo $cats;
				printf( $link, get_permalink( $parent ), $parent->post_title );
				if ( $showCurrent == 1 ) { echo $delimiter . $before . get_the_title() . $after; }
			} elseif ( is_page() && ! $post->post_parent ) {
				if ( $showCurrent == 1 ) { echo $before . get_the_title() . $after; }
			} elseif ( is_page() && $post->post_parent ) {
				$parent_id  = $post->post_parent;
				$breadcrumbs = array();
				while ( $parent_id ) {
					$page = get_page( $parent_id );
					$breadcrumbs[] = sprintf( $link, get_permalink( $page->ID ), get_the_title( $page->ID ) );
					$parent_id  = $page->post_parent;
				}
				$breadcrumbs = array_reverse( $breadcrumbs );
				for ( $i = 0; $i < count( $breadcrumbs ); $i++ ) {
					echo $breadcrumbs[ $i ];
					if ( $i != count( $breadcrumbs ) -1 ) { echo $delimiter; }
				}
				if ( $showCurrent == 1 ) { echo $delimiter . $before . get_the_title() . $after; }
			} elseif ( is_tag() ) {
				echo $before . sprintf( $text['tag'], single_tag_title( '', false ) ) . $after;

			} elseif ( is_author() ) {
				global $author;
				$userdata = get_userdata( $author );
				echo $before . sprintf( $text['author'], $userdata->display_name ) . $after;

			} elseif ( is_404() ) {
				echo $before . $text['404'] . $after;
			}

			if ( get_query_var( 'paged' ) ) {
				if ( is_category() || is_day() || is_month() || is_year() || is_search() || is_tag() || is_author() ) { echo ' ('; }
				echo __( 'Page', 'photo-perfect-pro' ) . ' ' . get_query_var( 'paged' );
				if ( is_category() || is_day() || is_month() || is_year() || is_search() || is_tag() || is_author() ) { echo ')'; }
			}

					echo '</div>';
		}

	}

endif;
