<?php
/**
 * Basic theme functions.
 *
 * This file contains hook functions attached to core hooks.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_customize_search_form' ) ) :
	/**
	 * Customize search form.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_customize_search_form() {

		$search_placeholder = photo_perfect_get_option( 'search_placeholder' );
		$form = '<form role="search" method="get" class="search-form" action="' . esc_url( home_url( '/' ) ) . '">
	      <label>
	        <span class="screen-reader-text">' . _x( 'Search for:', 'label', 'photo-perfect-pro' ) . '</span>
	        <input type="search" class="search-field" placeholder="' . esc_attr( $search_placeholder ) . '" value="' . get_search_query() . '" name="s" title="' . esc_attr_x( 'Search for:', 'label', 'photo-perfect-pro' ) . '" />
	      </label>
	      <input type="submit" class="search-submit" value="' . esc_attr_x( 'Search', 'submit button', 'photo-perfect-pro' ) . '" />
	    </form>';

		return $form;

	}

endif;

add_filter( 'get_search_form', 'photo_perfect_customize_search_form', 15 );


if ( ! function_exists( 'photo_perfect_add_custom_css' ) ) :

	/**
	 * Add custom CSS.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_custom_css() {

		$custom_css = photo_perfect_get_option( 'custom_css' );
		$output = '';
		if ( ! empty( $custom_css ) ) {
			echo "\n" . '<style type="text/css">' . "\n";
			echo esc_textarea( $custom_css );
			echo "\n" . '</style>' . "\n" ;
		}

	}

endif;

add_action( 'wp_head', 'photo_perfect_add_custom_css' );


if ( ! function_exists( 'photo_perfect_implement_excerpt_length' ) ) :

	/**
	 * Implement excerpt length.
	 *
	 * @since 1.0.0
	 *
	 * @param int $length The number of words.
	 * @return int Excerpt length.
	 */
	function photo_perfect_implement_excerpt_length( $length ) {

		$excerpt_length = photo_perfect_get_option( 'excerpt_length' );
		if ( empty( $excerpt_length ) ) {
			$excerpt_length = $length;
		}
		return apply_filters( 'photo_perfect_filter_excerpt_length', esc_attr( $excerpt_length ) );

	}

endif;
add_filter( 'excerpt_length', 'photo_perfect_implement_excerpt_length', 999 );


if ( ! function_exists( 'photo_perfect_implement_read_more' ) ) :

	/**
	 * Implement read more in excerpt.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more The string shown within the more link.
	 * @return string The excerpt.
	 */
	function photo_perfect_implement_read_more( $more ) {

		$flag_apply_excerpt_read_more = apply_filters( 'photo_perfect_filter_excerpt_read_more', true );
		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more;
		}

		$output = $more;
		$read_more_text = photo_perfect_get_option( 'read_more_text' );
		if ( ! empty( $read_more_text ) ) {
			$output = '... <a href="'. esc_url( get_permalink() ) . '" class="read-more">' . sprintf( '%s' , esc_html( $read_more_text ) ) . '</a>';
			$output = apply_filters( 'photo_perfect_filter_read_more_link' , $output );
		}
		return $output;

	}

endif;
add_filter( 'excerpt_more', 'photo_perfect_implement_read_more' );


if ( ! function_exists( 'photo_perfect_content_more_link' ) ) :

	/**
	 * Implement read more in content.
	 *
	 * @since 1.0.0
	 *
	 * @param string $more_link Read More link element.
	 * @param string $more_link_text Read More text.
	 * @return string Link.
	 */
	function photo_perfect_content_more_link( $more_link, $more_link_text ) {

		$flag_apply_excerpt_read_more = apply_filters( 'photo_perfect_filter_excerpt_read_more', true );
		if ( true !== $flag_apply_excerpt_read_more ) {
			return $more_link;
		}

		$read_more_text = photo_perfect_get_option( 'read_more_text' );
		if ( ! empty( $read_more_text ) ) {
			$more_link = str_replace( $more_link_text, esc_html( $read_more_text ), $more_link );
		}
		return $more_link;

	}

endif;

add_filter( 'the_content_more_link', 'photo_perfect_content_more_link', 10, 2 );


if ( ! function_exists( 'photo_perfect_custom_body_class' ) ) :
	/**
	 * Custom body class.
	 *
	 * @since 1.0.0
	 *
	 * @param string|array $input One or more classes to add to the class list.
	 * @return array Array of classes.
	 */
	function photo_perfect_custom_body_class( $input ) {

		global $post;

		// Site layout.
		$site_layout = photo_perfect_get_option( 'site_layout' );
		$input[] = 'site-layout-' . esc_attr( $site_layout );

		// Archive layout.
		$archive_layout = photo_perfect_get_option( 'archive_layout' );
		$input[] = 'archive-layout-' . esc_attr( $archive_layout );

		// Custom banner status.
		$custom_banner_detail = apply_filters( 'photo_perfect_filter_custom_banner_details', array() );
		$custom_banner_status = apply_filters( 'photo_perfect_filter_custom_banner_status', false );
		$flag_apply_slider = apply_filters( 'photo_perfect_filter_slider_status', true );
		$flag_banner_status = 'disabled';
		if ( true === $flag_apply_slider || true === $custom_banner_status ) {
			$flag_banner_status = 'enabled';
		}
		// $banner_status = ( ( false === $flag_apply_slider ) || empty( $custom_banner_detail ) || false === $custom_banner_status ) ? 'disabled' : 'enabled' ;
		$input[] = 'custom-header-' . $flag_banner_status;

		// Sticky menu in header.
		$header_sticky_menu_enabled = photo_perfect_get_option( 'header_sticky_menu_enabled' );
		if ( true === $header_sticky_menu_enabled ) {
			$input[] = 'sticky-menu-enabled';
		}

		// Shrink class.
		$enable_shrunk_header = photo_perfect_get_option( 'enable_shrunk_header' );
		$shrink_class = '';
		if ( ! is_front_page() ) {
			if ( true === $enable_shrunk_header ) {
				$shrink_class = 'shrunk-header';
			}
			// Check if single.
			if ( is_singular( array( 'post', 'page' ) ) ) {
				$post_options = get_post_meta( $post->ID, 'theme_settings', true );
				if ( isset( $post_options['shrunk_header'] ) && ! empty( $post_options['shrunk_header'] ) ) {
					switch ( $post_options['shrunk_header'] ) {
						case 'disable':
							$shrink_class = '';
							break;
						case 'enable':
							$shrink_class = 'shrunk-header';
							break;
						case 'default':
						default:
							break;
					}
				}
			}
		}
		if ( ! empty( $shrink_class ) ) {
			$input[] = $shrink_class;
		}

		// Disable zoom effect.
		$disable_zoom_effect = photo_perfect_get_option( 'disable_zoom_effect' );
		if ( true === $disable_zoom_effect ) {
			$input[] = 'image-zoom-effect-disabled';
		}

		// Global layout.
		$global_layout = photo_perfect_get_option( 'global_layout' );
		$global_layout = apply_filters( 'photo_perfect_filter_theme_global_layout', $global_layout );
		// Check if single.
		if ( $post  && is_singular() ) {
			$post_options = get_post_meta( $post->ID, 'theme_settings', true );
			if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
				$global_layout = $post_options['post_layout'];
			}
		}

		$input[] = 'global-layout-' . esc_attr( $global_layout );

		// Adds a class of group-blog to blogs with more than 1 published author.
		if ( is_multi_author() ) {
			$input[] = 'group-blog';
		}

		return $input;
	}
endif;

add_filter( 'body_class', 'photo_perfect_custom_body_class' );

/**
 * Load theme options from free theme.
 *
 * Checks if there are options already present from free version and adds it to the Pro theme options.
 *
 * @since 1.0.0
 */
function photo_perfect_import_free_options() {

	// Perform action only if theme_mods_XXX[theme_options] does not exist.
	if( ! get_theme_mod( 'theme_options' ) ) {

		// Perform action only if theme_mods_XXX free version exists.
		if ( $free_options = get_option ( 'theme_mods_photo-perfect' ) ) {
			if ( isset( $free_options['theme_options'] ) ) {
				$pro_default_options = photo_perfect_get_default_theme_options();

				$pro_theme_options = $free_options;

				$pro_theme_options['theme_options'] = array_merge( $pro_default_options , $free_options['theme_options'] );

				// WP default fields.
				$fields = array(
					'custom_logo',
					'header_image',
					'header_image_data',
					'background_image',
					'background_repeat',
					'background_position_x',
					'background_attachment',
				);

				foreach ( $fields as $key ) {
					if ( isset( $free_options[ $key ] ) && ! empty( $free_options[ $key ] ) ) {
						$pro_theme_options[ $key ] = $free_options[ $key ];
					}
				}

				update_option( 'theme_mods_photo-perfect-pro', $pro_theme_options );
			}
		}
	}
}

add_action( 'after_switch_theme', 'photo_perfect_import_free_options' );

if ( ! function_exists( 'photo_perfect_import_custom_css' ) ) :

	/**
	 * Import Custom CSS.
	 *
	 * @since 2.3.0
	 */
	function photo_perfect_import_custom_css() {

		// Bail if not WP 4.7.
		if ( ! function_exists( 'wp_get_custom_css_post' ) ) {
			return;
		}

		$custom_css = photo_perfect_get_option( 'custom_css' );

		// Bail if there is no Custom CSS.
		if ( empty( $custom_css ) ) {
			return;
		}

		$core_css = wp_get_custom_css();
		$return = wp_update_custom_css_post( $core_css . $custom_css );

		if ( ! is_wp_error( $return ) ) {

			// Remove from theme.
			$options = photo_perfect_get_options();
			$options['custom_css'] = '';
			set_theme_mod( 'theme_options', $options );
		}

	}
endif;

add_action( 'after_setup_theme', 'photo_perfect_import_custom_css', 99 );

if ( ! function_exists( 'photo_perfect_customizer_reset_callback' ) ) :

	/**
	 * Callback for reset in Customizer.
	 *
	 * @since 2.3.0
	 */
	function photo_perfect_customizer_reset_callback() {

		$reset_all_settings = photo_perfect_get_option( 'reset_all_settings' );

		if ( true === $reset_all_settings ) {

			// Reset custom theme options.
			set_theme_mod( 'theme_options', array() );

			// Reset custom header, logo and backgrounds.
			remove_theme_mod( 'custom_logo' );
			remove_theme_mod( 'header_image' );
			remove_theme_mod( 'header_image_data' );
			remove_theme_mod( 'background_image' );
			remove_theme_mod( 'background_color' );
		}

		// Reset color options.
		$reset_color_settings = photo_perfect_get_option( 'reset_color_settings' );
		if ( true === $reset_color_settings ) {
			$options = photo_perfect_get_options();
			$options['reset_color_settings'] = false;
			$color_fields = photo_perfect_get_color_theme_settings_options();
			$default = photo_perfect_get_default_theme_options();
			if ( ! empty( $color_fields ) ) {
				foreach ( $color_fields as $key => $val ) {
					$options[ $key ] = $default[ $key ];
				}
			}
			set_theme_mod( 'theme_options', $options );
			remove_theme_mod( 'background_color' );
		}

		// Reset font options.
		$reset_font_settings = photo_perfect_get_option( 'reset_font_settings' );
		if ( true === $reset_font_settings ) {
			$options = photo_perfect_get_options();
			$options['reset_font_settings'] = false;
			$font_settings = photo_perfect_get_font_family_theme_settings_options();
			if ( ! empty( $font_settings ) ) {
				foreach ( $font_settings as $key => $val ) {
					$options[ $key ] = $val['default'];
				}
			}
			set_theme_mod( 'theme_options', $options );
		}

	}
endif;

add_action( 'customize_save_after', 'photo_perfect_customizer_reset_callback' );
