<?php
/**
 * Custom theme functions.
 *
 * This file contains hook functions attached to theme hooks.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_skip_to_content' ) ) :
	/**
	 * Skip to content
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_skip_to_content() {
	?><a class="skip-link screen-reader-text" href="#content"><?php esc_html_e( 'Skip to content', 'photo-perfect-pro' ); ?></a><?php
	}
endif;

if ( ! function_exists( 'photo_perfect_custom_content_width' ) ) :

	/**
	 * Custom content width.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_custom_content_width() {

		global $post, $wp_query, $content_width;

		$global_layout = photo_perfect_get_option( 'global_layout' );
		$global_layout = apply_filters( 'photo_perfect_filter_theme_global_layout', $global_layout );

	  // Check if single.
		if ( $post  && is_singular() ) {
		  $post_options = get_post_meta( $post->ID, 'theme_settings', true );
		  if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
		    $global_layout = esc_attr( $post_options['post_layout'] );
		  }
		}
		switch ( $global_layout ) {

		  case 'no-sidebar':
		    $content_width = 1220;
		    break;

		  case 'three-columns':
		  case 'three-columns-pcs':
		  case 'three-columns-cps':
		  case 'three-columns-psc':
		  case 'no-sidebar-centered':
		    $content_width = 570;
		    break;

		  case 'three-columns-pcs-equal':
		  case 'three-columns-scp-equal':
		    $content_width = 353;
		    break;

		  case 'left-sidebar':
		  case 'right-sidebar':
		    $content_width = 895;
		    break;

		  default:
		    break;
		}

	}
endif;

add_filter( 'template_redirect', 'photo_perfect_custom_content_width' );

if ( ! function_exists( 'photo_perfect_add_separator_title' ) ) :

	/**
	 * Add separator heading.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_separator_title() {

		ob_start();
	?>
    <div id="separator-title">
		<?php
		// nsdump(is_front_page());
		if ( is_front_page() || is_home() ) {
			?>
			<h1 class="entry-title"><?php single_post_title(); ?></h1>
			<?php
		}
		elseif ( is_search() ) {
			?>
	          <h1 class="page-title"><?php printf( esc_html__( 'Search Results for: %s', 'photo-perfect-pro' ), '<span>' . get_search_query() . '</span>' ); ?></h1>
			<?php
		} elseif ( is_singular() ) {
			global $post;
			?>
			<?php if ( 'post' === get_post_type() ) :  ?>
            <div class="single-link-previous">
				<?php echo get_previous_post_link( '%link', '<i class="fa fa-long-arrow-left"></i><span class="screen-reader-text">%title</span>' ); ?>
            </div><!-- .single-link-previous -->
			<?php endif ?>

			<?php the_title( '<h1 class="entry-title">', '</h1>' ); ?>

			<?php if ( 'post' === get_post_type() ) :  ?>
            <div class="single-link-next">
				<?php echo get_next_post_link( '%link', '<i class="fa fa-long-arrow-right"></i><span class="screen-reader-text">%title</span>' ); ?>
            </div><!-- .single-link-next -->
			<?php endif ?>
			<?php if ( 'post' === get_post_type() ) :  ?>
            <div class="single-post-meta">
				<?php photo_perfect_posted_on_custom(); ?>
            </div><!-- .single-post-meta -->
			<?php endif ?>
			<?php
		} elseif ( is_archive() ) {
			?>
				<?php the_archive_title( '<h1 class="page-title">', '</h1>' ); ?>
				<?php the_archive_description( '<div class="taxonomy-description">', '</div>' ); ?>
			<?php
		}
		?>
    </div><!-- #separator-title -->
    <?php
    $output = ob_get_contents();
    ob_end_clean();

    $output = apply_filters( 'photo_perfect_filter_title_separator_content', $output );

    if ( $output ) {
	    echo $output;
    }

	}
endif;

add_action( 'photo_perfect_action_content', 'photo_perfect_add_separator_title' );

if ( ! function_exists( 'photo_perfect_add_front_page_widget_area' ) ) :

	/**
	 * Add Front Page Widget area.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_front_page_widget_area() {

		if ( is_front_page() && is_active_sidebar( 'sidebar-front-page-widget-area' ) ) {
		  echo '<div id="sidebar-front-page-widget-area" class="widget-area">';
	    dynamic_sidebar( 'sidebar-front-page-widget-area' );
		  echo '</div><!-- #sidebar-front-page-widget-area -->';
		}

	}
endif;

add_action( 'photo_perfect_action_content', 'photo_perfect_add_front_page_widget_area' );


if ( ! function_exists( 'photo_perfect_site_branding' ) ) :

	/**
	 * Site branding.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_site_branding() {

	?>
    <div class="container">

      <div class="site-branding">
	        <?php
	        if ( function_exists( 'has_custom_logo' ) ) {
	        	echo '<div id="site-logo">';
		        the_custom_logo();
	        	echo '</div><!-- #site-logo -->';
	        }
	        ?>

        <?php $show_title = photo_perfect_get_option( 'show_title' ); ?>
        <?php $show_tagline = photo_perfect_get_option( 'show_tagline' ); ?>

        <?php if ( true === $show_title || true === $show_tagline ) : ?>
        	<div id="site-identity">
        		<?php if ( true === $show_title ) :  ?>
        			<?php if ( is_front_page() ) : ?>
        				<h1 class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
        			<?php else : ?>
        				<p class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
        			<?php endif; ?>
        		<?php endif; ?>

        		<?php if ( true === $show_tagline ) :  ?>
        			<p class="site-description"><?php bloginfo( 'description' ); ?></p>
        		<?php endif; ?>
        	</div><!-- #site-identity -->
        <?php endif; ?>

      </div><!-- .site-branding -->

    </div><!-- .container -->

    <?php

	}

endif;

add_action( 'photo_perfect_action_header', 'photo_perfect_site_branding' );


if ( ! function_exists( 'photo_perfect_add_primary_navigation' ) ) :

	/**
	 * Add Primary navigation.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_primary_navigation() {

		if ( ! has_nav_menu( 'primary' ) ) {
			return;
		}
		$header_menu_text = photo_perfect_get_option( 'header_menu_text' );
	?>
    <div id="main-nav" class="clear-fix">
        <div class="container">
        <nav id="site-navigation" class="header-navigation" role="navigation">
          <button class="menu-toggle" aria-controls="primary-menu" aria-expanded="false"><span><?php echo esc_html( $header_menu_text ); ?></span>
            <i class="fa fa-align-justify"></i></button>
            <div class="wrap-menu-content">
					<?php
						wp_nav_menu(
							array(
								'theme_location' => 'primary',
								'menu_id'        => 'primary-menu',
								'item_spacing'   => 'discard',
							)
						);
					?>
            </div><!-- .menu-content -->
        </nav><!-- #site-navigation -->
       </div> <!-- .container -->
    </div> <!-- #main-nav -->
    <?php
	}

endif;

add_action( 'photo_perfect_action_before_header', 'photo_perfect_add_primary_navigation', 20 );


if ( ! function_exists( 'photo_perfect_add_category_navigation' ) ) :

	/**
	 * Category navigation.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_category_navigation() {

		$show_category_dropdown = photo_perfect_get_option( 'show_category_dropdown' );
		if ( true !== $show_category_dropdown ) {
			return;
		}
		$show_categories_as_navigation = photo_perfect_get_option( 'show_categories_as_navigation' );
		if ( true === $show_categories_as_navigation ) {
			return;
		}
	?>
    <div id="category-menu" class="clear-fix header-navigation">
      <div class="container">
        <?php photo_perfect_render_category_menu_listing(); ?>
      </div><!-- .container -->
    </div><!-- #category-menu -->
    <?php
	}

endif;


add_action( 'photo_perfect_action_before_header', 'photo_perfect_add_category_navigation', 22 );

if ( ! function_exists( 'photo_perfect_footer_copyright' ) ) :

	/**
	 * Footer copyright.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_footer_copyright() {

		// Check if footer is disabled.
		$footer_status = apply_filters( 'photo_perfect_filter_footer_status', true );
		if ( true !== $footer_status ) {
			return;
		}

		// Footer Menu content.
		$footer_menu_content = wp_nav_menu( array(
			'theme_location' => 'footer',
			'container'      => 'div',
			'container_id'   => 'footer-navigation',
			'depth'          => 1,
			'fallback_cb'    => false,
			'echo'           => false,
			'item_spacing'   => 'discard',
		) );

		// Copyright content.
		$copyright_text = photo_perfect_get_option( 'copyright_text' );
		$copyright_text = apply_filters( 'photo_perfect_filter_copyright_text', $copyright_text );
		if ( ! empty( $copyright_text ) ) {
			$copyright_text = wp_kses_data( $copyright_text );
			$copyright_text = photo_perfect_apply_theme_shortcode( $copyright_text );
		}

		// Powered by content.
		$powered_by_text = photo_perfect_get_option( 'powered_by_text' );
		$powered_by_text = apply_filters( 'photo_perfect_filter_powered_by_text', $powered_by_text );
		if ( ! empty( $powered_by_text ) ) {
			$powered_by_text = wp_kses_data( $powered_by_text );
			$powered_by_text = photo_perfect_apply_theme_shortcode( $powered_by_text );
		}

	?>

    <div class="inner-wrapper">

      <div class="footer-left">
		<?php the_widget( 'Photo_Perfect_Social_Widget', array() ); ?>
        <?php if ( ! empty( $copyright_text ) ) :  ?>
          <div class="copyright">
            <?php echo $copyright_text; ?>
          </div><!-- .copyright -->
        <?php endif ?>
      </div><!-- .footer-left -->

      <div class="footer-right">
        <?php if ( ! empty( $footer_menu_content ) ) :  ?>
			<?php echo $footer_menu_content; ?>
        <?php endif ?>
        <?php if ( ! empty( $powered_by_text ) ) :  ?>
        	<div class="site-info">
				<?php echo $powered_by_text; ?>
        	</div><!-- .site-info -->
        <?php endif ?>

      </div><!-- .footer-right -->

    </div><!-- .inner-wrapper -->
    <?php

	}

endif;

add_action( 'photo_perfect_action_footer', 'photo_perfect_footer_copyright', 10 );


if ( ! function_exists( 'photo_perfect_add_sidebar' ) ) :

	/**
	 * Add sidebar.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_sidebar() {

		global $post;

		$global_layout = photo_perfect_get_option( 'global_layout' );
		$global_layout = apply_filters( 'photo_perfect_filter_theme_global_layout', $global_layout );

		// Check if single.
		if ( $post && is_singular() ) {
			$post_options = get_post_meta( $post->ID, 'theme_settings', true );
			if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
				$global_layout = $post_options['post_layout'];
			}
		}

		// Include Primary sidebar.
		if ( 'no-sidebar' !== $global_layout ) {
			get_sidebar();
		}
		// Include Secondary sidebar.
		switch ( $global_layout ) {
		  case 'three-columns':
		  case 'three-columns-pcs':
		  case 'three-columns-cps':
		  case 'three-columns-psc':
		  case 'three-columns-pcs-equal':
		  case 'three-columns-scp-equal':
		    get_sidebar( 'secondary' );
		    break;

		  default:
		    break;
		}


	}

endif;

add_action( 'photo_perfect_action_sidebar', 'photo_perfect_add_sidebar' );


if ( ! function_exists( 'photo_perfect_custom_posts_navigation' ) ) :
	/**
	 * Posts navigation.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_custom_posts_navigation() {

		$pagination_type = photo_perfect_get_option( 'pagination_type' );

		switch ( $pagination_type ) {

			case 'default':
				the_posts_navigation();
			break;

			case 'numeric':
				if ( function_exists( 'wp_pagenavi' ) ) {
					wp_pagenavi();
				} else {
					the_posts_pagination();
				}
			break;

			default:
			break;
		}

	}
endif;

add_action( 'photo_perfect_action_posts_navigation', 'photo_perfect_custom_posts_navigation' );


if ( ! function_exists( 'photo_perfect_add_image_in_single_display' ) ) :

	/**
	 * Add image in single post.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_image_in_single_display() {

		global $post;

		if ( has_post_thumbnail() ) {

			$values = get_post_meta( $post->ID, 'theme_settings', true );
			$theme_settings_single_image = isset( $values['single_image'] ) ? esc_attr( $values['single_image'] ) : '';
			$theme_settings_single_image_alignment = isset( $values['single_image_alignment'] ) ? esc_attr( $values['single_image_alignment'] ) : '';
			$theme_settings_show_image_as_banner = isset( $values['show_image_as_banner'] ) ? absint( $values['show_image_as_banner'] ) : 0;

			// Return becasue we need to show image in banner, not here.
			if ( 1 === $theme_settings_show_image_as_banner ) {
				return;
			}

			if ( ! $theme_settings_single_image ) {
				$theme_settings_single_image = photo_perfect_get_option( 'single_image' );
			}
			if ( ! $theme_settings_single_image_alignment ) {
				$theme_settings_single_image_alignment = photo_perfect_get_option( 'single_image_alignment' );
			}

			if ( 'disable' != $theme_settings_single_image ) {

				$featured_image_full_url = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );

			  echo '<a href=" ' . esc_url( $featured_image_full_url ) .'">';
			  $args = array(
			    'class' => 'align' . esc_attr( $theme_settings_single_image_alignment ),
			  );
			  the_post_thumbnail( esc_attr( $theme_settings_single_image ), $args );
			  echo '</a>';
			}
		}

	}

endif;

add_action( 'photo_perfect_single_image', 'photo_perfect_add_image_in_single_display' );

if ( ! function_exists( 'photo_perfect_footer_goto_top' ) ) :

	/**
	 * Render Go to top.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_footer_goto_top() {

		$go_to_top = photo_perfect_get_option( 'go_to_top' );
		if ( true !== $go_to_top ) {
			return;
		}
		echo '<a href="#page" class="scrollup" id="btn-scrollup"><i class="fa fa-level-up"></i></a>';

	}

endif;

add_action( 'photo_perfect_action_after', 'photo_perfect_footer_goto_top', 20 );

if ( ! function_exists( 'photo_perfect_add_custom_banner' ) ) :

	/**
	 * Add custom banner.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_custom_banner() {

		$custom_banner_status = apply_filters( 'photo_perfect_filter_custom_banner_status', false );
		if ( true !== $custom_banner_status ) {
			return;
		}

		$custom_banner_details = array();
		$custom_banner_details = apply_filters( 'photo_perfect_filter_custom_banner_details', $custom_banner_details );

		// Bail if not custom banner details.
		if ( empty( $custom_banner_details ) ) {
		  return;
		}

		$flag_apply_slider = apply_filters( 'photo_perfect_filter_slider_status', true );
		if ( false === $flag_apply_slider ) {
			// Render custom banner.
			photo_perfect_render_custom_banner( $custom_banner_details );
		}
	}

endif;

add_action( 'photo_perfect_action_after_header', 'photo_perfect_add_custom_banner', 15 );

if ( ! function_exists( 'photo_perfect_get_custom_banner_details' ) ) :

	/**
	 * Get custom banner details.
	 *
	 * @since 1.0.0
	 *
	 * @param array $detail Banner detail
	 */
	function photo_perfect_get_custom_banner_details( $detail ) {

		$detail = array();

		if ( get_header_image() ) {
		  $detail['image_url']     = get_header_image();
		  $detail['image_width']   = get_custom_header()->width;
		  $detail['image_height']  = get_custom_header()->height;
		  $detail['image_alt']     = esc_attr( photo_perfect_get_option( 'ch_image_alt' ) );
		  $detail['link_url']      = esc_url( photo_perfect_get_option( 'ch_link_url' ) );
		  $detail['link_url_open'] = photo_perfect_get_option( 'ch_link_url_open' );
		}
		if ( ! is_front_page() ) {
			$ch_shrunk_header_image = photo_perfect_get_option( 'ch_shrunk_header_image' );
			$enable_shrunk_header = photo_perfect_get_option( 'enable_shrunk_header' );
			if ( ! empty( $ch_shrunk_header_image ) && true === $enable_shrunk_header ) {
				$detail['image_url']     = esc_url( $ch_shrunk_header_image );
				$detail['image_width']   = '';
				$detail['image_height']  = '';
				$detail['image_alt']     = '';
				$detail['link_url']      = '';
				$detail['link_url_open'] = '';
			}
		}

		if ( is_singular( array( 'post', 'page' ) ) && has_post_thumbnail() ) {
			global $post;
			$required_image_size = 'full';
			$values = get_post_meta( $post->ID, 'theme_settings', true );
			$theme_settings_show_image_as_banner = isset( $values['show_image_as_banner'] ) ? absint( $values['show_image_as_banner'] ) : 0;
			$theme_settings_shrunk_header = isset( $values['shrunk_header'] ) ? esc_attr( $values['shrunk_header'] ) : 'default';
			$default_shrink_status = ( true === photo_perfect_get_option( 'enable_shrunk_header' ) ) ? 'enable' : 'disable' ;
			if ( 'default' === $theme_settings_shrunk_header ) {
				$theme_settings_shrunk_header = $default_shrink_status;
			}
			if ( 'enable' === $theme_settings_shrunk_header) {
				$required_image_size = 'photo-perfect-small-banner';
			}
			if ( 1 === $theme_settings_show_image_as_banner ) {
				// Load page info in detail.
				$featured_image_info = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), $required_image_size );
				if ( ! empty( $featured_image_info ) ) {
					$detail = array();
					$detail['image_url']    = esc_url( $featured_image_info[0] );
					$detail['image_width']  = absint( $featured_image_info[1] );
					$detail['image_height'] = absint( $featured_image_info[2] );
					$detail['image_alt']    = esc_attr( get_the_title( $post->ID ) );

				}
			}
		}
		return $detail;

	}

endif;

add_filter( 'photo_perfect_filter_custom_banner_details', 'photo_perfect_get_custom_banner_details' );

if ( ! function_exists( 'photo_perfect_load_archive_loop_content' ) ) :

	/**
	 * Load Archive Loop Content.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_load_archive_loop_content() {

		$archive_layout = photo_perfect_get_option( 'archive_layout' );
		if ( 'masonry' === $archive_layout ) {
			get_template_part( 'template-parts/content', 'masonry' );
		} else if ( 'full' === $archive_layout ) {
			get_template_part( 'template-parts/content', 'full' );
		} else {
			get_template_part( 'template-parts/content', get_post_format() );
		}

	}

endif;

add_action( 'photo_perfect_action_loop', 'photo_perfect_load_archive_loop_content' );

if ( ! function_exists( 'photo_perfect_add_breadcrumb' ) ) :

	/**
	 * Add breadcrumb.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_breadcrumb() {

		// Bail if Breadcrumb disabled.
		$breadcrumb_type = photo_perfect_get_option( 'breadcrumb_type' );
		if ( 'disabled' === $breadcrumb_type ) {
		  return;
		}
		// Bail if front Page.
		if ( is_front_page() || is_home() ) {
		  return;
		}
		echo '<div id="breadcrumb">';
		switch ( $breadcrumb_type ) {
			case 'simple':
				$breadcrumb_separator = photo_perfect_get_option( 'breadcrumb_separator' );
				$args = array(
					'separator' => '',
					);
				photo_perfect_simple_breadcrumb( $args );
				break;

			case 'advanced':
				if ( function_exists( 'bcn_display' ) ) {
					bcn_display();
				}
				break;

			default:
				break;
		}
		echo '</div><!-- #breadcrumb -->';

	}

endif;

add_action( 'photo_perfect_action_content', 'photo_perfect_add_breadcrumb', 12 );

if ( ! function_exists( 'photo_perfect_check_custom_banner_status' ) ) :

	/**
	 * Get custom banner details.
	 *
	 * @since 1.0.0
	 *
	 * @param bool $input Banner status
	 * @return bool Status to whether show or hide
	 */
	function photo_perfect_check_custom_banner_status( $input ) {

		$ch_status = photo_perfect_get_option( 'ch_status' );

		if ( 'entire-site' === $ch_status ) {
		  $input = true;
		}
		elseif ( 'disabled' === $ch_status ) {
		  $input = false;
		}
		elseif ( 'home-page' === $ch_status ) {
		  if ( is_front_page() || is_home() ) {
		    $input = true;
		  }
		}
		elseif ( 'entire-site-except-home' === $ch_status ) {
		  if ( ! ( is_front_page() || is_home() ) ) {
		    $input = true;
		  }
		}

		if ( is_singular( array( 'post', 'page' ) ) ) {
			global $post;
			$values = get_post_meta( $post->ID, 'theme_settings', true );
			$theme_settings_show_image_as_banner = isset( $values['show_image_as_banner'] ) ? absint( $values['show_image_as_banner'] ) : 0;
			if ( 1 === $theme_settings_show_image_as_banner ) {
				$input = true;
			}
		}

		return $input;

	}

endif;

add_filter( 'photo_perfect_filter_custom_banner_status', 'photo_perfect_check_custom_banner_status' );

if ( ! function_exists( 'photo_perfect_add_header_search_form' ) ) :

	/**
	 * Add header search form.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_header_search_form() {

		$show_search_form = photo_perfect_get_option( 'show_search_form' );
		if ( true !== $show_search_form ) {
			return;
		}
		?>
		<div id="header-extra">
			<div class="container">
				<?php get_search_form(); ?>
			</div>
		</div>
		<?php

	}

endif;

add_action( 'photo_perfect_action_after_header', 'photo_perfect_add_header_search_form', 20 );

if ( ! function_exists( 'photo_perfect_add_header_category_menu' ) ) :

	/**
	 * Add header category menu.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_header_category_menu() {

		$show_category_dropdown = photo_perfect_get_option( 'show_category_dropdown' );
		if ( true !== $show_category_dropdown ) {
			return;
		}
		$show_categories_as_navigation = photo_perfect_get_option( 'show_categories_as_navigation' );
		if ( true !== $show_categories_as_navigation ) {
			return;
		}
		?>
	    <div id="category-menu" class="clear-fix header-navigation nav-cat">
	      <div class="container">
		      <?php photo_perfect_render_category_menu_listing(); ?>
	      </div><!-- .container -->
	    </div><!-- #category-menu -->
		<?php

	}

endif;

add_action( 'photo_perfect_action_after_header', 'photo_perfect_add_header_category_menu', 22 );


if ( ! function_exists( 'photo_perfect_swirl_design_customization' ) ) :

	/**
	 * Customize swirl design.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_swirl_design_customization( $input ) {

		global $post, $wp_query;
		// Get Page ID outside Loop.
		$page_id = $wp_query->get_queried_object_id();

		// Front page displays in Reading Settings.
		$page_on_front  = absint( get_option( 'page_on_front' ) );
		$page_for_posts = absint( get_option( 'page_for_posts' ) );
		$show_on_front  = esc_attr( get_option( 'show_on_front' ) );

		if ( is_front_page() || is_home() ) {

			$home_disable_swirl_design     = photo_perfect_get_option( 'home_disable_swirl_design' );

			if ( true === $home_disable_swirl_design ) {
				// Swirl design is disabled.
				$input = '';
			}
			else {
				// Swirl design is active.
				// Prepare swirl design.
				$swirl_url = get_template_directory_uri() . '/images/title-tag.png';
				$home_swirl_image = photo_perfect_get_option( 'home_swirl_image' );
				if ( ! empty( $home_swirl_image ) ) {
					$swirl_url = $home_swirl_image;
				}
				$swirl_html = '<img src="' . esc_url( $swirl_url ) . '" alt="" />';
				$swirl_html = '<span class="title-tag">' . $swirl_html . '</span>';
				$swirl_html = '<div id="separator-title">' . $swirl_html . '</div>';

				// Fix for latest posts setting.
				if ( 'posts' === $show_on_front ) {
					$input = $swirl_html;
				}
				// Static front page.
				if ( $page_id > 0 && $page_id === $page_on_front ) {
					$home_replace_front_page_title = photo_perfect_get_option( 'home_replace_front_page_title' );
					if ( true === $home_replace_front_page_title ) {
						$input = $swirl_html;
					}
				}
				// Blog page.
				if ( $page_id > 0 && $page_id === $page_for_posts ) {
					$input = $swirl_html;
				}
			}



		}

		return $input;

	}

endif;

add_filter( 'photo_perfect_filter_title_separator_content', 'photo_perfect_swirl_design_customization' );
