<?php
/**
 * Helper functions related to customizer and options.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_get_global_layout_options' ) ) :

	/**
	 * Returns global layout options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_global_layout_options() {

		$choices = array(
			'left-sidebar'            => esc_html__( 'Primary Sidebar - Content', 'photo-perfect-pro' ),
			'right-sidebar'           => esc_html__( 'Content - Primary Sidebar', 'photo-perfect-pro' ),
			'three-columns'           => esc_html__( 'Three Columns ( Secondary - Content - Primary )', 'photo-perfect-pro' ),
			'three-columns-pcs'       => esc_html__( 'Three Columns ( Primary - Content - Secondary )', 'photo-perfect-pro' ),
			'three-columns-cps'       => esc_html__( 'Three Columns ( Content - Primary - Secondary )', 'photo-perfect-pro' ),
			'three-columns-psc'       => esc_html__( 'Three Columns ( Primary - Secondary - Content )', 'photo-perfect-pro' ),
			'three-columns-pcs-equal' => esc_html__( 'Three Columns ( Equal Primary - Content - Secondary )', 'photo-perfect-pro' ),
			'three-columns-scp-equal' => esc_html__( 'Three Columns ( Equal Secondary - Content - Primary )', 'photo-perfect-pro' ),
			'no-sidebar'              => esc_html__( 'No Sidebar', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_layout_options', $choices );
		return $output;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_site_layout_options' ) ) :

	/**
	 * Returns site layout options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_site_layout_options() {

		$choices = array(
			'fluid' => esc_html__( 'Fluid', 'photo-perfect-pro' ),
			'boxed' => esc_html__( 'Boxed', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_site_layout_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_featured_slider_content_options' ) ) :

	/**
	 * Returns the featured slider content options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function photo_perfect_get_featured_slider_content_options() {

		$choices = array(
			'home-page'   => esc_html__( 'Enabled', 'photo-perfect-pro' ),
			'disabled'    => esc_html__( 'Disabled', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_featured_slider_content_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_featured_slider_type' ) ) :

	/**
	 * Returns the featured slider type.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function photo_perfect_get_featured_slider_type() {

		$choices = array(
			'featured-image' => __( 'Featured Images', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_featured_slider_type', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_featured_slider_transition_effects' ) ) :

	/**
	 * Returns the featured slider transition effects.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function photo_perfect_get_featured_slider_transition_effects() {

		$choices = array(
			'fade'       => _x( 'fade', 'Transition Effect', 'photo-perfect-pro' ),
			'fadeout'    => _x( 'fadeout', 'Transition Effect', 'photo-perfect-pro' ),
			'none'       => _x( 'none', 'Transition Effect', 'photo-perfect-pro' ),
			'scrollHorz' => _x( 'scrollHorz', 'Transition Effect', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_featured_slider_transition_effects', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;


if ( ! function_exists( 'photo_perfect_get_pagination_type_options' ) ) :

	/**
	 * Returns pagination type options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_pagination_type_options() {

		$choices = array(
			'default'               => esc_html__( 'Default (Older / Newer Post)', 'photo-perfect-pro' ),
			'numeric'               => esc_html__( 'Numeric', 'photo-perfect-pro' ),
			'infinite-scroll'       => esc_html__( 'Infinite Scroll - Scroll', 'photo-perfect-pro' ),
			'infinite-scroll-click' => esc_html__( 'Infinite Scroll - Click', 'photo-perfect-pro' ),

		);
		return $choices;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_archive_layout_options' ) ) :

	/**
	 * Returns archive layout options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_archive_layout_options() {

		$choices = array(
			'masonry' => esc_html__( 'Masonry', 'photo-perfect-pro' ),
			'excerpt' => esc_html__( 'Post Excerpt', 'photo-perfect-pro' ),
			'full'    => esc_html__( 'Full', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_archive_layout_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;


if ( ! function_exists( 'photo_perfect_get_image_sizes_options' ) ) :

	/**
	 * Returns image sizes options.
	 *
	 * @since 1.0.0
	 *
	 * @param bool  $add_disable True for adding No Image option.
	 * @param array $allowed Allowed image size options.
	 * @param bool $show_dimension Enable/disable dimention in options.
	 * @return array Image size options.
	 */
	function photo_perfect_get_image_sizes_options( $add_disable = true, $allowed = array(), $show_dimension = true ) {

		global $_wp_additional_image_sizes;
		$choices = array();
		if ( true === $add_disable ) {
			$choices['disable'] = esc_html__( 'No Image', 'photo-perfect-pro' );
		}
		$choices['thumbnail'] = esc_html__( 'Thumbnail', 'photo-perfect-pro' );
		$choices['medium']    = esc_html__( 'Medium', 'photo-perfect-pro' );
		$choices['large']     = esc_html__( 'Large', 'photo-perfect-pro' );
		$choices['full']      = esc_html__( 'Full (original)', 'photo-perfect-pro' );

		if ( true === $show_dimension ) {
			foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
				$choices[ $_size ] = ucfirst( $_size ) . ' (' . get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
			}
		}

		if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {
			foreach ( $_wp_additional_image_sizes as $key => $size ) {
				$choices[ $key ] = $key;
				if ( true === $show_dimension ){
					$choices[ $key ] .= ' ('. $size['width'] . 'x' . $size['height'] . ')';
				}
			}
		}

		if ( ! empty( $allowed ) ) {
			foreach ( $choices as $key => $value ) {
				if ( ! in_array( $key, $allowed ) ) {
					unset( $choices[ $key ] );
				}
			}
		}

		return $choices;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_image_options' ) ) :

	/**
	 * Returns image options.
	 *
	 * @since 1.0.0
	 * @deprecated 1.4
	 *
	 * @param bool  $add_disable True for adding No Image option.
	 * @param array $allowed Allowed image size options.
	 * @return array Image size options.
	 */
	function photo_perfect_get_image_options( $add_disable = true, $allowed = array() ) {

		_deprecated_function( 'photo_perfect_get_image_options', '1.4', 'photo_perfect_get_image_sizes_options' );

		return photo_perfect_get_image_sizes_options( $add_disable, $allowed );

	}

endif;


if ( ! function_exists( 'photo_perfect_get_image_alignment_options' ) ) :

	/**
	 * Returns image options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_image_alignment_options() {

		$choices = array(
			'none'   => esc_html__( 'None', 'photo-perfect-pro' ),
			'left'   => esc_html__( 'Left', 'photo-perfect-pro' ),
			'center' => esc_html__( 'Center', 'photo-perfect-pro' ),
			'right'  => esc_html__( 'Right', 'photo-perfect-pro' ),
		);
		return $choices;

	}

endif;


if ( ! function_exists( 'photo_perfect_get_breadcrumb_type_options' ) ) :

	/**
	 * Returns breadcrumb type options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_breadcrumb_type_options() {

		$choices = array(
			'disabled' => esc_html__( 'Disabled', 'photo-perfect-pro' ),
			'simple'   => esc_html__( 'Simple', 'photo-perfect-pro' ),
			'advanced' => esc_html__( 'Advanced', 'photo-perfect-pro' ),
		);
		return $choices;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_custom_header_status_options' ) ) :

	/**
	 * Returns custom header status options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_custom_header_status_options() {

		$choices = array(
			'disabled'                => esc_html__( 'Disabled', 'photo-perfect-pro' ),
			'entire-site'             => esc_html__( 'Entire Site', 'photo-perfect-pro' ),
			'entire-site-except-home' => esc_html__( 'Entire Site Except Home', 'photo-perfect-pro' ),
			'home-page'               => esc_html__( 'Home Page / Front Page', 'photo-perfect-pro' ),
		);
		$output = apply_filters( 'photo_perfect_filter_custom_header_options', $choices );
		if ( ! empty( $output ) ) {
			ksort( $output );
		}
		return $output;

	}

endif;

if ( ! function_exists( 'photo_perfect_get_masonry_column_options' ) ) :

	/**
	 * Returns masonry column options.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_get_masonry_column_options() {

		$choices = array(
			'3' => 3,
			'4' => 4,
		);
		return $choices;

	}

endif;

if( ! function_exists( 'photo_perfect_get_font_family_theme_settings_options' ) ) :

  /**
   * Returns font family theme settings options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_font_family_theme_settings_options(){

    $choices = array(
      'font_site_title' => array(
          'label'   => __( 'Site Title', 'photo-perfect-pro' ),
          'default' => 'arizonia',
        ),
      'font_site_tagline' => array(
          'label'   => __( 'Site Tagline', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
      'font_site_default' => array(
          'label'   => __( 'Default', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
      'font_content_title' => array(
          'label'   => __( 'Content Title', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
      'font_content_body' => array(
          'label'   => __( 'Content Body', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
      'font_heading_tags' => array(
          'label'   => __( 'Heading Tags', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
      'font_navigation' => array(
          'label'   => __( 'Navigation', 'photo-perfect-pro' ),
          'default' => 'open-sans',
        ),
    );
    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_customizer_font_options' ) ) :

  /**
   * Returns customizer font options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_customizer_font_options(){

    $web_fonts = photo_perfect_get_web_fonts();
    $os_fonts  = photo_perfect_get_os_fonts();

    $web_fonts = array_merge( $web_fonts, $os_fonts );

    if ( ! empty( $web_fonts ) ) {
      ksort( $web_fonts );
    }

    $choices = array();

    if ( ! empty( $web_fonts ) ) {
      foreach ( $web_fonts as $k => $v ) {
        $choices[$k] = esc_html( $v['label'] );
      }
    }
    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_web_fonts' ) ) :

  /**
   * Returns web font options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_web_fonts(){

    $choices = array(
      'open-sans' => array(
        'name'  => 'Open Sans',
        'label' => "'Open Sans', sans-serif",
      ),
      'roboto' => array(
        'name'  => 'Roboto',
        'label' => "'Roboto', sans-serif",
      ),
      'arizonia' => array(
        'name'  => 'Arizonia',
        'label' => "'Arizonia', cursive",
      ),
      'raleway' => array(
        'name'  => 'Raleway',
        'label' => "'Raleway', sans-serif",
      ),
      'droid-sans' => array(
        'name'  => 'Droid Sans',
        'label' => "'Droid Sans', sans-serif",
      ),
      'lato' => array(
        'name'  => 'Lato',
        'label' => "'Lato', sans-serif",
      ),
      'dosis' => array(
        'name'  => 'Dosis',
        'label' => "'Dosis', sans-serif",
      ),
      'slabo-27px' => array(
        'name'  => 'Slabo 27px',
        'label' => "'Slabo 27px', serif",
      ),
      'oswald' => array(
        'name'  => 'Oswald',
        'label' => "'Oswald', sans-serif",
      ),
      'pt-sans-narrow' => array(
        'name'  => 'PT Sans Narrow',
        'label' => "'PT Sans Narrow', sans-serif",
      ),
      'josefin-slab' => array(
        'name'  => 'Josefin Slab',
        'label' => "'Josefin Slab', serif",
      ),
      'alegreya' => array(
        'name'  => 'Alegreya',
        'label' => "'Alegreya', serif",
      ),
      'exo' => array(
        'name'  => 'Exo',
        'label' => "'Exo', sans-serif",
      ),
      'signika' => array(
        'name'  => 'Signika',
        'label' => "'Signika', sans-serif",
      ),
      'lobster' => array(
        'name'  => 'Lobster',
        'label' => "'Lobster', cursive",
      ),
      'indie-flower' => array(
        'name'  => 'Indie Flower',
        'label' => "'Indie Flower', cursive",
      ),
      'shadows-into-light' => array(
        'name'  => 'Shadows Into Light',
        'label' => "'Shadows Into Light', cursive",
      ),
      'kaushan-script' => array(
        'name'  => 'Kaushan Script',
        'label' => "'Kaushan Script', cursive",
      ),
      'dancing-script' => array(
        'name'  => 'Dancing Script',
        'label' => "'Dancing Script', cursive",
      ),
      'fredericka-the-great' => array(
        'name'  => 'Fredericka the Great',
        'label' => "'Fredericka the Great', cursive",
      ),
      'covered-by-your-grace' => array(
        'name'  => 'Covered By Your Grace',
        'label' => "'Covered By Your Grace', cursive",
      ),
    );
    $choices = apply_filters( 'photo_perfect_filter_web_fonts', $choices );

    if ( ! empty( $choices ) ) {
      ksort( $choices );
    }

    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_os_fonts' ) ) :

  /**
   * Returns OS font options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_os_fonts(){

    $choices = array(
      'arial' => array(
        'name'  => 'Arial',
        'label' => "'Arial', sans-serif",
      ),
      'georgia' => array(
        'name'  => 'Georgia',
        'label' => "'Georgia', serif",
      ),
      'cambria' => array(
        'name'  => 'Cambria',
        'label' => "'Cambria', Georgia, serif",
      ),
      'tahoma' => array(
        'name'  => 'Tahoma',
        'label' => "'Tahoma', Geneva, sans-serif",
      ),
      'sans-serif' => array(
        'name'  => 'Sans Serif',
        'label' => "'Sans Serif', Arial",
      ),
      'verdana' => array(
        'name'  => 'Verdana',
        'label' => "'Verdana', Geneva, sans-serif",
      ),
    );
    $choices = apply_filters( 'photo_perfect_filter_os_fonts', $choices );

    if ( ! empty( $choices ) ) {
      ksort( $choices );
    }
    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_font_family_from_key' ) ) :

  /**
   * Return font family from font slug.
   *
   * @since 1.0.0
   *
   * @param string $key Font slug.
   * @return string Font name.
   */
  function photo_perfect_get_font_family_from_key( $key ){

    $output = '';

    $web_fonts = photo_perfect_get_web_fonts();
    $os_fonts  = photo_perfect_get_os_fonts();

    $fonts = array_merge( $web_fonts, $os_fonts );

    if ( isset( $fonts[ $key ] ) ) {
      $output = $fonts[ $key ]['label'];
    }
    return $output;

  }

endif;


if( ! function_exists( 'photo_perfect_get_color_sections_options' ) ) :

  /**
   * Returns color sections options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_color_sections_options(){

    $choices = array(
      'color_section_basic' => array(
          'label' => __( 'Basic Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_header' => array(
          'label' => __( 'Header Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_primary_menu' => array(
          'label' => __( 'Primary Menu Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_category_menu' => array(
          'label' => __( 'Category Menu Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_content' => array(
          'label' => __( 'Content Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_content_masonry' => array(
          'label' => __( 'Content Masonry Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_content_excerpt' => array(
          'label' => __( 'Content Excerpt Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_home_widgets' => array(
          'label' => __( 'Home Page Widgets Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_breadcrumb' => array(
          'label' => __( 'Breadcrumb Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_goto_top' => array(
          'label' => __( 'Go To Top Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_pagination' => array(
          'label' => __( 'Pagination Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_footer_widgets' => array(
          'label' => __( 'Footer Widgets Color Options', 'photo-perfect-pro' ),
        ),
      'color_section_footer_area' => array(
          'label' => __( 'Footer Area Color Options', 'photo-perfect-pro' ),
        ),
    );
    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_color_theme_settings_options' ) ) :

  /**
   * Returns color theme settings options.
   *
   * @since 1.0.0
   */
  function photo_perfect_get_color_theme_settings_options(){

    $choices = array(

      // Basic.
      'color_basic_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_text' => array(
          'label'   => __( 'Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_heading' => array(
          'label'   => __( 'Heading Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_button_text' => array(
          'label'   => __( 'Button Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_button_background' => array(
          'label'   => __( 'Button Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_button_text_hover' => array(
          'label'   => __( 'Button Text Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),
      'color_basic_button_background_hover' => array(
          'label'   => __( 'Button Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_basic',
        ),

      // Header.
      'color_header_title' => array(
          'label'   => __( 'Site Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_title_hover' => array(
          'label'   => __( 'Site Title Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_title_border' => array(
          'label'   => __( 'Title Border Color', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_tagline' => array(
          'label'   => __( 'Site Tagline Color', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_identity_background' => array(
          'label'   => __( 'Site Identity Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_search_form_background' => array(
          'label'   => __( 'Search Form Background', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_search_button_background' => array(
          'label'   => __( 'Search Button Background', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_search_button_background_hover' => array(
          'label'   => __( 'Search Button Background Hover', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_search_button_text' => array(
          'label'   => __( 'Search Button Text', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),
      'color_header_search_button_text_hover' => array(
          'label'   => __( 'Search Button Text Hover', 'photo-perfect-pro' ),
          'section' => 'color_section_header',
        ),

      // Primary Menu.
      'color_primary_menu_main_background' => array(
          'label'   => __( 'Main Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_button_text' => array(
          'label'   => __( 'Button Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_button_text_hover' => array(
          'label'   => __( 'Button Text Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_button_background' => array(
          'label'   => __( 'Button Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_button_background_hover' => array(
          'label'   => __( 'Button Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),
      'color_primary_menu_link_background_hover' => array(
          'label'   => __( 'Link Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_primary_menu',
        ),

      // Category Menu.
      'color_category_menu_main_background' => array(
          'label'   => __( 'Main Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_button_text' => array(
          'label'   => __( 'Button Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_button_text_hover' => array(
          'label'   => __( 'Button Text Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_button_background' => array(
          'label'   => __( 'Button Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_button_background_hover' => array(
          'label'   => __( 'Button Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),
      'color_category_menu_link_background_hover' => array(
          'label'   => __( 'Link Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_category_menu',
        ),


      // Content.
      'color_content_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_title' => array(
          'label'   => __( 'Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_text' => array(
          'label'   => __( 'Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_meta_text' => array(
          'label'   => __( 'Meta Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_meta_link' => array(
          'label'   => __( 'Meta Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_meta_link_hover' => array(
          'label'   => __( 'Meta Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),
      'color_content_meta_icon' => array(
          'label'   => __( 'Meta Icon Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content',
        ),

      // Content masonry.
      'color_content_masonry_title' => array(
          'label'   => __( 'Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_masonry',
        ),
      'color_content_masonry_icon' => array(
          'label'   => __( 'Icon Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_masonry',
        ),
      'color_content_masonry_icon_hover' => array(
          'label'   => __( 'Icon Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_masonry',
        ),
      'color_content_masonry_icon_background_hover' => array(
          'label'   => __( 'Icon Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_masonry',
        ),

      // Content excerpt.
      'color_content_excerpt_thumbnail_border' => array(
          'label'   => __( 'Thumbnail Border Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_excerpt',
        ),
      'color_content_excerpt_content_background' => array(
          'label'   => __( 'Content Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_content_excerpt',
        ),

      // Sidebar
      'color_sidebar_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_sidebar',
        ),
      'color_sidebar_widget_title' => array(
          'label'   => __( 'Widget Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_sidebar',
        ),
      'color_sidebar_widget_text' => array(
          'label'   => __( 'Widget Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_sidebar',
        ),
      'color_sidebar_widget_link' => array(
          'label'   => __( 'Widget Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_sidebar',
        ),
      'color_sidebar_widget_link_hover' => array(
          'label'   => __( 'Widget Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_sidebar',
        ),

      // Home Page Widgets
      'color_home_widgets_area_background' => array(
          'label'   => __( 'Widget Area Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_home_widgets',
        ),
      'color_home_widgets_title' => array(
          'label'   => __( 'Widget Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_home_widgets',
        ),

      // Footer area.
      'color_footer_area_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_area',
        ),
      'color_footer_area_text' => array(
          'label'   => __( 'Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_area',
        ),
      'color_footer_area_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_area',
        ),
      'color_footer_area_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_area',
        ),

      // Go To Top.
      'color_goto_top_icon' => array(
          'label'   => __( 'Icon Color', 'photo-perfect-pro' ),
          'section' => 'color_section_goto_top',
        ),
      'color_goto_top_icon_hover' => array(
          'label'   => __( 'Icon Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_goto_top',
        ),
      'color_goto_top_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_goto_top',
        ),
      'color_goto_top_background_hover' => array(
          'label'   => __( 'Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_goto_top',
        ),

      // Pagination.
      'color_pagination_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_pagination',
        ),
      'color_pagination_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_pagination',
        ),
      'color_pagination_link_background' => array(
          'label'   => __( 'Link Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_pagination',
        ),
      'color_pagination_link_background_hover' => array(
          'label'   => __( 'Link Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_pagination',
        ),

      // Breadcrumb.
      'color_breadcrumb_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_breadcrumb',
        ),
      'color_breadcrumb_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_breadcrumb',
        ),
      'color_breadcrumb_link_background' => array(
          'label'   => __( 'Link Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_breadcrumb',
        ),
      'color_breadcrumb_link_background_hover' => array(
          'label'   => __( 'Link Background Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_breadcrumb',
        ),

      // Footer Widgets.
      'color_footer_widgets_background' => array(
          'label'   => __( 'Background Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_widgets',
        ),
      'color_footer_widgets_title' => array(
          'label'   => __( 'Title Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_widgets',
        ),
      'color_footer_widgets_text' => array(
          'label'   => __( 'Text Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_widgets',
        ),
      'color_footer_widgets_link' => array(
          'label'   => __( 'Link Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_widgets',
        ),
      'color_footer_widgets_link_hover' => array(
          'label'   => __( 'Link Hover Color', 'photo-perfect-pro' ),
          'section' => 'color_section_footer_widgets',
        ),


    );

    return $choices;

  }

endif;

if( ! function_exists( 'photo_perfect_get_default_colors' ) ) :

  /**
   * Returns default colors.
   *
   * @since 1.0.0
   *
   * @param string $scheme Color scheme.
   * @return array Color values based on scheme.
   */
  function photo_perfect_get_default_colors( $scheme = 'default' ){

    $output = array();

    switch ( $scheme ) {

      case 'default':
      default:
        $output = array(

          // Basic.
          'color_basic_background'              => '#f9f9f9',
          'color_basic_text'                    => '#666666',
          'color_basic_link'                    => '#1f1f29',
          'color_basic_link_hover'              => '#333333',
          'color_basic_heading'                 => '#666666',
          'color_basic_button_background'       => '#7f7f7f',
          'color_basic_button_text'             => '#ffffff',
          'color_basic_button_background_hover' => '#1f1f29',
          'color_basic_button_text_hover'       => '#ffffff',

          // Header.
					'color_header_title'                          => '#ffffff',
					'color_header_title_hover'                    => '#ffffff',
					'color_header_tagline'                        => '#ffffff',
					'color_header_title_border'                   => '#ffffff',
					'color_header_identity_background'            => '#000000',
					'color_header_search_form_background'         => '#000000',
					'color_header_search_button_background'       => '#9f9f9f',
					'color_header_search_button_background_hover' => '#1f1f29',
					'color_header_search_button_text'             => '#ffffff',
					'color_header_search_button_text_hover'       => '#ffffff',


          // Content.
          'color_content_background'      => '#f9f9f9',
          'color_content_title'           => '#666666',
          'color_content_text'            => '#666666',
          'color_content_link'            => '#1f1f29',
          'color_content_link_hover'      => '#333333',
          'color_content_meta_text'       => '#666666',
          'color_content_meta_link'       => '#1f1f29',
          'color_content_meta_link_hover' => '#333333',
          'color_content_meta_icon'       => '#666666',

          // Content masonry.
					'color_content_masonry_title'                 => '#ffffff',
					'color_content_masonry_icon'                  => '#ffffff',
					'color_content_masonry_icon_hover'            => '#ffffff',
					'color_content_masonry_icon_background_hover' => '#422e5b',

          // Content excerpt.
					'color_content_excerpt_thumbnail_border'   => '#ffffff',
					'color_content_excerpt_content_background' => '#ffffff',

          // Sidebar.
          'color_sidebar_background'        => '#ffffff',
          'color_sidebar_widget_title'      => '#666666',
          'color_sidebar_widget_text'       => '#666666',
          'color_sidebar_widget_link'       => '#1f1f29',
          'color_sidebar_widget_link_hover' => '#333333',

          // Home Page Widgets.
					'color_home_widgets_title'           => '#666666',
					'color_home_widgets_area_background' => '#ffffff',

          // Primary Menu.
					'color_primary_menu_main_background'         => '#4b4b4b',
					'color_primary_menu_button_text'             => '#ffffff',
					'color_primary_menu_button_text_hover'       => '#ffffff',
					'color_primary_menu_button_background'       => '#4b4b4b',
					'color_primary_menu_button_background_hover' => '#1f1f29',
					'color_primary_menu_link'                    => '#ffffff',
					'color_primary_menu_link_hover'              => '#ffffff',
					'color_primary_menu_link_background_hover'   => '#ffffff',

          // Category Menu.
					'color_category_menu_main_background'         => '#4b4b4b',
					'color_category_menu_button_text'             => '#ffffff',
					'color_category_menu_button_text_hover'       => '#ffffff',
					'color_category_menu_button_background'       => '#4b4b4b',
					'color_category_menu_button_background_hover' => '#1f1f29',
					'color_category_menu_link'                    => '#ffffff',
					'color_category_menu_link_hover'              => '#ffffff',
					'color_category_menu_link_background_hover'   => '#ffffff',

          // Footer Widgets.
          'color_footer_widgets_background' => '#dddddd',
          'color_footer_widgets_title'      => '#666666',
          'color_footer_widgets_text'       => '#ffffff',
          'color_footer_widgets_link'       => '#1f1f29',
          'color_footer_widgets_link_hover' => '#333333',

          // Footer area.
					'color_footer_area_background' => '#1f1f29',
					'color_footer_area_text'       => '#ffffff',
					'color_footer_area_link'       => '#ffffff',
					'color_footer_area_link_hover' => '#ffffff',

          // Go To Top.
					'color_goto_top_icon'             => '#ffffff',
					'color_goto_top_icon_hover'       => '#ffffff',
					'color_goto_top_background'       => '#1f1f29',
					'color_goto_top_background_hover' => '#1f1f29',

          // Pagination.
          'color_pagination_link'                  => '#1f1f29',
          'color_pagination_link_hover'            => '#ffffff',
          'color_pagination_link_background'       => '#ffffff',
          'color_pagination_link_background_hover' => '#1f1f29',

          // Breadcrumb.
					'color_breadcrumb_link'                  => '#1f1f29',
					'color_breadcrumb_link_hover'            => '#ffffff',
					'color_breadcrumb_link_background'       => '#dddddd',
					'color_breadcrumb_link_background_hover' => '#ffffff',

        );
        break;

    }
    return $output;

  }

endif;
