<?php
/**
 * Implement custom meta boxes.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_add_theme_meta_box' ) ) :

	/**
	 * Add the Meta Box.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_add_theme_meta_box() {

		$apply_metabox_post_types = array( 'post', 'page' );

		foreach ( $apply_metabox_post_types as $key => $type ) {
			add_meta_box(
				'theme-settings',
				esc_html__( 'Theme Settings', 'photo-perfect-pro' ),
				'photo_perfect_render_theme_settings_metabox',
				$type
			);
		}

	}

endif;

add_action( 'add_meta_boxes', 'photo_perfect_add_theme_meta_box' );

if ( ! function_exists( 'photo_perfect_render_theme_settings_metabox' ) ) :

	/**
	 * Render theme settings meta box.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_render_theme_settings_metabox() {

		global $post;
		$post_id = $post->ID;

		// Meta box nonce for verification.
		wp_nonce_field( basename( __FILE__ ), 'photo_perfect_theme_settings_meta_box_nonce' );

		// Fetch Options list.
		$global_layout_options   = photo_perfect_get_global_layout_options();
		$image_size_options      = photo_perfect_get_image_sizes_options();
		$image_alignment_options = photo_perfect_get_image_alignment_options();

		// Fetch values of current post meta.
		$values = get_post_meta( $post_id, 'theme_settings', true );
		$theme_settings_post_layout = isset( $values['post_layout'] ) ? esc_attr( $values['post_layout'] ) : '';
		$theme_settings_single_image = isset( $values['single_image'] ) ? esc_attr( $values['single_image'] ) : '';
		$theme_settings_single_image_alignment = isset( $values['single_image_alignment'] ) ? esc_attr( $values['single_image_alignment'] ) : '';
		$theme_settings_show_image_as_banner = isset( $values['show_image_as_banner'] ) ? absint( $values['show_image_as_banner'] ) : 0;
		$theme_settings_shrunk_header = isset( $values['shrunk_header'] ) ? esc_attr( $values['shrunk_header'] ) : 'default';
		$theme_settings_sidebar_location_primary = isset( $values['sidebar_location_primary'] ) ? esc_attr( $values['sidebar_location_primary'] ) : '';
		$theme_settings_sidebar_location_secondary = isset( $values['sidebar_location_secondary'] ) ? esc_attr( $values['sidebar_location_secondary'] ) : '';
	?>
    <!-- Layout option -->
    <p><strong><?php echo esc_html__( 'Choose Layout', 'photo-perfect-pro' ); ?></strong></p>
    <select name="theme_settings[post_layout]" id="theme_settings_post_layout">
      <option value=""><?php echo esc_html__( 'Default', 'photo-perfect-pro' ); ?></option>
		<?php if ( ! empty( $global_layout_options ) ) :  ?>
        <?php foreach ( $global_layout_options as $key => $val ) :  ?>

          <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $theme_settings_post_layout, $key ); ?> ><?php echo esc_html( $val ); ?></option>

        <?php endforeach ?>
		<?php endif ?>
    </select>
    <!-- Image in single post/page -->
    <p><strong><?php echo esc_html__( 'Choose Image Size in Single Post/Page', 'photo-perfect-pro' ); ?></strong></p>
    <select name="theme_settings[single_image]" id="theme_settings_single_image">
      <option value=""><?php echo esc_html__( 'Default', 'photo-perfect-pro' ); ?></option>
		<?php if ( ! empty( $image_size_options ) ) :  ?>
        <?php foreach ( $image_size_options as $key => $val ) : ?>

          <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $theme_settings_single_image, $key ); ?> ><?php echo esc_html( $val ); ?></option>

        <?php endforeach ?>
		<?php endif ?>
    </select>
    <!-- Image Alignment in single post/page -->
    <p><strong><?php echo esc_html__( 'Alignment of Image in Single Post/Page', 'photo-perfect-pro' ); ?></strong></p>
    <select name="theme_settings[single_image_alignment]" id="theme_settings_single_image_alignment">
      <option value=""><?php echo esc_html__( 'Default', 'photo-perfect-pro' ); ?></option>
		<?php if ( ! empty( $image_alignment_options ) ) :  ?>
        <?php foreach ( $image_alignment_options as $key => $val ) : ?>

          <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $theme_settings_single_image_alignment, $key ); ?> ><?php echo esc_html( $val ); ?></option>

        <?php endforeach ?>
		<?php endif ?>
    </select>
    <!-- Show image as banner -->
    <p><strong><?php echo esc_html__( 'Use Featured Image as Banner', 'photo-perfect-pro' ); ?></strong></p>
    <input type="hidden" value="0" name="theme_settings[show_image_as_banner]" id="theme_settings_show_image_as_banner">
    <input type="checkbox" value="1" name="theme_settings[show_image_as_banner]" id="theme_settings_show_image_as_banner" <?php checked( $theme_settings_show_image_as_banner, 1 ); ?>>&nbsp;<?php echo esc_html__( 'Enable', 'photo-perfect-pro' ); ?>
    <p><em><?php echo esc_html__( 'If enabled, featured image will be displayed as banner image and it will not displayed in the content.', 'photo-perfect-pro' ); ?></em></p>

    <!-- Shrunk header -->
    <p><strong><?php echo esc_html__( 'Shrink Header', 'photo-perfect-pro' ); ?></strong></p>
    <select name="theme_settings[shrunk_header]" id="theme_settings_shrunk_header">
    	<option value="default" <?php selected( $theme_settings_shrunk_header, 'default' ); ?>><?php echo esc_html__( 'Default', 'photo-perfect-pro' ); ?></option>
    	<option value="enable" <?php selected( $theme_settings_shrunk_header, 'enable' ); ?>><?php echo esc_html__( 'Enable', 'photo-perfect-pro' ); ?></option>
    	<option value="disable" <?php selected( $theme_settings_shrunk_header, 'disable' ); ?>><?php echo esc_html__( 'Disable', 'photo-perfect-pro' ); ?></option>
    </select>
    <p><em><?php echo esc_html__( 'Select Enable to shrink header.', 'photo-perfect-pro' ); ?></em></p>

    <!-- Sidebar option -->
    <?php $sidebar_options = photo_perfect_get_sidebar_options(); ?>
    <h4 style="font-size:1.2em;"><?php echo __( 'Sidebar Settings', 'photo-perfect-pro' ); ?></h4>
    <p><strong><?php echo __( 'Primary Location', 'photo-perfect-pro' ); ?></strong></p>
     <select name="theme_settings[sidebar_location_primary]" id="theme_settings_sidebar_location_primary">
       <option value=""><?php echo __( 'Default', 'photo-perfect-pro' ); ?></option>
       <?php if ( ! empty( $sidebar_options ) ): ?>
         <?php foreach ( $sidebar_options as $key => $val ) : ?>
           <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $theme_settings_sidebar_location_primary, $key ); ?> ><?php echo esc_html( $val ); ?></option>
         <?php endforeach ?>
       <?php endif ?>
     </select>
    <p><strong><?php echo __( 'Secondary Location', 'photo-perfect-pro' ); ?></strong></p>
     <select name="theme_settings[sidebar_location_secondary]" id="theme_settings_sidebar_location_secondary">
       <option value=""><?php echo __( 'Default', 'photo-perfect-pro' ); ?></option>
       <?php if ( ! empty( $sidebar_options ) ): ?>
         <?php foreach ( $sidebar_options as $key => $val ) : ?>
           <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $theme_settings_sidebar_location_secondary, $key ); ?> ><?php echo esc_html( $val ); ?></option>
         <?php endforeach ?>
       <?php endif ?>
     </select>


    <?php
	}

endif;



if ( ! function_exists( 'photo_perfect_save_theme_settings_meta' ) ) :

	/**
	 * Save theme settings meta box value.
	 *
	 * @since 1.0.0
	 *
	 * @param int $post_id Post ID.
	 * @param WP_Post $post Post object.
	 */
	function photo_perfect_save_theme_settings_meta( $post_id, $post ) {

		// Verify nonce.
		if (
			! ( isset( $_POST['photo_perfect_theme_settings_meta_box_nonce'] )
			&& wp_verify_nonce( sanitize_key( $_POST['photo_perfect_theme_settings_meta_box_nonce'] ), basename( __FILE__ ) ) )
		) {
			return;
		}

		// Bail if auto save or revision.
		if ( defined( 'DOING_AUTOSAVE' ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
			return;
		}

		// Check permission.
		if ( isset( $_POST['post_type'] ) && 'page' === $_POST['post_type'] ) {
			if ( ! current_user_can( 'edit_page', $post_id ) ) {
				return;
			}
		} else if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		if ( isset( $_POST['theme_settings'] ) && is_array( $_POST['theme_settings'] ) ) {

			$raw_value = wp_unslash( $_POST['theme_settings'] );

			if ( ! array_filter( $raw_value ) ) {

				// No value.
				delete_post_meta( $post_id, 'theme_settings' );

			} else {

				$meta_fields = array(
					'post_layout' => array(
						'type' => 'select',
						),
					'single_image' => array(
						'type' => 'select',
						),
					'single_image_alignment' => array(
						'type' => 'select',
						),
					'show_image_as_banner' => array(
						'type' => 'checkbox',
						),
					'shrunk_header' => array(
						'type' => 'select',
						),
					'sidebar_location_primary' => array(
						'type' => 'select',
						),
					'sidebar_location_secondary' => array(
						'type' => 'select',
						),
					);

				$sanitized_values = array();

				foreach ( $raw_value as $mk => $mv ) {

					if ( isset( $meta_fields[ $mk ]['type'] ) ) {
						switch ( $meta_fields[ $mk ]['type'] ) {
							case 'select':
								$sanitized_values[ $mk ] = sanitize_key( $mv );
								break;
							case 'checkbox':
								$sanitized_values[ $mk ] = absint( $mv ) > 0 ? 1 : 0;
								break;
							default:
								$sanitized_values[ $mk ] = sanitize_text_field( $mv );
								break;
						}
					} // End if.

				}

				update_post_meta( $post_id, 'theme_settings', $sanitized_values );
			}
		} // End if theme settings.

	}

endif;

add_action( 'save_post', 'photo_perfect_save_theme_settings_meta', 10, 2 );
