<?php
/**
 * Theme widgets.
 *
 * @package Photo_Perfect
 */

if ( ! function_exists( 'photo_perfect_load_widgets' ) ) :

	/**
	 * Load widgets.
	 *
	 * @since 1.0.0
	 */
	function photo_perfect_load_widgets() {

		// Register Social widget.
		register_widget( 'Photo_Perfect_Social_Widget' );

		// Register Recent Posts widget.
		register_widget( 'Photo_Perfect_Recent_Posts_Widget' );

		// Register Featured Posts widget.
		register_widget( 'Photo_Perfect_Featured_Posts_Widget' );

	}

endif;

add_action( 'widgets_init', 'photo_perfect_load_widgets' );


if ( ! class_exists( 'Photo_Perfect_Social_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Photo_Perfect_Social_Widget extends WP_Widget {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'photo_perfect_widget_social',
				'description'                 => esc_html__( 'Social Icons Widget', 'photo-perfect-pro' ),
				'customize_selective_refresh' => true,
			);
			parent::__construct( 'photo-perfect-social', esc_html__( 'PP: Social', 'photo-perfect-pro' ), $opts );
		}

		/**
		 * Echo the widget content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments including before_title, after_title,
		 *                        before_widget, and after_widget.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		function widget( $args, $instance ) {

			$title = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );

			echo $args['before_widget'];

			// Render title.
			if ( ! empty( $title ) ) {
				echo $args['before_title'] . $title . $args['after_title'];
			}

			if ( has_nav_menu( 'social' ) ) {
				wp_nav_menu( array(
					'theme_location' => 'social',
					'container'      => false,
					'depth'          => 1,
					'link_before'    => '<span class="screen-reader-text">',
					'link_after'     => '</span>',
					'item_spacing'   => 'discard',
				) );
			}

			echo $args['after_widget'];

		}

		/**
		 * Update widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $new_instance New settings for this instance as input by the user via
		 *                            {@see WP_Widget::form()}.
		 * @param array $old_instance Old settings for this instance.
		 * @return array Settings to save or bool false to cancel saving.
		 */
		function update( $new_instance, $old_instance ) {
			$instance = $old_instance;

			$instance['title'] = sanitize_text_field( $new_instance['title'] );

			return $instance;
		}

		/**
		 * Output the settings update form.
		 *
		 * @since 1.0.0
		 *
		 * @param array $instance Current settings.
		 */
		function form( $instance ) {

			// Defaults.
			$instance = wp_parse_args( (array) $instance, array(
				'title' => '',
			) );
			$title = esc_attr( $instance['title'] );

			// Fetch navigation.
			$nav_menu_locations = get_nav_menu_locations();
			$is_menu_set = false;
			if ( isset( $nav_menu_locations['social'] ) && absint( $nav_menu_locations['social'] ) > 0 ) {
				$is_menu_set = true;
			}
			?>
		  <p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'photo-perfect-pro' ); ?></label>
        <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
		  </p>

		<?php if ( true !== $is_menu_set ) :  ?>
        <p>
			<?php echo esc_html__( 'Social menu is not set. Please create menu and assign it to Social', 'photo-perfect-pro' ); ?>
        </p>
        <?php endif ?>
		<?php
		}
	}

endif;

if ( ! class_exists( 'Photo_Perfect_Recent_Posts_Widget' ) ) :

	/**
	 * Recent Posts widget Class.
	 *
	 * @since 1.0.0
	 */
	class Photo_Perfect_Recent_Posts_Widget extends WP_Widget {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'photo_perfect_widget_recent_posts',
				'description'                 => esc_html__( 'Recent Posts Widget', 'photo-perfect-pro' ),
				'customize_selective_refresh' => true,
			);
			parent::__construct( 'photo-perfect-recent-posts', esc_html__( 'PP: Recent Posts', 'photo-perfect-pro' ), $opts );
		}

		/**
		 * Echo the widget content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments including before_title, after_title,
		 *                        before_widget, and after_widget.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		function widget( $args, $instance ) {

			$title         = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );
			$post_category = ! empty( $instance['post_category'] ) ? $instance['post_category'] : 0;
			$post_column   = ! empty( $instance['post_column'] ) ? $instance['post_column'] : 2;
			$post_number   = ! empty( $instance['post_number'] ) ? $instance['post_number'] : 6;
			$post_order_by = ! empty( $instance['post_order_by'] ) ? $instance['post_order_by'] : 'date';
			$post_order    = ! empty( $instance['post_order'] ) ? $instance['post_order'] : 'desc';

			// Validating post order.
			if ( in_array( $post_order, array( 'asc', 'desc' ) ) ) {
				$post_order = strtoupper( $post_order );
			} else {
				$post_order = 'DESC';
			}
			// Validating Order By.
			switch ( $post_order_by ) {
				case 'date':
					$post_order_by = 'date';
			    break;
				case 'title':
					$post_order_by = 'title';
			    break;
				case 'comment-count':
					$post_order_by = 'comment_count';
			    break;
				case 'random':
					$post_order_by = 'rand';
			    break;
				case 'menu-order':
					$post_order_by = 'menu_order';
			    break;
				default:
					$post_order_by = 'date';
			    break;
			}

			echo $args['before_widget'];

			// Render title.
			if ( ! empty( $title ) ) {
				echo $args['before_title'] . $title . $args['after_title'];
			}

			$qargs = array(
			  'posts_per_page' => $post_number,
			  'no_found_rows'  => true,
			  'orderby'        => $post_order_by,
			  'order'          => $post_order,
			  'meta_query'     => array(
			      array( 'key' => '_thumbnail_id' ), // Show only posts with featured images.
			    ),
			  );
			if ( absint( $post_category ) > 0  ) {
				$qargs['cat'] = $post_category;
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>
				<?php global $post; ?>
				<div class="recent-posts-widget-wrap <?php echo 'column-' . esc_attr( $post_column );  ?>">
					<?php foreach ( $all_posts as $key => $post ) : ?>
						<?php setup_postdata( $post ); ?>
						<div class="recent-posts-item">
							<a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>"><?php the_post_thumbnail( 'thumbnail' ); ?></a>
						</div><!-- .recent-posts-item -->
					<?php endforeach; ?>

				</div><!-- .recent-posts-widget-wrap -->
				<?php wp_reset_postdata(); ?>
			<?php endif; ?>

			<?php
			echo $args['after_widget'];

		}

		/**
		 * Update widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $new_instance New settings for this instance as input by the user via
		 *                            {@see WP_Widget::form()}.
		 * @param array $old_instance Old settings for this instance.
		 * @return array Settings to save or bool false to cancel saving.
		 */
		function update( $new_instance, $old_instance ) {

			$instance = $old_instance;
			$instance['title']         = sanitize_text_field( $new_instance['title'] );
			$instance['post_category'] = absint( $new_instance['post_category'] );
			$instance['post_number']   = absint( $new_instance['post_number'] );
			$instance['post_column']   = absint( $new_instance['post_column'] );
			$instance['post_order_by'] = sanitize_key( $new_instance['post_order_by'] );
			$instance['post_order']    = sanitize_key( $new_instance['post_order'] );
			return $instance;

		}

		/**
		 * Output the settings update form.
		 *
		 * @since 1.0.0
		 *
		 * @param array $instance Current settings.
		 */
		function form( $instance ) {

			// Defaults.
			$instance = wp_parse_args( (array) $instance, array(
				'title'         => '',
				'post_category' => '',
				'post_column'   => 2,
				'post_number'   => 6,
				'post_order_by' => 'date',
				'post_order'    => 'desc',
			) );
			$title         = esc_attr( $instance['title'] );
			$post_category = absint( $instance['post_category'] );
			$post_column   = absint( $instance['post_column'] );
			$post_number   = absint( $instance['post_number'] );
			$post_order_by = esc_attr( $instance['post_order_by'] );
			$post_order    = esc_attr( $instance['post_order'] );
			?>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'photo-perfect-pro' ); ?></label>
        <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" /></p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_category' ) ); ?>"><?php esc_html_e( 'Category:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $cat_args = array(
					  'orderby'         => 'name',
					  'hide_empty'      => 0,
					  'taxonomy'        => 'category',
					  'name'            => $this->get_field_name( 'post_category' ),
					  'id'              => $this->get_field_id( 'post_category' ),
					  'selected'        => $post_category,
					  'show_option_all' => __( 'All Categories','photo-perfect-pro' ),
					);
				  wp_dropdown_categories( $cat_args );
				?>
      </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_column' ) ); ?>"><?php esc_html_e( 'Number of Columns:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $this->dropdown_post_columns( array(
					  'id'       => $this->get_field_id( 'post_column' ),
					  'name'     => $this->get_field_name( 'post_column' ),
					  'selected' => $post_column,
					  )
				  );
				?>
		  </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_number' ) ); ?>"><?php esc_html_e( 'Number of Posts:', 'photo-perfect-pro' ); ?></label>
        <input id="<?php echo esc_attr( $this->get_field_id( 'post_number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'post_number' ) ); ?>" type="number" value="<?php echo esc_attr( $post_number ); ?>" min="1" style="max-width:55px;"/></p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_order_by' ) ); ?>"><?php esc_html_e( 'Order By:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $this->dropdown_post_order_by( array(
					  'id'       => $this->get_field_id( 'post_order_by' ),
					  'name'     => $this->get_field_name( 'post_order_by' ),
					  'selected' => $post_order_by,
					  )
				  );
				?>
		  </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_order' ) ); ?>"><?php esc_html_e( 'Order:', 'photo-perfect-pro' ); ?></label>
        <select id="<?php echo esc_attr( $this->get_field_id( 'post_order' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'post_order' ) ); ?>">
          <option value="asc" <?php selected( $post_order, 'asc' ) ?>><?php esc_html_e( 'Ascending', 'photo-perfect-pro' ) ?></option>
          <option value="desc" <?php selected( $post_order, 'desc' ) ?>><?php esc_html_e( 'Descending', 'photo-perfect-pro' ) ?></option>
        </select></p>
		<?php
		}

		/**
		 * Render post columns dropdown.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args Arguments.
		 */
		function dropdown_post_columns( $args ) {
			$defaults = array(
		    'id'       => '',
		    'name'     => '',
		    'selected' => 0,
		    'echo'     => 1,
			);

			$r = wp_parse_args( $args, $defaults );
			$output = '';

			$choices = array(
		    '1' => 1,
		    '2' => 2,
		    '3' => 3,
		    '4' => 4,
			);

			if ( ! empty( $choices ) ) {

				$output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
				foreach ( $choices as $key => $choice ) {
					$output .= '<option value="' . esc_attr( $key ) . '" ';
					$output .= selected( $r['selected'], $key, false );
					$output .= '>' . esc_html( $choice ) . '</option>\n';
				}
				$output .= "</select>\n";
			}

			if ( $r['echo'] ) {
				echo $output;
			}
			return $output;

		}

		/**
		 * Render post order by dropdown.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args Arguments.
		 */
		function dropdown_post_order_by( $args ) {
			$defaults = array(
		    'id'       => '',
		    'name'     => '',
		    'selected' => 0,
		    'echo'     => 1,
			);

			$r = wp_parse_args( $args, $defaults );
			$output = '';

			$choices = array(
		    'date'          => __( 'Date','photo-perfect-pro' ),
		    'title'         => __( 'Title','photo-perfect-pro' ),
		    'comment-count' => __( 'Comment Count','photo-perfect-pro' ),
		    'menu-order'    => __( 'Menu Order','photo-perfect-pro' ),
		    'random'        => __( 'Random','photo-perfect-pro' ),
			);

			if ( ! empty( $choices ) ) {

				$output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
				foreach ( $choices as $key => $choice ) {
					$output .= '<option value="' . esc_attr( $key ) . '" ';
					$output .= selected( $r['selected'], $key, false );
					$output .= '>' . esc_html( $choice ) . '</option>\n';
				}
				$output .= "</select>\n";
			}

			if ( $r['echo'] ) {
				echo $output;
			}
			return $output;

		}
	}

endif;

if ( ! class_exists( 'Photo_Perfect_Featured_Posts_Widget' ) ) :

	/**
	 * Featured Posts widget Class.
	 *
	 * @since 1.0.0
	 */
	class Photo_Perfect_Featured_Posts_Widget extends WP_Widget {

		/**
		 * Constructor.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'photo_perfect_widget_featured_posts',
				'description'                 => esc_html__( 'Featured Posts Widget', 'photo-perfect-pro' ),
				'customize_selective_refresh' => true,
			);
			parent::__construct( 'photo-perfect-featured-posts', esc_html__( 'PP: Featured Posts', 'photo-perfect-pro' ), $opts );
		}

		/**
		 * Echo the widget content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments including before_title, after_title,
		 *                        before_widget, and after_widget.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		function widget( $args, $instance ) {

			$title          = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );
			$post_category  = ! empty( $instance['post_category'] ) ? $instance['post_category'] : 0;
			$post_column    = ! empty( $instance['post_column'] ) ? $instance['post_column'] : 4;
			$post_number    = ! empty( $instance['post_number'] ) ? $instance['post_number'] : 4;
			$featured_image = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'medium';
			$post_order_by  = ! empty( $instance['post_order_by'] ) ? $instance['post_order_by'] : 'date';
			$post_order     = ! empty( $instance['post_order'] ) ? $instance['post_order'] : 'desc';

			// Validating post order.
			if ( in_array( $post_order, array( 'asc', 'desc' ) ) ) {
				$post_order = strtoupper( $post_order );
			} else {
				$post_order = 'DESC';
			}
			// Validating Order By.
			switch ( $post_order_by ) {
				case 'date':
					$post_order_by = 'date';
			    break;
				case 'title':
					$post_order_by = 'title';
			    break;
				case 'comment-count':
					$post_order_by = 'comment_count';
			    break;
				case 'random':
					$post_order_by = 'rand';
			    break;
				case 'menu-order':
					$post_order_by = 'menu_order';
			    break;
				default:
					$post_order_by = 'date';
			    break;
			}

			echo $args['before_widget'];

			// Render title.
			if ( ! empty( $title ) ) {
				echo $args['before_title'] . $title . $args['after_title'];
			}

			$qargs = array(
			  'posts_per_page' => $post_number,
			  'no_found_rows'  => true,
			  'orderby'        => $post_order_by,
			  'order'          => $post_order,
			  'meta_query'     => array(
			      array( 'key' => '_thumbnail_id' ), // Show only posts with featured images.
			    ),
			  );
			if ( absint( $post_category ) > 0  ) {
				$qargs['cat'] = $post_category;
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>
				<?php global $post; ?>
				<div class="featured-posts-widget-wrap <?php echo 'column-' . esc_attr( $post_column );  ?>">
					<?php foreach ( $all_posts as $key => $post ) : ?>
						<?php setup_postdata( $post ); ?>
						<div class="featured-posts-item">
							<?php
							$featured_image_full_url = '';
							if ( has_post_thumbnail() ) {
								$featured_image_full_url = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
							}
							?>
							<a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>"><?php the_post_thumbnail( $featured_image ); ?></a>

							<div class="post-content">
								<a href="<?php echo esc_url( $featured_image_full_url ); ?>" class="popup-link"><i class="fa fa-eye"></i></a>
								<a href="<?php the_permalink(); ?>"><i class="fa fa-link"></i></a>
								<h2><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h2>
							</div>

						</div><!-- .featured-posts-item -->
					<?php endforeach; ?>

				</div><!-- .featured-posts-widget-wrap -->
				<?php wp_reset_postdata(); ?>
			<?php endif; ?>

			<?php
			echo $args['after_widget'];

		}

		/**
		 * Update widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $new_instance New settings for this instance as input by the user via
		 *                            {@see WP_Widget::form()}.
		 * @param array $old_instance Old settings for this instance.
		 * @return array Settings to save or bool false to cancel saving.
		 */
		function update( $new_instance, $old_instance ) {

			$instance = $old_instance;
			$instance['title']          = sanitize_text_field( $new_instance['title'] );
			$instance['post_category']  = absint( $new_instance['post_category'] );
			$instance['post_number']    = absint( $new_instance['post_number'] );
			$instance['post_column']    = absint( $new_instance['post_column'] );
			$instance['featured_image'] = esc_attr( $new_instance['featured_image'] );
			$instance['post_order_by']  = sanitize_key( $new_instance['post_order_by'] );
			$instance['post_order']     = sanitize_key( $new_instance['post_order'] );
			return $instance;

		}

		/**
		 * Output the settings update form.
		 *
		 * @since 1.0.0
		 *
		 * @param array $instance Current settings.
		 */
		function form( $instance ) {

			// Defaults.
			$instance = wp_parse_args( (array) $instance, array(
				'title'          => '',
				'post_category'  => '',
				'post_column'    => 4,
				'post_number'    => 4,
				'featured_image' => 'medium',
				'post_order_by'  => 'date',
				'post_order'     => 'desc',
			) );
			$title          = esc_attr( $instance['title'] );
			$post_category  = absint( $instance['post_category'] );
			$post_column    = absint( $instance['post_column'] );
			$post_number    = absint( $instance['post_number'] );
			$featured_image = esc_attr( $instance['featured_image'] );
			$post_order_by  = esc_attr( $instance['post_order_by'] );
			$post_order     = esc_attr( $instance['post_order'] );
			?>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'photo-perfect-pro' ); ?></label>
        <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" /></p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_category' ) ); ?>"><?php esc_html_e( 'Category:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $cat_args = array(
					  'orderby'         => 'name',
					  'hide_empty'      => 0,
					  'taxonomy'        => 'category',
					  'name'            => $this->get_field_name( 'post_category' ),
					  'id'              => $this->get_field_id( 'post_category' ),
					  'selected'        => $post_category,
					  'show_option_all' => __( 'All Categories','photo-perfect-pro' ),
					);
				  wp_dropdown_categories( $cat_args );
				?>
      </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_column' ) ); ?>"><?php esc_html_e( 'Number of Columns:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $this->dropdown_post_columns( array(
					  'id'       => $this->get_field_id( 'post_column' ),
					  'name'     => $this->get_field_name( 'post_column' ),
					  'selected' => $post_column,
					  )
				  );
				?>
		  </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_number' ) ); ?>"><?php esc_html_e( 'Number of Posts:', 'photo-perfect-pro' ); ?></label>
        <input id="<?php echo esc_attr( $this->get_field_id( 'post_number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'post_number' ) ); ?>" type="number" value="<?php echo esc_attr( $post_number ); ?>" min="1" style="max-width:55px;"/></p>
        <p>
        	<label for="<?php echo esc_attr( $this->get_field_id( 'featured_image' ) ); ?>"><?php esc_html_e( 'Image Size:', 'photo-perfect-pro' ); ?></label>
        	<?php
        	  $this->dropdown_image_sizes( array(
        	    'id'       => $this->get_field_id( 'featured_image' ),
        	    'name'     => $this->get_field_name( 'featured_image' ),
        	    'selected' => $featured_image,
        	    )
        	  );
        	?>
        </p>

		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_order_by' ) ); ?>"><?php esc_html_e( 'Order By:', 'photo-perfect-pro' ); ?></label>
        <?php
				  $this->dropdown_post_order_by( array(
					  'id'       => $this->get_field_id( 'post_order_by' ),
					  'name'     => $this->get_field_name( 'post_order_by' ),
					  'selected' => $post_order_by,
					  )
				  );
				?>
		  </p>
		  <p><label for="<?php echo esc_attr( $this->get_field_id( 'post_order' ) ); ?>"><?php esc_html_e( 'Order:', 'photo-perfect-pro' ); ?></label>
        <select id="<?php echo esc_attr( $this->get_field_id( 'post_order' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'post_order' ) ); ?>">
          <option value="asc" <?php selected( $post_order, 'asc' ) ?>><?php esc_html_e( 'Ascending', 'photo-perfect-pro' ) ?></option>
          <option value="desc" <?php selected( $post_order, 'desc' ) ?>><?php esc_html_e( 'Descending', 'photo-perfect-pro' ) ?></option>
        </select></p>
		<?php
		}

		/**
		 * Render image size dropdown.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args Arguments.
		 * @return string Rendered content.
		 */
		function dropdown_image_sizes( $args ){
			$defaults = array(
				'id'       => '',
				'name'     => '',
				'selected' => 0,
				'echo'     => 1,
			);

			$r = wp_parse_args( $args, $defaults );
			$output = '';

			$choices = photo_perfect_get_image_sizes_options( false );

			if ( ! empty( $choices ) ) {

				$output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
				foreach ( $choices as $key => $choice ) {
					$output .= '<option value="' . esc_attr( $key ) . '" ';
					$output .= selected( $r['selected'], $key, false );
					$output .= '>' . esc_html( $choice ) . '</option>\n';
				}
				$output .= "</select>\n";
			}

			if ( $r['echo'] ) {
				echo $output;
			}
			return $output;

		}

		/**
		 * Render post columns dropdown.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args Arguments.
		 */
		function dropdown_post_columns( $args ) {
			$defaults = array(
		    'id'       => '',
		    'name'     => '',
		    'selected' => 0,
		    'echo'     => 1,
			);

			$r = wp_parse_args( $args, $defaults );
			$output = '';

			$choices = array(
		    '1' => 1,
		    '2' => 2,
		    '3' => 3,
		    '4' => 4,
			);

			if ( ! empty( $choices ) ) {

				$output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
				foreach ( $choices as $key => $choice ) {
					$output .= '<option value="' . esc_attr( $key ) . '" ';
					$output .= selected( $r['selected'], $key, false );
					$output .= '>' . esc_html( $choice ) . '</option>\n';
				}
				$output .= "</select>\n";
			}

			if ( $r['echo'] ) {
				echo $output;
			}
			return $output;

		}

		/**
		 * Render post order by dropdown.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args Arguments.
		 */
		function dropdown_post_order_by( $args ) {
			$defaults = array(
				'id'       => '',
				'name'     => '',
				'selected' => 0,
				'echo'     => 1,
			);

			$r = wp_parse_args( $args, $defaults );
			$output = '';

			$choices = array(
				'date'          => __( 'Date','photo-perfect-pro' ),
				'title'         => __( 'Title','photo-perfect-pro' ),
				'comment-count' => __( 'Comment Count','photo-perfect-pro' ),
				'menu-order'    => __( 'Menu Order','photo-perfect-pro' ),
				'random'        => __( 'Random','photo-perfect-pro' ),
			);

			if ( ! empty( $choices ) ) {

				$output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
				foreach ( $choices as $key => $choice ) {
					$output .= '<option value="' . esc_attr( $key ) . '" ';
					$output .= selected( $r['selected'], $key, false );
					$output .= '>' . esc_html( $choice ) . '</option>\n';
				}
				$output .= "</select>\n";
			}

			if ( $r['echo'] ) {
				echo $output;
			}
			return $output;

		}
	}

endif;
