<?php
/**
 * Theme widgets.
 *
 * @package Bizroot
 */

// Load widget base.
require_once get_template_directory() . '/lib/widget-base/class-widget-base.php';

if ( ! function_exists( 'bizroot_load_widgets' ) ) :

	/**
	 * Load widgets.
	 *
	 * @since 1.0.0
	 */
	function bizroot_load_widgets() {

		// Social widget.
		register_widget( 'Bizroot_Social_Widget' );

		// Featured Page widget.
		register_widget( 'Bizroot_Featured_Page_Widget' );

		// Latest News widget.
		register_widget( 'Bizroot_Latest_News_Widget' );

		// Call To Action widget.
		register_widget( 'Bizroot_Call_To_Action_Widget' );

		// Services widget.
		register_widget( 'Bizroot_Services_Widget' );

		// Contact widget.
		register_widget( 'Bizroot_Contact_Widget' );

		// Testimonial slider widget.
		register_widget( 'Bizroot_Testimonial_Slider_Widget' );

		// Portfolio slider widget.
		register_widget( 'Bizroot_Portfolio_Slider_Widget' );

		// Clients widget.
		register_widget( 'Bizroot_Clients_Widget' );

		// Pricing Table widget.
		register_widget( 'Bizroot_Pricing_Table_Widget' );

		// Recent Posts widget.
		register_widget( 'Bizroot_Recent_Posts_Widget' );

		// Teams widget.
		register_widget( 'Bizroot_Teams_Widget' );

		// Skills widget.
		register_widget( 'Bizroot_Skills_Widget' );


	}

endif;

add_action( 'widgets_init', 'bizroot_load_widgets' );

if ( ! class_exists( 'Bizroot_Social_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Social_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_social',
				'description'                 => __( 'Displays social icons.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				);

			if ( false === has_nav_menu( 'social' ) ) {
				$fields['message'] = array(
					'label' => __( 'Social menu is not set. Please create menu and assign it to Social Menu.', 'bizroot-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					);
			}

			parent::__construct( 'bizroot-social', __( 'Bizroot: Social', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}

			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			if ( has_nav_menu( 'social' ) ) {
				wp_nav_menu( array(
					'theme_location' => 'social',
					'container'      => false,
					'depth'          => 1,
					'link_before'    => '<span class="screen-reader-text">',
					'link_after'     => '</span>',
				) );
			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Bizroot_Featured_Page_Widget' ) ) :

	/**
	 * Featured page widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Featured_Page_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_featured_page',
				'description'                 => __( 'Displays single featured Page or Post.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'use_page_title' => array(
					'label'   => __( 'Use Page/Post Title as Widget Title', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'featured_page' => array(
					'label'            => __( 'Select Page:', 'bizroot-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'bizroot-pro' ),
					),
				'id_message' => array(
					'label'            => '<strong>' . _x( 'OR', 'Featured Page Widget', 'bizroot-pro' ) . '</strong>',
					'type'             => 'message',
					),
				'featured_post' => array(
					'label'             => __( 'Post ID:', 'bizroot-pro' ),
					'placeholder'       => __( 'Eg: 1234', 'bizroot-pro' ),
					'type'              => 'text',
					'sanitize_callback' => 'bizroot_widget_sanitize_post_id',
					),
				'content_type' => array(
					'label'   => __( 'Show Content:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 'full',
					'options' => array(
						'excerpt' => __( 'Excerpt', 'bizroot-pro' ),
						'full'    => __( 'Full', 'bizroot-pro' ),
						),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'bizroot-pro' ),
					'description' => __( 'Applies when Excerpt is selected in Content option.', 'bizroot-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 40,
					'min'         => 1,
					'max'         => 400,
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'bizroot-pro' ),
					'type'    => 'select',
					'options' => bizroot_get_image_sizes_options(),
					),
				'featured_image_alignment' => array(
					'label'   => __( 'Image Alignment:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 'center',
					'options' => bizroot_get_image_alignment_options(),
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-featured-page', __( 'Bizroot: Featured Page', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			// ID validation.
			$our_post_object = null;
			$our_id = '';
			if ( absint( $params['featured_post'] ) > 0 ) {
				$our_id = absint( $params['featured_post'] );
			}
			if ( absint( $params['featured_page'] ) > 0 ) {
				$our_id = absint( $params['featured_page'] );
			}
			if ( absint( $our_id ) > 0 ) {
				$raw_object = get_post( $our_id );
				if ( ! in_array( $raw_object->post_type, array( 'attachment', 'nav_menu_item', 'revision' ) ) ) {
					$our_post_object = $raw_object;
				}
			}
			if ( ! $our_post_object ) {
				// No valid object; bail now!
				return;
			}

			echo $args['before_widget'];

			global $post;
			// Setup global post.
			$post = $our_post_object;
			setup_postdata( $post );

			// Override title if checkbox is selected.
			if ( true === $params['use_page_title'] ) {
				$params['title'] = get_the_title( $post );
			}

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			?>
			<div class="featured-page-widget entry-content">
				<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
					<?php the_post_thumbnail( esc_attr( $params['featured_image'] ), array( 'class' => 'align' . esc_attr( $params['featured_image_alignment'] ) ) ); ?>
				<?php endif; ?>
				<?php if ( 'excerpt' === $params['content_type'] ) : ?>
					<?php
						$excerpt = bizroot_the_excerpt( absint( $params['excerpt_length'] ) );
						echo wp_kses_post( wpautop( $excerpt ) );
						?>
				<?php else : ?>
					<?php the_content(); ?>
				<?php endif; ?>

			</div><!-- .featured-page-widget -->
			<?php
			// Reset.
			wp_reset_postdata();

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Bizroot_Latest_News_Widget' ) ) :

	/**
	 * Latest news widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Latest_News_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {
			$opts = array(
				'classname'                   => 'bizroot_widget_latest_news',
				'description'                 => __( 'Displays latest posts in grid.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'bizroot-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'bizroot-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'bizroot-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => __( 'Number of Columns:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => bizroot_get_numbers_dropdown_options( 1, 4 ),
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 'bizroot-thumb',
					'options' => bizroot_get_image_sizes_options(),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'bizroot-pro' ),
					'description' => __( 'in words', 'bizroot-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'more_text' => array(
					'label'   => __( 'More Text:', 'bizroot-pro' ),
					'type'    => 'text',
					'default' => __( 'Read more', 'bizroot-pro' ),
					),
				'disable_date' => array(
					'label'   => __( 'Disable Date', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'disable_more_text' => array(
					'label'   => __( 'Disable More Text', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-latest-news', __( 'Bizroot: Latest News', 'bizroot-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0  ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) : ?>

				<?php global $post; ?>

				<div class="latest-news-widget latest-news-col-<?php echo esc_attr( $params['post_column'] ); ?>">

					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $key => $post ) : ?>
							<?php setup_postdata( $post ); ?>

							<div class="latest-news-item">

									<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
										<div class="latest-news-thumb">
											<a href="<?php the_permalink(); ?>">
												<?php
												$img_attributes = array( 'class' => 'aligncenter' );
												the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
												?>
											</a>
										</div><!-- .latest-news-thumb -->
									<?php endif; ?>
									<div class="latest-news-text-wrap">

										<div class="latest-news-text-content">
											<h3 class="latest-news-title">
												<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
											</h3><!-- .latest-news-title -->

											<?php if ( false === $params['disable_excerpt'] ) :  ?>
												<div class="latest-news-summary">
												<?php
												$summary = bizroot_the_excerpt( esc_attr( $params['excerpt_length'] ), $post );
												echo wp_kses_post( wpautop( $summary ) );
												?>
												</div><!-- .latest-news-summary -->
											<?php endif; ?>
										</div><!-- .latest-news-text-content -->

										<?php if ( false === $params['disable_date'] || false === $params['disable_more_text'] ) : ?>
											<div class="latest-news-meta">
												<ul>
													<?php if ( false === $params['disable_date'] ) :  ?>
														<li><span class="latest-news-date"><?php the_time( 'j M Y' ); ?></span></li>
													<?php endif; ?>
													<?php if ( false === $params['disable_more_text'] ) :  ?>
														<li><a href="<?php the_permalink(); ?>" class="custom-button"><?php echo esc_html( $params['more_text'] ); ?><span class="screen-reader-text">"<?php the_title(); ?>"</span>
														</a></li>
													<?php endif; ?>
												</ul>
											</div><!-- .latest-news-meta -->
										<?php endif; ?>

									</div><!-- .latest-news-text-wrap -->

							</div><!-- .latest-news-item -->

						<?php endforeach; ?>

					</div><!-- .row -->

				</div><!-- .latest-news-widget -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Bizroot_Call_To_Action_Widget' ) ) :

	/**
	 * Call to action widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Call_To_Action_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_call_to_action',
				'description'                 => __( 'Call To Action Widget.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'description' => array(
					'label' => __( 'Description:', 'bizroot-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
					),
				'primary_button_text' => array(
					'label'   => __( 'Primary Button Text:', 'bizroot-pro' ),
					'default' => __( 'Learn more', 'bizroot-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'primary_button_url' => array(
					'label' => __( 'Primary Button URL:', 'bizroot-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				'secondary_button_text' => array(
					'label'   => __( 'Secondary Button Text:', 'bizroot-pro' ),
					'default' => '',
					'type'    => 'text',
					'class'   => 'widefat',
					),
				'secondary_button_url' => array(
					'label' => __( 'Secondary Button URL:', 'bizroot-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-call-to-action', __( 'Bizroot: Call To Action', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			?>
			<div class="call-to-action-content">
				<?php if ( ! empty( $params['description'] ) ) : ?>
				    <div class="call-to-action-description">
				        <?php echo wp_kses_post( wpautop( $params['description'] ) ); ?>
				    </div><!-- .call-to-action-description -->
				<?php endif; ?>
				<?php if ( ! empty( $params['primary_button_text'] ) || ! empty( $params['secondary_button_text'] ) ) : ?>
					<div class="call-to-action-buttons">
						<?php if ( ! empty( $params['primary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['primary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-primary"><?php echo esc_html( $params['primary_button_text'] ); ?></a>
						<?php endif; ?>
						<?php if ( ! empty( $params['secondary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['secondary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-secondary"><?php echo esc_html( $params['secondary_button_text'] ); ?></a>
						<?php endif; ?>
					</div><!-- .call-to-action-buttons -->
				<?php endif; ?>
			</div><!-- .call-to-action-content -->
			<?php

			echo $args['after_widget'];

		}
	}
endif;


if ( ! class_exists( 'Bizroot_Services_Widget' ) ) :

	/**
	 * Social widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Services_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_services',
				'description'                 => __( 'Show your services with icon and read more link.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'bizroot-pro' ),
					'description' => __( 'in words', 'bizroot-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'more_text' => array(
					'label'   => __( 'Read More Text:', 'bizroot-pro' ),
					'type'    => 'text',
					'default' => __( 'Read more', 'bizroot-pro' ),
					),
				'disable_more_text' => array(
					'label'   => __( 'Disable Read More', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			for( $i = 1; $i <= 4; $i++ ) {
				$fields[ 'block_heading_' . $i ] = array(
					'label' => __( 'Block', 'bizroot-pro' ) . ' #' . $i,
					'type'  => 'heading',
					'class' => 'widefat',
					);
				$fields[ 'block_page_' . $i ] = array(
					'label'            => __( 'Select Page:', 'bizroot-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => __( '&mdash; Select &mdash;', 'bizroot-pro' ),
					);
				$fields[ 'block_icon_' . $i ] = array(
					'label'       => __( 'Icon:', 'bizroot-pro' ),
					'description' => __( 'Eg: fa-cogs', 'bizroot-pro' ),
					'type'        => 'text',
					'default'     => 'fa-cogs',
					);
			}

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-services', __( 'Bizroot: Services', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$service_arr = array();
			for ( $i = 0; $i < 4 ; $i++ ) {
				$block = ( $i + 1 );
				$service_arr[ $i ] = array(
					'page' => $params[ 'block_page_' . $block ],
					'icon' => $params[ 'block_icon_' . $block ],
					);
			}
			$refined_arr = array();
			if ( ! empty( $service_arr ) ) {
				foreach ( $service_arr as $item ) {
					if ( ! empty( $item['page'] ) ) {
						$refined_arr[ $item['page'] ] = $item;
					}
				}
			}

			if ( ! empty( $refined_arr ) ) {
				$this->render_widget_content( $refined_arr, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Render services content.
		 *
		 * @since 1.0.0
		 *
		 * @param array $service_arr Services array.
		 * @param array $params      Parameters array.
		 */
		function render_widget_content( $service_arr, $params ) {

			$column = count( $service_arr );

			$page_ids = array_keys( $service_arr );

			$qargs = array(
				'post__in'      => $page_ids,
				'post_type'     => 'page',
				'orderby'       => 'post__in',
				'no_found_rows' => true,
				);

			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>

				<?php global $post; ?>

				<div class="service-block-list service-col-<?php echo esc_attr( $column ); ?>">
					<div class="inner-wrapper">

						<?php foreach ( $all_posts as $post ) :  ?>
							<?php setup_postdata( $post ); ?>
							<div class="service-block-item">
								<div class="service-block-inner">
									<?php if ( isset( $service_arr[ $post->ID ]['icon'] ) && ! empty( $service_arr[ $post->ID ]['icon'] ) ) : ?>
										<a href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>"><i class="<?php echo 'fa ' . esc_attr( $service_arr[ $post->ID ]['icon'] ); ?>"></i></a>
									<?php endif; ?>
									<div class="service-block-inner-content">
										<h3 class="service-item-title">
											<a href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>">
												<?php echo get_the_title( $post->ID ); ?>
											</a>
										</h3>
										<?php if ( true !== $params['disable_excerpt'] ) :  ?>
											<div class="service-block-item-excerpt">
												<?php
												$excerpt = bizroot_the_excerpt( $params['excerpt_length'], $post );
												echo wp_kses_post( wpautop( $excerpt ) );
												?>
											</div><!-- .service-block-item-excerpt -->
										<?php endif; ?>

										<?php if ( true !== $params['disable_more_text'] ) :  ?>
											<a href="<?php echo esc_url( get_permalink( $post -> ID ) ); ?>" class="custom-button"><?php echo esc_html( $params['more_text'] ); ?></a>
										<?php endif; ?>
									</div><!-- .service-block-inner-content -->
								</div><!-- .service-block-inner -->
							</div><!-- .service-block-item -->
						<?php endforeach; ?>

					</div><!-- .inner-wrapper -->

				</div><!-- .service-block-list -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
		}

	}
endif;

if ( ! class_exists( 'Bizroot_Contact_Widget' ) ) :

	/**
	 * Contact widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Contact_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_contact',
				'description'                 => __( 'Displays contact form.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'form_id' => array(
					'label'   => __( 'Select Contact Form:', 'bizroot-pro' ),
					'type'    => 'select',
					'options' => bizroot_get_contact_form_options(),
					),
				'form_message' => array(
					'label' => _x( 'OR', 'Bizroot Contact', 'bizroot-pro' ),
					'type'  => 'heading',
					),
				'form_shortcode' => array(
					'label' => __( 'Enter Form Shortcode:', 'bizroot-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-contact', __( 'Bizroot: Contact', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$shortcode_text = null;

			if ( ! empty( $params['form_id'] ) && absint( $params['form_id'] ) > 0 && defined( 'WPCF7_VERSION' ) ) {
				$shortcode_text = '[contact-form-7 id="' . absint( $params['form_id'] ) . '"]';
			}

			if ( empty( $shortcode_text ) ) {
				$shortcode_text = wp_kses_data( $params['form_shortcode'] );
			}

			if ( ! empty( $shortcode_text ) ) {
				echo do_shortcode( $shortcode_text );
			}

			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Bizroot_Testimonial_Slider_Widget' ) ) :

	/**
	 * Testimonial Slider widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Testimonial_Slider_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'bizroot_widget_testimonial_slider',
				'description' => __( 'Displays Testimonials as a Slider.', 'bizroot-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'bizroot-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'bizroot-pro' ),
					),
				'category_message' => array(
					'label' => __( 'OR', 'bizroot-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					),
				'use_jetpack' => array(
					'label'   => __( 'Use Jetpack Testimonials', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'bizroot-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'bizroot-pro' ),
					'description' => __( 'in words', 'bizroot-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 30,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'background_image' => array(
					'label'   => __( 'Background Image:', 'bizroot-pro' ),
					'type'    => 'image',
					'default' => '',
					),
				'enable_background_overlay' => array(
					'label'   => __( 'Enable Background Overlay', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				'slider_heading' => array(
					'label'   => __( 'SLIDER OPTIONS', 'bizroot-pro' ),
					'type'    => 'heading',
					),
				'transition_delay' => array(
					'label'       => __( 'Transition Delay:', 'bizroot-pro' ),
					'description' => __( 'in seconds', 'bizroot-pro' ),
					'type'        => 'number',
					'default'     => 3,
					'css'         => 'max-width:50px;',
					'min'         => 1,
					'max'         => 10,
					'adjacent'         => true,
					),
				'enable_autoplay' => array(
					'label'   => __( 'Enable Autoplay', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-testimonial-slider', __( 'Bizroot: Testimonial Slider', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			// Add background image.
			if ( ! empty( $params['background_image'] ) ) {
				$background_style = '';
				$background_style .= ' style="background-image:url(' . esc_url( $params['background_image'] ) . ');" ';
				$args['before_widget'] = implode( $background_style . ' ' . 'class="', explode( 'class="', $args['before_widget'], 2 ) );
			}

			// Add overlay class.
			$overlay_class = ( true === $params['enable_background_overlay'] ) ? 'overlay-enabled' : 'overlay-disabled' ;
			$args['before_widget'] = implode( 'class="' . $overlay_class . ' ', explode( 'class="', $args['before_widget'], 2 ) );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$testimonial_posts = $this->get_testimonials( $params );

			if ( ! empty( $testimonial_posts ) ) {
				$this->render_testimonials( $testimonial_posts, $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Return testimonial posts detail.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return array Posts details.
		 */
		function get_testimonials( $params ) {

			$output = array();

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0  ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			// Check if Jetpack enabled.
			if ( true === $params['use_jetpack'] ) {
				$qargs['category']  = null;
				$qargs['post_type'] = 'jetpack-testimonial';
			}

			$all_posts = get_posts( $qargs );

			if ( ! empty( $all_posts ) ) {
				$cnt = 0;
				global $post;
				foreach ( $all_posts as $key => $post ) {

					setup_postdata( $post );

					$item = array();
					$item['title']   = get_the_title( $post->ID );
					$item['excerpt'] = bizroot_the_excerpt( absint( $params['excerpt_length'] ), $post );
					$item['image']   = null;
					$item['thumb']   = null;
					if ( has_post_thumbnail( $post->ID ) ) {
						$image_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'thumbnail' );
						if ( ! empty( $image_detail ) ) {
							$item['image'] = $image_detail;
						}
						$thumb_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'thumbnail' );
						if ( ! empty( $thumb_detail ) ) {
							$item['thumb'] = $thumb_detail;
						}
					}
					if ( empty( $item['thumb'] ) ) {
						$item['thumb'][0] = get_template_directory_uri() . '/images/no-image-testimonial.png';
					}

					$output[$cnt] = $item;
					$cnt++;

				}
				wp_reset_postdata();
			}

			return $output;

		}

		/**
		 * Render testimonial slider.
		 *
		 * @since 1.0.0
		 *
		 * @param array $testimonials Testimonials.
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_testimonials( $testimonials, $params ) {

			$timeout = 0;
			if ( true === $params['enable_autoplay'] ) {
				$timeout = 1000 * absint( $params['transition_delay'] );
			}
			?>
			<div class="cycle-slideshow"
				data-cycle-slides="> article"
				data-cycle-auto-height="container"
				data-cycle-timeout="<?php echo absint( $timeout ); ?>"
				data-pager-template="<span><img src='{{thumb}}'/></span>"
				>
				<div class="cycle-pager"></div>
				<?php foreach ( $testimonials as $testimonial ) : ?>
					<article data-cycle-thumb="<?php echo esc_url( $testimonial['thumb'][0] ); ?>">
						<?php if ( ! empty( $testimonial['image'] ) ) : ?>
							<img src="<?php echo esc_url( $testimonial['image'][0] ); ?>" />
						<?php endif; ?>
						<div class="testimonial-content-area">
						<?php if ( ! empty( $testimonial['excerpt'] ) ) : ?>
							<div class="testimonial-excerpt">
								<?php echo wp_kses_post( wpautop( $testimonial['excerpt'] ) ); ?>
							</div><!-- .testimonial-excerpt -->
						<?php endif; ?>
						<h4><?php echo esc_html( $testimonial['title'] ); ?></h4>
						</div> <!-- .testimonial-content-area -->
					</article>

				<?php endforeach; ?>

			</div><!-- .cycle-slideshow -->
			<?php

		}

	}
endif;

if ( ! class_exists( 'Bizroot_Portfolio_Slider_Widget' ) ) :

	/**
	 * Portfolio widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Portfolio_Slider_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'bizroot_widget_portfolio_slider',
				'description' => __( 'Displays portfolio slider.', 'bizroot-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_number' => array(
					'label'   => __( 'Number of Portfolios:', 'bizroot-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'enable_category_bar' => array(
					'label'   => __( 'Enable Category Bar', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				'transition_delay' => array(
					'label'       => __( 'Transition Delay:', 'bizroot-pro' ),
					'description' => __( 'in seconds', 'bizroot-pro' ),
					'type'        => 'number',
					'default'     => 3,
					'css'         => 'max-width:60px;',
					'min'         => 1,
					'max'         => 10,
					),
				'enable_autoplay' => array(
					'label'   => __( 'Enable Autoplay', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => true,
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			if ( ! ( class_exists( 'Jetpack') && Jetpack::is_module_active( 'custom-content-types' ) ) ) {
				$fields = array();
				$fields['message'] = array(
					'label' => __( 'Please activate Custom Content Type module of Jetpack plugin to use this widget.', 'bizroot-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					);
			}

			parent::__construct( 'bizroot-portfolio-slider', __( 'Bizroot: Portfolio Slider', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			if ( class_exists( 'Jetpack') && Jetpack::is_module_active( 'custom-content-types' ) ) {
				$this->render_portfolio( $params );
			}
			else {
				if ( current_user_can( 'manage_options' ) ) {
					echo __( 'Custom Content Type module of Jetpack plugin is not active.', 'bizroot-pro' );
				}
				else {
					echo __( 'No portfolio found.', 'bizroot-pro' );
				}
			}
			echo $args['after_widget'];

		}

		/**
		 * Render portfolio slider.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_portfolio( $params ) {

			$portfolios = array();

			$qargs = array(
				'post_type'      => 'jetpack-portfolio',
				'posts_per_page' => absint( $params['post_number'] ),
				'meta_query'     => array(
					array( 'key' => '_thumbnail_id' ),
					),
				);

			$all_posts = get_posts( $qargs );
			if ( ! empty( $all_posts ) ) {
				foreach ( $all_posts as $post ) {
					$item            = array();
					$item['id']      = $post->ID;
					$item['title']   = get_the_title( $post->ID );
					$item['url']     = get_permalink( $post->ID );
					$item['excerpt'] = bizroot_the_excerpt( 20, $post );
					$item['image']   = array();
					$image_detail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'bizroot-portfolio-slider' );
					if ( $image_detail ) {
						$item['image'] = $image_detail;
					}
					$portfolios[] = $item;
				}
			}

			if ( empty( $portfolios ) ) {
				return;
			}

			$timeout = 0;
			if ( true === $params['enable_autoplay'] ) {
				$timeout = 1000 * absint( $params['transition_delay'] );
			}

			?>
			<?php if ( true === $params['enable_category_bar'] ) : ?>
				<?php
					$all_categories = array();
					foreach ( $portfolios as $p ) {
						$cats = get_the_terms( $p['id'], 'jetpack-portfolio-type' );
						if ( $cats && ! is_wp_error( $cats ) ) {
							foreach ( $cats as $c ) {
								$all_categories[ $c->term_id ] = $c;
							}
						}
					}
				?>
				<?php if ( ! empty( $all_categories ) ) : ?>
					<div class="portfolio-category-list">
						<ul>
						<?php foreach ( $all_categories as $cat ) : ?>
							<li><a href="<?php echo esc_url( get_term_link( $cat ) ); ?>"><?php echo esc_html( $cat->name ) ?></a></li>
						<?php endforeach; ?>
						</ul>
					</div><!-- .portfolio-category-list -->
				<?php endif; ?>
			<?php endif; ?>

			<div class="cycle-slideshow"
				data-cycle-slides="> article"
				data-cycle-auto-height="container"
				data-cycle-timeout="<?php echo absint( $timeout ); ?>"
				>
				<div class="cycle-prev"><i class="fa fa-angle-left" aria-hidden="true"></i></div>
				<div class="cycle-next"><i class="fa fa-angle-right" aria-hidden="true"></i></div>

				<?php $cnt = 1; ?>
				<?php foreach ( $portfolios as $portfolio ) : ?>
					<?php $class_text = ( 1 === $cnt ) ? 'first' : ''; ?>
					<article class="<?php echo esc_attr( $class_text ); ?>">
						<?php
						$link_open  = '<a href="' . esc_url( $portfolio['url'] ) . '">';
						$link_close = '</a>';
						?>
						<?php if ( ! empty( $portfolio['image'] ) ) : ?>
							<?php echo $link_open; ?><img src="<?php echo esc_url( $portfolio['image'][0] ); ?>" /><?php echo $link_close; ?>
						<?php endif; ?>
						<div class="portfolio-content">
							<h3><?php echo $link_open; ?><?php echo esc_html( $portfolio['title'] ); ?><?php echo $link_close; ?></h3>
							<?php if ( ! empty( $portfolio['excerpt'] ) ) : ?>
								<div class="portfolio-excerpt">
									<?php echo wp_kses_post( wpautop( $portfolio['excerpt'] ) ); ?>
								</div><!-- .portfolio-excerpt -->
							<?php endif; ?>
							<div class="portfolio-meta">
								<ul>
									<li><span class="portfolio-date"><?php the_time( get_option('date_format') ); ?></span></li>
									<?php if ( comments_open( $portfolio['id'] ) && get_comments_number( $portfolio['id'] ) > 0 ) : ?>
										<?php
										echo '<li><span class="portfolio-comment">';
										comments_popup_link( '<span class="leave-reply">' . __( 'No Comment', 'bizroot-pro' ) . '</span>', __( '1 Comment', 'bizroot-pro' ), __( '% Comments', 'bizroot-pro' ) );
										echo '</span></li>';
										?>
									<?php endif; ?>
									<li><a href="<?php echo esc_url( $portfolio['url'] ); ?>" class="read-more"><?php _e( 'Read More', 'bizroot-pro' ); ?></a></li>
								</ul>
							</div><!-- .portfolio-meta -->
						</div><!-- .portfolio-content -->

					</article>

					<?php $cnt++; ?>

				<?php endforeach; ?>

			</div><!-- .cycle-slideshow -->

			<?php

		}

	}
endif;

if ( ! class_exists( 'Bizroot_Clients_Widget' ) ) :

	/**
	 * Clients widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Clients_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_clients',
				'description'                 => __( 'Displays clients.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				);

			$fields['message'] = array(
				'label' => __( 'This displays client logos entered in the Theme Options. To add/update logos, go to Appearance -> Customize -> Theme Options -> Clients Options', 'bizroot-pro' ),
				'type'  => 'message',
				'class' => 'widefat',
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-clients', __( 'Bizroot: Clients', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$featured_clients_number = absint( bizroot_get_option( 'featured_clients_number' ) );

			if ( absint( $featured_clients_number ) > 0 ) {
				$this->render_clients( $params );
			}

			//
			echo $args['after_widget'];

		}

		/**
		 * Render clients.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_clients( $params ) {

			$featured_clients_number = absint( bizroot_get_option( 'featured_clients_number' ) );

			$clients_logos = array();
			for( $i=1 ; $i <= $featured_clients_number ; $i ++ ){
				$clients_image =   bizroot_get_option( 'featured_clients_image_' .$i);
				$clients_link  =   bizroot_get_option( 'featured_clients_link_' .$i);
					if ( ! empty( $clients_image ) ) {
						$clients_logos[$i]['image'] = $clients_image;
						$clients_logos[$i]['link']  = $clients_link;
					}
			}
			if ( empty( $clients_logos ) ) {
				return;
			}
			?>
			<div class="clients-wrapper">

			<?php foreach ( $clients_logos as $client ) : ?>
				<div class="client-item">
					<?php
					$link_open  = '';
					$link_close = '';
					if ( ! empty( $client['link'] ) ) {
						$link_open  = '<a href="' . esc_url( $client['link'] ) . '" target="_blank">';
						$link_close = '</a>';
					}

					?>
					<?php echo $link_open; ?><img src="<?php echo esc_url( $client['image'] ); ?>" alt="" /><?php echo $link_close; ?>
				</div><!-- .client-item -->

			<?php endforeach; ?>

			</div><!-- .clients-wrapper -->
			<?php
		}

	}
endif;


if ( ! class_exists( 'Bizroot_Pricing_Table_Widget' ) ) :

	/**
	 * Pricing Table widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Pricing_Table_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_pricing_table',
				'description'                 => __( 'Displays Pricing Table.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'form_id' => array(
					'label'   => __( 'Select Pricing Table:', 'bizroot-pro' ),
					'type'    => 'select',
					'options' => bizroot_get_pricing_table_options(),
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			if ( ! shortcode_exists( 'rpt' ) ) {
				$fields = array();
				$fields['message'] = array(
					'label' => __( 'Please activate Responsive Pricing Table plugin to use this widget.', 'bizroot-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
					);
			}

			parent::__construct( 'bizroot-pricing-table', __( 'Bizroot: Pricing Table', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			if ( absint( $params['form_id'] ) > 0 && 'publish' === get_post_status( $params['form_id'] ) && shortcode_exists( 'rpt' ) ) {
				$this->render_pricing_table( $params );
			}

			echo $args['after_widget'];

		}

		/**
		 * Render pricing table.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_pricing_table( $params ) {

			$form_id = absint( $params['form_id'] );
			$entries = get_post_meta( $form_id, '_rpt_plan_group', true );

			$currency = get_post_meta( $form_id, '_rpt_currency', true );
			$newcurrentwindow = get_post_meta( $form_id, '_rpt_open_newwindow', true );

			$column = count( $entries );
			if ( $column < 1 ) {
				return;
			}
			?>
			<div class="pricing-table-wrapper">
				<div class="pricing-table-inner pricing-table-grid-<?php echo absint( $column ); ?>">

					<?php foreach ( $entries as $key => $plan ) : ?>

						<?php
						$is_recommended = isset( $plan['_rpt_recommended'] ) ? $plan['_rpt_recommended'] : '';
						$recommended_class = '';
						if ( 'on' === $is_recommended ) {
							$recommended_class = 'pricing-plan-recommended';
						}
						$plan_image_style = '';
						if ( ! empty( $plan['_rpt_plan_image'] ) ) {
							$plan_image_style = 'background-image:url(' . esc_url( $plan['_rpt_plan_image'] ) . ');';
						}
						?>

						<div class="pricing-plan-item <?php echo esc_attr( $recommended_class ); ?>">

							<div class="pricing-plan-header" style="<?php echo $plan_image_style ?>">

								<?php if ( ! empty( $plan['_rpt_price'] ) ) : ?>
									<div class="pricing-plan-cost">
										<?php if ( ! empty( $currency ) ) : ?>
											<span class="pricing-plan-currency"><?php echo esc_html( $currency ); ?></span>
										<?php endif; ?>
										<span class="pricing-plan-price"><?php echo esc_html( $plan['_rpt_price'] ); ?></span>
										<?php if ( ! empty( $plan['_rpt_recurrence'] ) ) : ?>
											<span class="pricing-plan-currency"><?php echo esc_html( $plan['_rpt_recurrence'] ); ?></span>
										<?php endif; ?>
									</div><!-- .pricing-plan-cost -->
								<?php endif; ?>
								<?php if ( ! empty( $plan['_rpt_title'] ) ) : ?>
									<div class="pricing-plan-title">
										<h4><?php echo esc_html( $plan['_rpt_title'] ); ?></h4>
									</div><!-- .pricing-plan-title -->
								<?php endif; ?>
							</div><!-- .pricing-plan-header -->

							<div class="pricing-plan-content">
								<?php if ( ! empty( $plan['_rpt_features'] ) ) : ?>
									<div class="pricing-plan-features">
										<?php echo $this->parse_pricing_feature( $plan['_rpt_features'] ); ?>
									</div><!-- .pricing-plan-features -->
								<?php endif; ?>

							</div><!-- .pricing-plan-content -->

							<div class="pricing-plan-footer">
								<?php if ( ! empty( $plan['_rpt_btn_custom_btn'] ) ) : ?>
									<div class="pricing-plan-custom-button"><?php echo do_shortcode( $plan['_rpt_btn_custom_btn'] ); ?></div><!-- .pricing-plan-custom-button -->
								<?php else : ?>
									<?php if ( ! empty( $plan['_rpt_btn_text'] ) && ! empty( $plan['_rpt_btn_link'] ) ) : ?>
										<div class="pricing-plan-button">
										<?php
											$link_open = '<a class="custom-button" href="' . esc_url( $plan['_rpt_btn_link'] ) . '"';
											if ( 'newwindow' === $newcurrentwindow  ) {
												$link_open .= ' target="_blank"';
											}
											$link_open .= '>';
											$link_close = '</a>';
										?>
										<?php echo $link_open . esc_html( $plan['_rpt_btn_text'] ) . $link_close; ?>
										</div><!-- .pricing-plan-button -->
									<?php endif; ?>

								<?php endif; ?>

							</div><!-- .pricing-plan-footer -->

						</div><!-- .pricing-plan-item -->

					<?php endforeach; ?>

				</div><!-- .pricing-table-inner -->
			</div><!-- .pricing-table-wrapper -->

			<?php

		}

		/**
		 * Parse pricing plan features.
		 *
		 * @since 1.0.0
		 *
		 * @param string $feature Feature.
		 * @return string Parsed feature content.
		 */
		function parse_pricing_feature( $feature ) {

			$output = '';

			$raw_features = explode( "\n", $feature );
			$raw_features = array_filter( $raw_features, 'trim' );

			$features = array();

			foreach ( $raw_features as $f ) {
				$features[] = strip_tags( $f,'<strong></strong><br><br/></br><img><a>' );
			}

			if ( ! empty( $features ) ) {
				foreach ( $features as $fkey => $ft ) {
					$ft = trim( $ft );
					$check = substr( $ft, 0, 2 );
					if ( '-n' === $check ) {
						$ft = substr( $ft, 2 );
						$output .= '<span class="pricing-plan-feature pricing-plan-feature-dim">' . esc_html( $ft ) . '</span>';
					}
					else {
						$output .= '<span class="pricing-plan-feature">' . esc_html( $ft ) . '</span>';
					}
				}
			}

			return $output;

		}

	}
endif;

if ( ! class_exists( 'Bizroot_Recent_Posts_Widget' ) ) :

	/**
	 * Recent posts widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Recent_Posts_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_recent_posts',
				'description'                 => __( 'Displays recent posts.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'bizroot-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'bizroot-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'bizroot-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 'thumbnail',
					'options' => bizroot_get_image_sizes_options( true, array( 'disable', 'thumbnail' ), false ),
					),
				'image_width' => array(
					'label'       => __( 'Image Width:', 'bizroot-pro' ),
					'type'        => 'number',
					'description' => __( 'px', 'bizroot-pro' ),
					'css'         => 'max-width:60px;',
					'adjacent'    => true,
					'default'     => 90,
					'min'         => 1,
					'max'         => 150,
					),
				'disable_date' => array(
					'label'   => __( 'Disable Date', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			parent::__construct( 'bizroot-recent-posts', __( 'Bizroot: Recent Posts', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				);
			if ( absint( $params['post_category'] ) > 0  ) {
				$qargs['cat'] = $params['post_category'];
			}
			$all_posts = get_posts( $qargs );

			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>

				<?php global $post; ?>

				<div class="recent-posts-wrapper">

					<?php foreach ( $all_posts as $key => $post ) :  ?>
						<?php setup_postdata( $post ); ?>

						<div class="recent-posts-item">

							<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) :  ?>
								<div class="recent-posts-thumb">
									<a href="<?php the_permalink(); ?>">
										<?php
										$img_attributes = array(
											'class' => 'alignleft',
											'style' => 'max-width:' . esc_attr( $params['image_width'] ). 'px;',
											);
										the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
										?>
									</a>
								</div><!-- .recent-posts-thumb -->
							<?php endif ?>
							<div class="recent-posts-text-wrap">
								<h3 class="recent-posts-title">
									<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
								</h3><!-- .recent-posts-title -->

								<?php if ( false === $params['disable_date'] ) :  ?>
									<div class="recent-posts-meta">

										<?php if ( false === $params['disable_date'] ) :  ?>
											<span class="recent-posts-date"><?php the_time( get_option( 'date_format' ) ); ?></span><!-- .recent-posts-date -->
										<?php endif; ?>

									</div><!-- .recent-posts-meta -->
								<?php endif; ?>

							</div><!-- .recent-posts-text-wrap -->

						</div><!-- .recent-posts-item -->

					<?php endforeach; ?>

				</div><!-- .recent-posts-wrapper -->

				<?php wp_reset_postdata(); // Reset. ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];

		}
	}
endif;

if ( ! class_exists( 'Bizroot_Teams_Widget' ) ) :

	/**
	 * Teams widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Teams_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'                   => 'bizroot_widget_teams',
				'description'                 => __( 'Displays teams.', 'bizroot-pro' ),
				'customize_selective_refresh' => true,
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'post_category' => array(
					'label'           => __( 'Select Category:', 'bizroot-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => __( 'All Categories', 'bizroot-pro' ),
					),
				'post_number' => array(
					'label'   => __( 'Number of Posts:', 'bizroot-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
					),
				'post_column' => array(
					'label'   => __( 'Number of Columns:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 3,
					'options' => bizroot_get_numbers_dropdown_options( 1, 4 ),
					),
				'featured_image' => array(
					'label'   => __( 'Featured Image:', 'bizroot-pro' ),
					'type'    => 'select',
					'default' => 'medium',
					'options' => bizroot_get_image_sizes_options( false, array( 'thumbnail', 'medium' ), false ),
					),
				'excerpt_length' => array(
					'label'       => __( 'Excerpt Length:', 'bizroot-pro' ),
					'description' => __( 'in words', 'bizroot-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 20,
					'min'         => 1,
					'max'         => 400,
					'adjacent'    => true,
					),
				'disable_excerpt' => array(
					'label'   => __( 'Disable Excerpt', 'bizroot-pro' ),
					'type'    => 'checkbox',
					'default' => false,
					),
				);

			$animation_fields = bizroot_get_widget_animation_fields();
			$fields = array_merge( $fields, $animation_fields );

			parent::__construct( 'bizroot-teams', __( 'Bizroot: Teams', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );
			$args = $this->add_animation( $args, $params );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}
			$this->render_teams( $params );
			echo $args['after_widget'];

		}
		/**
		 * Render teams.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		function render_teams( $params ) {

			$qargs = array(
				'posts_per_page' => esc_attr( $params['post_number'] ),
				'no_found_rows'  => true,
				'meta_key'       => '_thumbnail_id',
				);
			if ( absint( $params['post_category'] ) > 0  ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}
			$all_posts = get_posts( $qargs );
			?>
			<?php if ( ! empty( $all_posts ) ) :  ?>

				<?php global $post; ?>

				<div class="teams-widget teams-col-<?php echo esc_attr( $params['post_column'] ); ?>">
					<div class="inner-wrapper">
						<?php foreach ( $all_posts as $key => $post ) :  ?>
							<?php setup_postdata( $post ); ?>
							<div class="team-item">
							<?php if ( has_post_thumbnail() ) :  ?>
								<div class="thumb-summary-wrap">
									<div class="team-thumb">
										<a href="<?php the_permalink(); ?>">
											<?php the_post_thumbnail( esc_attr( $params['featured_image'] ), array( 'class' => 'aligncenter' ) ); ?>
										</a>
									</div><!-- .team-thumb-->
									<div class="team-text-wrap">
										<h3 class="team-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
										<?php if ( true !== $params['disable_excerpt'] ) : ?>
											<div class="team-summary">
												<?php
												$summary = bizroot_the_excerpt( absint( $params['excerpt_length'] ), $post );
												echo wp_kses_post( wpautop( $summary ) );
												 ?>
											</div><!-- .team-summary -->
										<?php endif; ?>

									</div><!-- .team-text-wrap -->
								</div><!-- .thumb-summary-wrap-->

							<?php endif; ?>

							</div><!-- .team-item -->
						<?php endforeach; ?>

					</div><!-- .inner-wrapper -->
				</div><!-- .teams-widget -->

				<?php wp_reset_postdata(); // Reset. ?>

			<?php endif; ?>

			<?php

		}

	}
endif;

if ( ! class_exists( 'Bizroot_Skills_Widget' ) ) :

	/**
	 * Skills widget Class.
	 *
	 * @since 1.0.0
	 */
	class Bizroot_Skills_Widget extends Bizroot_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		function __construct() {

			$opts = array(
				'classname'   => 'bizroot_widget_skills',
				'description' => __( 'Displays skill charts.', 'bizroot-pro' ),
				);
			$fields = array(
				'title' => array(
					'label' => __( 'Title:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'subtitle' => array(
					'label' => __( 'Subtitle:', 'bizroot-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
					),
				'skill_id' => array(
					'label'   => __( 'Select Skills:', 'bizroot-pro' ),
					'type'    => 'select',
					'options' => bizroot_get_skills_options(),
					),
				'skill_message' => array(
					'label' => _x( 'OR', 'Bizroot Skills', 'bizroot-pro' ),
					'type'  => 'heading',
					),
				'skills_shortcode' => array(
					'label' => __( 'Enter Skills Shortcode:', 'bizroot-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
					),
				);

			parent::__construct( 'bizroot-skills', __( 'Bizroot: Skills', 'bizroot-pro' ), $opts, array(), $fields );

		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		function widget( $args, $instance ) {

			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . $params['title'] . $args['after_title'];
			}
			if ( ! empty( $params['subtitle'] ) ) {
				echo '<p class="widget-subtitle">' . esc_html( $params['subtitle'] ) . '</p>';
			}

			$shortcode_text = null;

			if ( ! empty( $params['skill_id'] ) && absint( $params['skill_id'] ) > 0 && class_exists( 'WEN_Skill_Charts' ) ) {
				$shortcode_text = '[wsc id="' . absint( $params['skill_id'] ) . '"]';
			}

			if ( empty( $shortcode_text ) ) {
				$shortcode_text = wp_kses_data( $params['skills_shortcode'] );
			}

			if ( ! empty( $shortcode_text ) ) {
				echo do_shortcode( $shortcode_text );
			}

			echo $args['after_widget'];

		}
	}
endif;
