<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Busyness
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function busyness_body_classes( $classes ) {
	$options = busyness_get_theme_options();

	$classes[] = 'classic-menu';

	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	if ( $options['menu_sticky'] ) {
		$classes[] = 'menu-sticky';
	}

	// Add a class for typography
	$typography = (  $options['theme_typography'] == 'default' ) ? '' :  $options['theme_typography'];
	$classes[] = esc_attr( $typography );

	$body_typography = (  $options['body_theme_typography'] == 'default' ) ? '' :  $options['body_theme_typography'];
	$classes[] = esc_attr( $body_typography );

	// Add a class for layout
	$classes[] = esc_attr( $options['site_layout'] );

	// Add a class for sidebar
	$sidebar_position = busyness_layout();
	$sidebar = 'sidebar-1';
	if ( is_singular() || is_home() ) {
		$id = ( is_home() && ! is_front_page() ) ? get_option( 'page_for_posts' ) : get_the_id();
	  	$sidebar = get_post_meta( $id, 'busyness-pro-selected-sidebar', true );
	  	$sidebar = ! empty( $sidebar ) ? $sidebar : 'sidebar-1';
	}

	if ( is_active_sidebar( $sidebar ) ) {
		$classes[] = esc_attr( $sidebar_position );
	} else {
		$classes[] = 'no-sidebar';
	}

	$theme_version	= ! empty( $options['theme_version'] ) ? $options['theme_version'] : '' ;
	$classes[]		= esc_attr( $theme_version ) ;

	return $classes;
}
add_filter( 'body_class', 'busyness_body_classes' );

/**
 * Load theme options from free theme.
 *
 * Checks if there are options already present from free version and adds it to the Pro theme options.
 *
 * @since 1.4
 */
function busyness_import_free_options( $old_theme_name = 'knowledge' ) {
	$free_version_slug = array(
		'busyness',
		'knowledge',
	);
	
	$old_theme_slug = sanitize_title( $old_theme_name );

	if ( ! in_array( $old_theme_slug, $free_version_slug ) ) {
		// Bail if free theme's not of this pro theme.
		return;
	}

	// Perform action only if theme_mods_XXX[theme_options] does not exist.
	if( ! get_theme_mod( 'busyness_theme_options' ) ) {
		$old_theme_slug = sanitize_title( $old_theme_name );

		$free_options = get_option( 'theme_mods_' . $old_theme_slug );

		// Perform action only if theme_mods_XXX free version exists.
		if ( $free_options ) {
			if ( isset( $free_options['busyness_theme_options'] ) ) {
				if ( 'knowledge' === $old_theme_slug ) {
					$free_options['busyness_theme_options']['theme_version'] = 'knowledge-version';
				}

				$pro_default_options = busyness_get_default_theme_options();
				$pro_theme_options = $free_options;


				$pro_theme_options['busyness_theme_options'] = array_merge( $pro_default_options , $free_options['busyness_theme_options'] );

				// WP default fields.
				$fields = array(
					'custom_logo',
					'header_image',
					'header_image_data',
					'background_image',
					'background_repeat',
					'background_position_x',
					'background_attachment',
				);

				foreach ( $fields as $key ) {
					if ( isset( $free_options[ $key ] ) && ! empty( $free_options[ $key ] ) ) {
						$pro_theme_options[ $key ] = $free_options[ $key ];
					}
				}

				update_option( 'theme_mods_busyness-pro', $pro_theme_options );
			}
		}
	}
}
add_action( 'after_switch_theme', 'busyness_import_free_options' );
