<?php
/**
 * Helper functions related to customizer and options
 *
 * @package Magazine_Power
 */

if ( ! function_exists( 'magazine_power_get_global_layout_options' ) ) :

	/**
	 * Returns global layout options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_global_layout_options() {
		$choices = array(
			'left-sidebar'      => esc_html__( 'Primary Sidebar - Content', 'magazine-power-pro' ),
			'right-sidebar'     => esc_html__( 'Content - Primary Sidebar', 'magazine-power-pro' ),
			'three-columns'     => esc_html__( 'Three Columns ( Secondary - Content - Primary )', 'magazine-power-pro' ),
			'three-columns-pcs' => esc_html__( 'Three Columns ( Primary - Content - Secondary )', 'magazine-power-pro' ),
			'three-columns-cps' => esc_html__( 'Three Columns ( Content - Primary - Secondary )', 'magazine-power-pro' ),
			'three-columns-psc' => esc_html__( 'Three Columns ( Primary - Secondary - Content )', 'magazine-power-pro' ),
			'no-sidebar'        => esc_html__( 'No Sidebar', 'magazine-power-pro' ),
		);

		$output = apply_filters( 'magazine_power_filter_layout_options', $choices );

		return $output;
	}

endif;

if ( ! function_exists( 'magazine_power_get_site_layout_options' ) ) :

	/**
	 * Returns site layout options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_site_layout_options() {
		$choices = array(
			'fluid' => esc_html__( 'Fluid', 'magazine-power-pro' ),
			'boxed' => esc_html__( 'Boxed', 'magazine-power-pro' ),
		);

		$output = apply_filters( 'magazine_power_filter_site_layout_options', $choices );

		if ( ! empty( $output ) ) {
			ksort( $output );
		}

		return $output;
	}

endif;

if ( ! function_exists( 'magazine_power_get_archive_layout_options' ) ) :

	/**
	 * Returns archive layout options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_archive_layout_options() {
		$choices = array(
			'full'    => esc_html__( 'Full Post', 'magazine-power-pro' ),
			'excerpt' => esc_html__( 'Post Excerpt', 'magazine-power-pro' ),
		);

		$output = apply_filters( 'magazine_power_filter_archive_layout_options', $choices );

		if ( ! empty( $output ) ) {
			ksort( $output );
		}

		return $output;
	}

endif;

if ( ! function_exists( 'magazine_power_get_image_sizes_options' ) ) :

	/**
	 * Returns image sizes options.
	 *
	 * @since 1.0.0
	 *
	 * @param bool  $add_disable True for adding No Image option.
	 * @param array $allowed Allowed image size options.
	 * @param bool  $show_dimension Show image dimension.
	 * @return array Image size options.
	 */
	function magazine_power_get_image_sizes_options( $add_disable = true, $allowed = array(), $show_dimension = true ) {
		global $_wp_additional_image_sizes;

		$get_intermediate_image_sizes = get_intermediate_image_sizes();

		$choices = array();

		if ( true === $add_disable ) {
			$choices['disable'] = esc_html__( 'No Image', 'magazine-power-pro' );
		}

		$choices['thumbnail'] = esc_html__( 'Thumbnail', 'magazine-power-pro' );
		$choices['medium']    = esc_html__( 'Medium', 'magazine-power-pro' );
		$choices['large']     = esc_html__( 'Large', 'magazine-power-pro' );
		$choices['full']      = esc_html__( 'Full (original)', 'magazine-power-pro' );

		if ( true === $show_dimension ) {
			foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
				$choices[ $_size ] = $choices[ $_size ] . ' (' . get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
			}
		}

		if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {
			foreach ( $_wp_additional_image_sizes as $key => $size ) {
				$choices[ $key ] = $key;
				if ( true === $show_dimension ) {
					$choices[ $key ] .= ' (' . $size['width'] . 'x' . $size['height'] . ')';
				}
			}
		}

		if ( ! empty( $allowed ) ) {
			foreach ( $choices as $key => $value ) {
				if ( ! in_array( $key, $allowed, true ) ) {
					unset( $choices[ $key ] );
				}
			}
		}

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_image_alignment_options' ) ) :

	/**
	 * Returns image alignment options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_image_alignment_options() {
		$choices = array(
			'none'   => _x( 'None', 'Alignment', 'magazine-power-pro' ),
			'left'   => _x( 'Left', 'Alignment', 'magazine-power-pro' ),
			'center' => _x( 'Center', 'Alignment', 'magazine-power-pro' ),
			'right'  => _x( 'Right', 'Alignment', 'magazine-power-pro' ),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_numbers_dropdown_options' ) ) :

	/**
	 * Returns numbers dropdown options.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $min    Min.
	 * @param int    $max    Max.
	 * @param string $prefix Prefix.
	 * @param string $suffix Suffix.
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_numbers_dropdown_options( $min = 1, $max = 4, $prefix = '', $suffix = '' ) {
		$output = array();

		if ( $min <= $max ) {
			for ( $i = $min; $i <= $max; $i++ ) {
				$string       = $prefix . $i . $suffix;
				$output[ $i ] = $string;
			}
		}

		return $output;
	}

endif;

if ( ! function_exists( 'magazine_power_get_pagination_type_options' ) ) :

	/**
	 * Returns pagination type options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_pagination_type_options() {
		$choices = array(
			'default' => esc_html__( 'Default (Older / Newer Post)', 'magazine-power-pro' ),
			'numeric' => esc_html__( 'Numeric', 'magazine-power-pro' ),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_breadcrumb_type_options' ) ) :

	/**
	 * Returns breadcrumb type options.
	 *
	 * @since 1.0.0
	 *
	 * @return array Options array.
	 */
	function magazine_power_get_breadcrumb_type_options() {
		$choices = array(
			'disabled' => esc_html__( 'Disabled', 'magazine-power-pro' ),
			'simple'   => esc_html__( 'Simple', 'magazine-power-pro' ),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_customizer_font_options' ) ) :

	/**
	 * Returns customizer font options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_customizer_font_options() {
		$web_fonts = magazine_power_get_web_fonts();
		$os_fonts  = magazine_power_get_os_fonts();

		$web_fonts = array_merge( $web_fonts, $os_fonts );

		if ( ! empty( $web_fonts ) ) {
			ksort( $web_fonts );
		}

		$choices = array();

		if ( ! empty( $web_fonts ) ) {
			foreach ( $web_fonts as $k => $v ) {
				$choices[ $k ] = esc_html( $v['label'] );
			}
		}

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_web_fonts' ) ) :

	/**
	 * Returns web font options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_web_fonts() {
		$choices = array(
			'philosopher'           => array(
				'name'  => 'Philosopher',
				'label' => "'Philosopher', sans-serif",
			),
			'open-sans'             => array(
				'name'  => 'Open Sans',
				'label' => "'Open Sans', sans-serif",
			),
			'pt-sans'               => array(
				'name'  => 'PT Sans',
				'label' => "'PT Sans', sans-serif",
			),
			'merriweather-sans'     => array(
				'name'  => 'Merriweather Sans',
				'label' => "'Merriweather Sans', sans-serif",
			),
			'roboto-condensed'                => array(
				'name'  => 'Roboto Condensed',
				'label' => "'Roboto Condensed', sans-serif",
			),
			'libre-Franklin'                => array(
				'name'  => 'Libre Franklin',
				'label' => "'Libre Franklin', sans-serif",
			),
			'roboto'                => array(
				'name'  => 'Roboto',
				'label' => "'Roboto', sans-serif",
			),
			'arizonia'              => array(
				'name'  => 'Arizonia',
				'label' => "'Arizonia', cursive",
			),
			'raleway'               => array(
				'name'  => 'Raleway',
				'label' => "'Raleway', sans-serif",
			),
			'droid-sans'            => array(
				'name'  => 'Droid Sans',
				'label' => "'Droid Sans', sans-serif",
			),
			'lato'                  => array(
				'name'  => 'Lato',
				'label' => "'Lato', sans-serif",
			),
			'dosis'                 => array(
				'name'  => 'Dosis',
				'label' => "'Dosis', sans-serif",
			),
			'slabo-27px'            => array(
				'name'  => 'Slabo 27px',
				'label' => "'Slabo 27px', serif",
			),
			'oswald'                => array(
				'name'  => 'Oswald',
				'label' => "'Oswald', sans-serif",
			),
			'pt-sans-narrow'        => array(
				'name'  => 'PT Sans Narrow',
				'label' => "'PT Sans Narrow', sans-serif",
			),
			'josefin-slab'          => array(
				'name'  => 'Josefin Slab',
				'label' => "'Josefin Slab', serif",
			),
			'alegreya'              => array(
				'name'  => 'Alegreya',
				'label' => "'Alegreya', serif",
			),
			'exo'                   => array(
				'name'  => 'Exo',
				'label' => "'Exo', sans-serif",
			),
			'signika'               => array(
				'name'  => 'Signika',
				'label' => "'Signika', sans-serif",
			),
			'lobster'               => array(
				'name'  => 'Lobster',
				'label' => "'Lobster', cursive",
			),
			'indie-flower'          => array(
				'name'  => 'Indie Flower',
				'label' => "'Indie Flower', cursive",
			),
			'shadows-into-light'    => array(
				'name'  => 'Shadows Into Light',
				'label' => "'Shadows Into Light', cursive",
			),
			'kaushan-script'        => array(
				'name'  => 'Kaushan Script',
				'label' => "'Kaushan Script', cursive",
			),
			'dancing-script'        => array(
				'name'  => 'Dancing Script',
				'label' => "'Dancing Script', cursive",
			),
			'fredericka-the-great'  => array(
				'name'  => 'Fredericka the Great',
				'label' => "'Fredericka the Great', cursive",
			),
			'covered-by-your-grace' => array(
				'name'  => 'Covered By Your Grace',
				'label' => "'Covered By Your Grace', cursive",
			),
			'rajdhani'              => array(
				'name'  => 'Rajdhani',
				'label' => "'Rajdhani', sans-serif",
			),
			'source-sans-pro'       => array(
				'name'  => 'Source Sans Pro',
				'label' => "'Source Sans Pro', sans-serif",
			),
			'archivo-narrow'       => array(
				'name'  => 'Archivo Narrow',
				'label' => "'Archivo Narrow', sans-serif",
			),
			'markazi-text'       => array(
				'name'  => 'Markazi Text',
				'label' => "'Markazi Text', serif",
			),
		);

		$choices = apply_filters( 'magazine_power_filter_web_fonts', $choices );

		if ( ! empty( $choices ) ) {
			ksort( $choices );
		}

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_os_fonts' ) ) :

	/**
	 * Returns OS font options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_os_fonts() {
		$choices = array(
			'arial'      => array(
				'name'  => 'Arial',
				'label' => "'Arial', sans-serif",
			),
			'georgia'    => array(
				'name'  => 'Georgia',
				'label' => "'Georgia', serif",
			),
			'cambria'    => array(
				'name'  => 'Cambria',
				'label' => "'Cambria', Georgia, serif",
			),
			'tahoma'     => array(
				'name'  => 'Tahoma',
				'label' => "'Tahoma', Geneva, sans-serif",
			),
			'sans-serif' => array(
				'name'  => 'Sans Serif',
				'label' => "'Sans Serif', Arial",
			),
			'verdana'    => array(
				'name'  => 'Verdana',
				'label' => "'Verdana', Geneva, sans-serif",
			),
		);

		$choices = apply_filters( 'magazine_power_filter_os_fonts', $choices );

		if ( ! empty( $choices ) ) {
			ksort( $choices );
		}

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_font_family_from_key' ) ) :

	/**
	 * Return font family from font slug.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key Font slug.
	 * @return string Font name.
	 */
	function magazine_power_get_font_family_from_key( $key ) {
		$output = '';

		$web_fonts = magazine_power_get_web_fonts();
		$os_fonts  = magazine_power_get_os_fonts();

		$fonts = array_merge( $web_fonts, $os_fonts );

		if ( isset( $fonts[ $key ] ) ) {
			$output = $fonts[ $key ]['label'];
		}

		return $output;
	}

endif;

if ( ! function_exists( 'magazine_power_get_font_family_theme_settings_options' ) ) :

	/**
	 * Returns font family theme settings options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_font_family_theme_settings_options() {
		$choices = array(
			'font_site_title'    => array(
				'label'   => esc_html__( 'Site Title', 'magazine-power-pro' ),
				'default' => 'archivo-narrow',
			),
			'font_site_tagline'  => array(
				'label'   => esc_html__( 'Site Tagline', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
			'font_site_default'  => array(
				'label'   => esc_html__( 'Default', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
			'font_content_title' => array(
				'label'   => esc_html__( 'Content Title', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
			'font_content_body'  => array(
				'label'   => esc_html__( 'Content Body', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
			'font_heading_tags'  => array(
				'label'   => esc_html__( 'Heading Tags', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
			'font_navigation'    => array(
				'label'   => esc_html__( 'Navigation', 'magazine-power-pro' ),
				'default' => 'roboto',
			),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_color_panels_options' ) ) :

	/**
	 * Returns color panels options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_color_panels_options() {
		$choices = array(
			'color_panel_main' => array(
				'label' => esc_html__( 'Color Options', 'magazine-power-pro' ),
			),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_color_sections_options' ) ) :

	/**
	 * Returns color sections options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_color_sections_options() {
		$choices = array(
			'color_section_basic'             => array(
				'label' => esc_html__( 'Basic Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_top_header'        => array(
				'label' => esc_html__( 'Top Header Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_header'            => array(
				'label' => esc_html__( 'Header Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_primary_menu'      => array(
				'label' => esc_html__( 'Primary Menu Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_featured_content'  => array(
				'label' => esc_html__( 'Featured Content Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_content'           => array(
				'label' => esc_html__( 'Content Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_sidebar'           => array(
				'label' => esc_html__( 'Sidebar Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_breadcrumb'        => array(
				'label' => esc_html__( 'Breadcrumb Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_goto_top'          => array(
				'label' => esc_html__( 'Go To Top Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_pagination'        => array(
				'label' => esc_html__( 'Pagination Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_footer_widgets'    => array(
				'label' => esc_html__( 'Footer Widgets Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_footer_area'       => array(
				'label' => esc_html__( 'Footer Area Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
			'color_section_home_widgets_area' => array(
				'label' => esc_html__( 'Home Widget Area Color Options', 'magazine-power-pro' ),
				'panel' => 'color_panel_main',
			),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_color_theme_settings_options' ) ) :

	/**
	 * Returns color theme settings options.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_get_color_theme_settings_options() {
		$choices = array(
			// Basic.
			'color_basic_text'                         => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_link'                         => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_link_hover'                   => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_heading'                      => array(
				'label'   => esc_html__( 'Heading Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_button_text'                  => array(
				'label'   => esc_html__( 'Button Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_button_background'            => array(
				'label'   => esc_html__( 'Button Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_button_text_hover'            => array(
				'label'   => esc_html__( 'Button Text Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),
			'color_basic_button_background_hover'      => array(
				'label'   => esc_html__( 'Button Background Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_basic',
			),

			// Top Header.
			'color_top_header_background'              => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),

			'color_top_header_link'                    => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),
			'color_top_header_link_hover'              => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),
			'color_top_header_text'                    => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),
			'color_top_header_date_background'         => array(
				'label'   => esc_html__( 'Date Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),
			'color_top_header_date_text'               => array(
				'label'   => esc_html__( ' Date Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_top_header',
			),

			// Header.
			'color_header_background'                  => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_header',
			),
			'color_header_title'                       => array(
				'label'   => esc_html__( 'Site Title Color', 'magazine-power-pro' ),
				'section' => 'color_section_header',
			),
			'color_header_title_hover'                 => array(
				'label'   => esc_html__( 'Site Title Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_header',
			),
			'color_header_tagline'                     => array(
				'label'   => esc_html__( 'Site Tagline Color', 'magazine-power-pro' ),
				'section' => 'color_section_header',
			),

			// Primary Menu.
			'color_primary_menu_backgorund'            => array(
				'label'   => esc_html__( 'Main Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_primary_menu',
			),
			'color_primary_menu_link'                  => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_primary_menu',
			),
			'color_primary_menu_link_background_hover' => array(
				'label'   => esc_html__( 'Link Background Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_primary_menu',
			),
			'color_primary_menu_link_hover'            => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_primary_menu',
			),

			// Content.
			'color_content_background'                 => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_title'                      => array(
				'label'   => esc_html__( 'Title Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_text'                       => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_link'                       => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_link_hover'                 => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_meta_link'                  => array(
				'label'   => esc_html__( 'Meta Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_meta_link_hover'            => array(
				'label'   => esc_html__( 'Meta Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),
			'color_content_meta_icon'                  => array(
				'label'   => esc_html__( 'Meta Icon Color', 'magazine-power-pro' ),
				'section' => 'color_section_content',
			),

			// Home Page Widgets area.
			'color_home_widgets_area_title'            => array(
				'label'   => esc_html__( 'Title Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),

			'color_home_widgets_area_title_border'     => array(
				'label'   => esc_html__( 'Title border Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_text'             => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_link'             => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_link_hover'       => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-color'       => array(
				'label'   => esc_html__( 'Category Color', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-background-1'       => array(
				'label'   => esc_html__( 'Category Background First', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-background-2'       => array(
				'label'   => esc_html__( 'Category Background Second', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-background-3'       => array(
				'label'   => esc_html__( 'Category Background Third', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-background-4'       => array(
				'label'   => esc_html__( 'Category Background Fourth', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			'color_home_widgets_area_cat-background-5'       => array(
				'label'   => esc_html__( 'Category Background Fifth', 'magazine-power-pro' ),
				'section' => 'color_section_home_widgets_area',
			),
			// Sidebar.
			'color_sidebar_title'                      => array(
				'label'   => esc_html__( 'Title Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),
			'color_sidebar_title_border'               => array(
				'label'   => esc_html__( 'Title border Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),
			'color_sidebar_text'                       => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),
			'color_sidebar_link'                       => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),
			'color_sidebar_link_hover'                 => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),
			'color_sidebar_list_icon'                  => array(
				'label'   => esc_html__( 'List Icon Color', 'magazine-power-pro' ),
				'section' => 'color_section_sidebar',
			),

			// Footer area.
			'color_footer_area_background'             => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_area',
			),
			'color_footer_area_text'                   => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_area',
			),
			'color_footer_area_link'                   => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_area',
			),
			'color_footer_area_link_hover'             => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_area',
			),

			// Go To Top.
			'color_goto_top_icon'                      => array(
				'label'   => esc_html__( 'Icon Color', 'magazine-power-pro' ),
				'section' => 'color_section_goto_top',
			),
			'color_goto_top_icon_hover'                => array(
				'label'   => esc_html__( 'Icon Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_goto_top',
			),
			'color_goto_top_background'                => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_goto_top',
			),
			'color_goto_top_background_hover'          => array(
				'label'   => esc_html__( 'Background Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_goto_top',
			),

			// Pagination.
			'color_pagination_link'                    => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_pagination',
			),
			'color_pagination_link_hover'              => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_pagination',
			),
			'color_pagination_link_background'         => array(
				'label'   => esc_html__( 'Link Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_pagination',
			),
			'color_pagination_link_background_hover'   => array(
				'label'   => esc_html__( 'Link Background Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_pagination',
			),

			// Breadcrumb.
			'color_breadcrumb_background'                    => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_breadcrumb',
			),
			'color_breadcrumb_link'                    => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_breadcrumb',
			),
			'color_breadcrumb_link_hover'              => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_breadcrumb',
			),
			'color_breadcrumb_text'                    => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_breadcrumb',
			),

			// Footer widgets.
			'color_footer_widgets_background'          => array(
				'label'   => esc_html__( 'Background Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
			'color_footer_widgets_title'               => array(
				'label'   => esc_html__( 'Title Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
			'color_footer_widgets_title_border'        => array(
				'label'   => esc_html__( 'Title Border Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
			'color_footer_widgets_text'                => array(
				'label'   => esc_html__( 'Text Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
			'color_footer_widgets_link'                => array(
				'label'   => esc_html__( 'Link Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
			'color_footer_widgets_link_hover'          => array(
				'label'   => esc_html__( 'Link Hover Color', 'magazine-power-pro' ),
				'section' => 'color_section_footer_widgets',
			),
		);

		return $choices;
	}

endif;

if ( ! function_exists( 'magazine_power_get_default_colors' ) ) :

	/**
	 * Returns default colors.
	 *
	 * @since 1.0.0
	 *
	 * @param string $scheme Color scheme.
	 * @return array Color values based on scheme.
	 */
	function magazine_power_get_default_colors( $scheme = 'default' ) {
		$output = array();

		switch ( $scheme ) {

			case 'default':
			default:
				$output = array(
					// Basic.
					'color_basic_text'                     => '#242424',
					'color_basic_link'                     => '#222222',
					'color_basic_link_hover'               => '#ee3333',
					'color_basic_heading'                  => '#242424',
					'color_basic_button_background'        => '#ee3333',
					'color_basic_button_text'              => '#ffffff',
					'color_basic_button_background_hover'  => '#242424',
					'color_basic_button_text_hover'        => '#ffffff',

					// Top Header.
					'color_top_header_background'          => '#232323',
					'color_top_header_link'                => '#d7d7d7',
					'color_top_header_link_hover'          => '#ffffff',
					'color_top_header_text'                => '#d7d7d7',
					'color_top_header_date_background'     => '#ee3333',
					'color_top_header_date_text'           => '#ffffff',

					// Header.
					'color_header_background'              => '#ffffff',
					'color_header_title'                   => '#232323',
					'color_header_title_hover'             => '#232323',
					'color_header_tagline'                 => '#666666',

					// Primary Menu.
					'color_primary_menu_backgorund'        => '#232323',
					'color_primary_menu_link'              => '#d7d7d7',
					'color_primary_menu_link_hover'        => '#ffffff',
					'color_primary_menu_link_background_hover' => '#ee3333',

					// Content.
					'color_content_background'             => '#ffffff',
					'color_content_title'                  => '#242424',
					'color_content_text'                   => '#242424',
					'color_content_link'                   => '#ee3333',
					'color_content_link_hover'             => '#ee3333',
					'color_content_meta_link'              => '#242424',
					'color_content_meta_link_hover'        => '#ee3333',
					'color_content_meta_icon'              => '#242424',

					// Sidebar.
					'color_sidebar_title'                  => '#242424',
					'color_sidebar_title_border'           => '#242424',
					'color_sidebar_text'                   => '#242424',
					'color_sidebar_link'                   => '#242424',
					'color_sidebar_link_hover'             => '#ee3333',
					'color_sidebar_list_icon'              => '#ee3333',

					// Home Page Widgets area.
					'color_home_widgets_area_title'        => '#242424',
					'color_home_widgets_area_title_border' => '#242424',
					'color_home_widgets_area_text'         => '#242424',
					'color_home_widgets_area_link'         => '#ee3333',
					'color_home_widgets_area_link_hover'   => '#ee3333',
					'color_home_widgets_area_cat-color'   => '#ffffff',
					'color_home_widgets_area_cat-background-1'   => '#ee3333',
					'color_home_widgets_area_cat-background-2'   => '#0000ff',
					'color_home_widgets_area_cat-background-3'   => '#d2b210',
					'color_home_widgets_area_cat-background-4'   => '#6ec16e',
					'color_home_widgets_area_cat-background-5'   => '#c32bc3',



					// Footer Widgets.
					'color_footer_widgets_background'      => '#bbbbbb',
					'color_footer_widgets_title'           => '#bbbbbb',
					'color_footer_widgets_title_border'    => '#bbbbbb',
					'color_footer_widgets_text'            => '#bbbbbb',
					'color_footer_widgets_link'            => '#bbbbbb',
					'color_footer_widgets_link_hover'      => '#bbbbbb',

					// Footer area.
					'color_footer_area_background'         => '#171717',
					'color_footer_area_text'               => '#bbbbbb',
					'color_footer_area_link'               => '#ffffff',
					'color_footer_area_link_hover'         => '#ee3333',

					// Go To Top.
					'color_goto_top_icon'                  => '#ffffff',
					'color_goto_top_icon_hover'            => '#ffffff',
					'color_goto_top_background'            => '#242424',
					'color_goto_top_background_hover'      => '#ee3333',

					// Pagination.
					'color_pagination_link'                => '#ee3333',
					'color_pagination_link_hover'          => '#fff',
					'color_pagination_link_background'     => '#ffffff',
					'color_pagination_link_background_hover' => '#ee3333',

					// Breadcrumb.
					'color_breadcrumb_background'                => '#ffffff',
					'color_breadcrumb_link'                => '#242424',
					'color_breadcrumb_link_hover'          => '#ee3333',
					'color_breadcrumb_text'                => '#ee3333',
				);

				break;

		} // End switch.

		return $output;
	}

endif;
