<?php
/**
 * Theme widgets
 *
 * @package Magazine_Power
 */

// Load widget base.
require_once get_template_directory() . '/lib/widget-base/class-widget-base.php';

if ( ! function_exists( 'magazine_power_load_widgets' ) ) :

	/**
	 * Load widgets.
	 *
	 * @since 1.0.0
	 */
	function magazine_power_load_widgets() {
		// Social widget.
		register_widget( 'Magazine_Power_Social_Widget' );

		// Main Block widget.
		register_widget( 'Magazine_Power_Main_Block_Widget' );

		// Categories widget.
		register_widget( 'Magazine_Power_Categories_Widget' );

		// Featured Page widget.
		register_widget( 'Magazine_Power_Featured_Page_Widget' );

		// Latest News widget.
		register_widget( 'Magazine_Power_Latest_News_Widget' );

		// Recent Posts widget.
		register_widget( 'Magazine_Power_Recent_Posts_Widget' );

		// News Block widget.
		register_widget( 'Magazine_Power_News_Block_Widget' );

		// News Slider widget.
		register_widget( 'Magazine_Power_News_Slider_Widget' );

		// News Carousel widget.
		register_widget( 'Magazine_Power_News_Carousel_Widget' );

		// Tabbed widget.
		register_widget( 'Magazine_Power_Tabbed_Widget' );

		// Categorized News widget.
		register_widget( 'Magazine_Power_Categorized_News_Widget' );

		// Categorized Columns widget.
		register_widget( 'Magazine_Power_Categorized_Columns_Widget' );

		// Call To Action widget.
		register_widget( 'Magazine_Power_Call_To_Action_Widget' );
	}

endif;

add_action( 'widgets_init', 'magazine_power_load_widgets' );

if ( ! class_exists( 'Magazine_Power_Social_Widget' ) ) :

	/**
	 * Social Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Social_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_social',
				'description'                 => esc_html__( 'Displays social icons.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
			);

			if ( false === has_nav_menu( 'social' ) ) {
				$fields['message'] = array(
					'label' => esc_html__( 'Social menu is not set. Please create menu and assign it to Social Menu.', 'magazine-power-pro' ),
					'type'  => 'message',
					'class' => 'widefat',
				);
			}

			parent::__construct( 'magazine-power-social', esc_html__( 'MP: Social', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			if ( has_nav_menu( 'social' ) ) {
				wp_nav_menu(
					array(
						'theme_location' => 'social',
						'container'      => false,
						'depth'          => 1,
						'link_before'    => '<span class="screen-reader-text">',
						'link_after'     => '</span>',
					)
				);
			}

			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Main_Block_Widget' ) ) :

	/**
	 * Main Block Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Main_block_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'   => 'magazine_power_widget_main_block',
				'description' => esc_html__( 'Displays Main Block.', 'magazine-power-pro' ),
			);

			$fields = array(
				'title' => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'heading_1'  => array(
					'label' => esc_html__( 'First Section', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'post_category_1'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_number_1'   => array(
					'label'   => esc_html__( 'No. of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'css'     => 'max-width:60px;',
					'default' => 4,
					'min'     => 1,
					'max'     => 10,
				),
				'heading_2'  => array(
					'label' => esc_html__( 'Second Section', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'post_category_2'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'heading_3'  => array(
					'label' => esc_html__( 'Third Section', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'post_category_3'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_display_3'   => array(
					'label'    => esc_html__( 'No. of Posts in Display:', 'magazine-power-pro' ),
					'type'     => 'select',
					'default'  => 4,
					'options'  => magazine_power_get_numbers_dropdown_options( 3, 5 ),
				),
				'post_number_3'   => array(
					'label'   => esc_html__( 'No. of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'css'     => 'max-width:60px;',
					'default' => 6,
					'min'     => 1,
					'max'     => 10,
				),
			);

			parent::__construct( 'magazine-power-main-block', esc_html__( 'MP: Main Block', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}
			?>

			<div class="inner-wrapper">
				<?php
				$carousel_args = array(
					'slidesToShow'   => 1,
					'slidesToScroll' => 1,
					'dots'           => false,
					'prevArrow'      => '<span class="left-arrow carousel-arrow"><i class="fas fa-angle-left" aria-hidden="true"></i></span>',
					'nextArrow'      => '<span class="right-arrow carousel-arrow"><i class="fas fa-angle-right" aria-hidden="true"></i></span>',
					'responsive'     => array(
						array(
							'breakpoint' => 1024,
							'settings'   => array(
								'slidesToShow' => 1,
							),
						),
						array(
							'breakpoint' => 768,
							'settings'   => array(
								'slidesToShow' => 1,
							),
						),
						array(
							'breakpoint' => 480,
							'settings'   => array(
								'slidesToShow' => 1,
							),
						),
					),
				);

				$carousel_args_encoded = wp_json_encode( $carousel_args );
				?>

				<div class="main-block-item main-block-slider" >
					<div class="main-block-item-inner magazine-power-carousel" data-slick='<?php echo $carousel_args_encoded; ?>'>
						<?php
						$qargs = array(
							'posts_per_page'      => absint( $params['post_number_1'] ),
							'no_found_rows'       => true,
							'ignore_sticky_posts' => true,
						);

						if ( absint( $params['post_category_1'] ) > 0 ) {
							$qargs['cat'] = absint( $params['post_category_1'] );
						}

						$the_query = new WP_Query( $qargs );
						?>

						<?php if ( $the_query->have_posts() ) : ?>
							<?php
							while ( $the_query->have_posts() ) :
								$the_query->the_post();
								?>
								<div class="main-block-slider-item">
									<div class="main-block-slider-inner">
										<div class="main-block-item-thumb">
											<?php if ( has_post_thumbnail() ) : ?>
												<a href="<?php the_permalink(); ?>" class="images-zoom">
													<?php the_post_thumbnail( 'magazine-power-square' ); ?>
												</a>
											<?php else : ?>
												<a href="<?php the_permalink(); ?>">
													<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
												</a>
											<?php endif; ?>
										</div>
										<div class="main-block-content">
											<?php
											$category = magazine_power_get_single_post_category( get_the_ID() );
											?>
											<?php if ( ! empty( $category ) ) : ?>
												<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
											<?php endif; ?>

											<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
											<div class="entry-meta">
												<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>
												<?php
												if ( comments_open( get_the_ID() ) ) {
													echo '<span class="comments-link">';
													comments_popup_link( '0', '1', '%' );
													echo '</span>';
												}
												?>
											</div><!-- .entry-meta -->
										</div><!-- .main-block-content -->
									</div><!-- .main-block-slider-inner -->
								</div><!-- .main-block-slider-item -->
							<?php endwhile; ?>

							<?php wp_reset_postdata(); ?>
						<?php endif; ?>

					</div><!-- .main-block-item-inner -->
				</div><!-- .main-block-slider -->

				<div class="main-block-item main-block-list">
					<div class="main-block-item-inner">
						<?php
						$qargs = array(
							'posts_per_page'      => 2,
							'no_found_rows'       => true,
							'ignore_sticky_posts' => true,
						);

						if ( absint( $params['post_category_2'] ) > 0 ) {
							$qargs['cat'] = absint( $params['post_category_2'] );
						}

						$the_query = new WP_Query( $qargs );
						?>

						<?php if ( $the_query->have_posts() ) : ?>

							<?php
							while ( $the_query->have_posts() ) :
								$the_query->the_post();
								?>
								<div class="main-block-list-item">
									<div class="block-list-inner">
									<div class="main-block-list-thumb">
										<?php if ( has_post_thumbnail() ) : ?>
											<a href="<?php the_permalink(); ?>" class="images-zoom">
												<?php the_post_thumbnail( 'magazine-power-square' ); ?>
											</a>
										<?php else : ?>
											<a href="<?php the_permalink(); ?>">
												<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
											</a>
										<?php endif; ?>
									</div>
									<div class="main-block-list-content">
										<?php
										$category = magazine_power_get_single_post_category( get_the_ID() );
										?>
										<?php if ( ! empty( $category ) ) : ?>
											<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
										<?php endif; ?>
										<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
										<div class="entry-meta">
											<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>
											<?php
											if ( comments_open( get_the_ID() ) ) {
												echo '<span class="comments-link">';
												comments_popup_link( '0', '1', '%' );
												echo '</span>';
											}
											?>
										</div><!-- .entry-meta -->
									</div><!-- .main-block-list-content -->
									</div><!-- .block-list-inner -->
								</div><!-- main-block-list-item -->

							<?php endwhile; ?>

							<?php wp_reset_postdata(); ?>

						<?php endif; ?>

					</div><!-- .main-block-item-inner -->
				</div><!-- .main-block-list -->

				<div class="main-block-item main-block-carousel">
					<?php
					$carousel_args = array(
						'slidesToShow'   => absint( $params['post_display_3'] ),
						'slidesToScroll' => 1,
						'dots'           => false,
						'vertical'       => true,
						'verticalSwiping' => true,
						'prevArrow'      => '<span class="left-arrow carousel-arrow"><i class="fas fa-angle-up" aria-hidden="true"></i></span>',
						'nextArrow'      => '<span class="right-arrow carousel-arrow"><i class="fas fa-angle-down" aria-hidden="true"></i></span>',
						'responsive'     => array(
							array(
								'breakpoint' => 1024,
								'settings'   => array(
									'slidesToShow' => absint( $params['post_display_3'] ),
								),
							),
							array(
								'breakpoint' => 1050,
								'settings'   => array(
									'slidesToShow' => 3,
								),
							),
							array(
								'breakpoint' => 480,
								'settings'   => array(
									'slidesToShow' => 3,
								),
							),
						),
					);

					$carousel_args_encoded = wp_json_encode( $carousel_args );
					?>

					<div class="main-block-item-inner magazine-power-carousel" data-slick='<?php echo $carousel_args_encoded; ?>'>
						<?php
						$qargs = array(
							'posts_per_page'      => absint( $params['post_number_3'] ),
							'no_found_rows'       => true,
							'ignore_sticky_posts' => true,
						);

						if ( absint( $params['post_category_3'] ) > 0 ) {
							$qargs['cat'] = absint( $params['post_category_3'] );
						}

						$the_query = new WP_Query( $qargs );
						?>

						<?php if ( $the_query->have_posts() ) : ?>
							<?php
							while ( $the_query->have_posts() ) :
								$the_query->the_post();
								?>

								<div class="main-block-carousel-item">
									<div class="main-block-carousel-thumb">
										<?php if ( has_post_thumbnail() ) : ?>
											<a href="<?php the_permalink(); ?>">
												<?php the_post_thumbnail( 'magazine-power-square' ); ?>
											</a>
										<?php else : ?>
											<a href="<?php the_permalink(); ?>">
												<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
											</a>
										<?php endif; ?>
									</div>
									<div class="main-block-carousel-content">
										<?php
										$category = magazine_power_get_single_post_category( get_the_ID() );
										?>
										<?php if ( ! empty( $category ) ) : ?>
											<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
										<?php endif; ?>
										<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
										<div class="entry-meta">
											<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>
										</div><!-- .entry-meta -->
									</div><!-- .main-block-carousel-content -->
								</div><!-- .main-block-carousel-item -->

							<?php endwhile; ?>

							<?php wp_reset_postdata(); ?>
						<?php endif; ?>

					</div><!-- .main-block-item-inner -->
				</div><!-- .main-block-carousel -->

			</div><!-- .inner-wrapper -->

			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Categories_Widget' ) ) :

	/**
	 * Categories Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Categories_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_categories',
				'description'                 => esc_html__( 'Displays category links.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'       => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'browse_text' => array(
					'label'   => esc_html__( 'Browse Text:', 'magazine-power-pro' ),
					'default' => esc_html__( 'Browse:', 'magazine-power-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
				),
				'cat_number'  => array(
					'label'   => esc_html__( 'Number:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
				),
			);

			parent::__construct( 'magazine-power-categories', esc_html__( 'MP: Categories', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$terms = get_terms(
				'category',
				array(
					'orderby'      => 'count',
					'hide_empty'   => true,
					'hierarchical' => false,
					'number'       => absint( $params['cat_number'] ),
				)
			);

			echo '<div class="terms-content-wrapper">';

			if ( ! empty( $params['browse_text'] ) ) {
				echo '<span class="term-heading">' . esc_html( $params['browse_text'] ) . '</span>';
			}

			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				echo '<ul class="terms-list">';

				foreach ( $terms as $term ) {
					echo '<li class="term-list-item"><a href="' . esc_url( get_term_link( $term ) ) . '">' . esc_html( $term->name ) . '</a></li>';
				}

				echo '</ul>';
			}

			echo '</div>';

			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Featured_Page_Widget' ) ) :

	/**
	 * Featured Page Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Featured_Page_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {

			$opts = array(
				'classname'                   => 'magazine_power_widget_featured_page',
				'description'                 => esc_html__( 'Displays single featured Page or Post.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'                    => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'featured_page'            => array(
					'label'            => esc_html__( 'Select Page:', 'magazine-power-pro' ),
					'type'             => 'dropdown-pages',
					'show_option_none' => esc_html__( '&mdash; Select &mdash;', 'magazine-power-pro' ),
				),
				'content_type'             => array(
					'label'   => esc_html__( 'Show Content:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'full',
					'options' => array(
						'excerpt' => esc_html__( 'Excerpt', 'magazine-power-pro' ),
						'full'    => esc_html__( 'Full', 'magazine-power-pro' ),
					),
				),
				'excerpt_length'           => array(
					'label'       => esc_html__( 'Excerpt Length:', 'magazine-power-pro' ),
					'description' => esc_html__( 'Applies when Excerpt is selected in Content option.', 'magazine-power-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 40,
					'min'         => 1,
					'max'         => 400,
				),
				'featured_image'           => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'options' => magazine_power_get_image_sizes_options(),
				),
				'featured_image_alignment' => array(
					'label'   => esc_html__( 'Image Alignment:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'center',
					'options' => magazine_power_get_image_alignment_options(),
				),
			);

			parent::__construct( 'magazine-power-featured-page', esc_html__( 'MP: Featured Page', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}
			?>

			<?php if ( absint( $params['featured_page'] ) > 0 ) : ?>

				<?php
				$qargs = array(
					'p'              => absint( $params['featured_page'] ),
					'posts_per_page' => 1,
					'no_found_rows'  => true,
					'post_type'      => 'page',
				);

				$the_query = new WP_Query( $qargs );
				?>

				<?php if ( $the_query->have_posts() ) : ?>

					<?php
					while ( $the_query->have_posts() ) :
						$the_query->the_post();
						?>

						<div class="featured-page-widget entry-content">
							<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
								<a href="<?php the_permalink(); ?>"><?php the_post_thumbnail( esc_attr( $params['featured_image'] ), array( 'class' => 'align' . esc_attr( $params['featured_image_alignment'] ) ) ); ?></a>
							<?php endif; ?>
							<?php if ( 'excerpt' === $params['content_type'] ) : ?>
								<?php
								$excerpt = magazine_power_the_excerpt( absint( $params['excerpt_length'] ) );
								echo wp_kses_post( wpautop( $excerpt ) );
								?>
							<?php else : ?>
								<?php the_content(); ?>
							<?php endif; ?>

						</div><!-- .featured-page-widget -->

					<?php endwhile; ?>

					<?php wp_reset_postdata(); ?>

				<?php endif; // End if have_posts(). ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Latest_News_Widget' ) ) :

	/**
	 * Latest News Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Latest_News_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_latest_news',
				'description'                 => esc_html__( 'Displays latest posts in grid.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'          => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_layout'    => array(
					'label'    => esc_html__( 'Post Layout:', 'magazine-power-pro' ),
					'type'     => 'select',
					'default'  => 1,
					'adjacent' => true,
					'options'  => magazine_power_get_numbers_dropdown_options( 1, 2, esc_html__( 'Layout', 'magazine-power-pro' ) . ' ' ),
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 10,
				),
				'post_column'    => array(
					'label'   => esc_html__( 'Number of Columns:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 2,
					'options' => magazine_power_get_numbers_dropdown_options( 1, 4 ),
				),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'magazine-power-thumb',
					'options' => magazine_power_get_image_sizes_options(),
				),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'magazine-power-pro' ),
					'description' => esc_html__( 'in words', 'magazine-power-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 20,
					'min'         => 0,
					'max'         => 200,
				),
			);

			parent::__construct( 'magazine-power-latest-news', esc_html__( 'MP: Latest News', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$qargs = array(
				'posts_per_page'      => absint( $params['post_number'] ),
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );
			?>

			<?php if ( $the_query->have_posts() ) : ?>

				<div class="latest-news-widget latest-news-layout-<?php echo esc_attr( $params['post_layout'] ); ?> latest-news-col-<?php echo esc_attr( $params['post_column'] ); ?>">

					<div class="inner-wrapper">

						<?php
						while ( $the_query->have_posts() ) :
							$the_query->the_post();
							?>

							<div class="latest-news-item">
							<div class="latest-news-inner">
								<?php if ( 'disable' !== $params['featured_image'] ) : ?>
									<?php if ( has_post_thumbnail() ) : ?>
										<div class="latest-news-thumb">
											<a class="images-zoom" href="<?php the_permalink(); ?>">
												<?php the_post_thumbnail( esc_attr( $params['featured_image'] ) ); ?>
											</a>
										</div><!-- .latest-news-thumb -->
									<?php else : ?>
										<div class="latest-news-thumb">
											<a class="images-zoom" href="<?php the_permalink(); ?>">
												<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
											</a>
										</div><!-- .latest-news-thumb -->
									<?php endif; ?>
								<?php endif; ?>
								<div class="latest-news-text-wrap">

									<div class="latest-news-text-content">
										<h3 class="latest-news-title">
											<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
										</h3><!-- .latest-news-title -->
									</div><!-- .latest-news-text-content -->

									<div class="latest-news-meta entry-meta">
											<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>
											<?php
											if ( comments_open( get_the_ID() ) ) {
												echo '<span class="comments-link">';
												comments_popup_link( '0', '1', '%' );
												echo '</span>';
											}
											?>
									</div><!-- .latest-news-meta -->

									<?php if ( absint( $params['excerpt_length'] ) > 0 ) : ?>
										<div class="latest-news-excerpt">
											<?php
											$excerpt = magazine_power_the_excerpt( absint( $params['excerpt_length'] ) );
											echo wp_kses_post( wpautop( $excerpt ) );
											?>
											<a href="<?php the_permalink(); ?>" class="more-link"><?php esc_html_e( 'Read More', 'magazine-power-pro' ); ?></a>
										</div><!-- .latest-news-excerpt -->
									<?php endif; ?>

								</div><!-- .latest-news-text-wrap -->
								</div><!-- .latest-news-inner -->
							</div><!-- .latest-news-item -->

						<?php endwhile; ?>

					</div><!-- .inner-wrapper -->

				</div><!-- .latest-news-widget -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Recent_Posts_Widget' ) ) :

	/**
	 * Recent Posts Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Recent_Posts_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {

			$opts = array(
				'classname'                   => 'magazine_power_widget_recent_posts',
				'description'                 => esc_html__( 'Displays recent posts.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'          => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 100,
				),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'thumbnail',
					'options' => magazine_power_get_image_sizes_options( true, array( 'disable', 'thumbnail' ), false ),
				),
				'image_width'    => array(
					'label'       => esc_html__( 'Image Width:', 'magazine-power-pro' ),
					'type'        => 'number',
					'description' => esc_html__( 'px', 'magazine-power-pro' ),
					'css'         => 'max-width:60px;',
					'adjacent'    => true,
					'default'     => 80,
					'min'         => 1,
					'max'         => 150,
				),
				'disable_date'   => array(
					'label'   => esc_html__( 'Disable Date', 'magazine-power-pro' ),
					'type'    => 'checkbox',
					'default' => false,
				),
			);

			parent::__construct( 'magazine-power-recent-posts', esc_html__( 'MP: Recent Posts', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$qargs = array(
				'posts_per_page'      => absint( $params['post_number'] ),
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );
			?>
			<?php if ( $the_query->have_posts() ) : ?>

				<div class="recent-posts-wrapper">

					<?php
					while ( $the_query->have_posts() ) :
						$the_query->the_post();
						?>

						<div class="recent-posts-item">

							<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
								<div class="recent-posts-thumb">
									<a href="<?php the_permalink(); ?>">
										<?php
										$img_attributes = array(
											'class' => 'alignleft',
											'style' => 'max-width:' . esc_attr( $params['image_width'] ) . 'px;',
										);
										the_post_thumbnail( esc_attr( $params['featured_image'] ), $img_attributes );
										?>
									</a>
								</div><!-- .recent-posts-thumb -->
							<?php endif ?>
							<div class="recent-posts-text-wrap">
								<h3 class="recent-posts-title">
									<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
								</h3>

								<?php if ( false === $params['disable_date'] ) : ?>
									<div class="recent-posts-meta entry-meta">

										<?php if ( false === $params['disable_date'] ) : ?>
											<span class="posted-on"><?php the_time( 'j M Y' ); ?></span><!-- .recent-posts-date -->
										<?php endif; ?>

									</div><!-- .recent-posts-meta -->
								<?php endif; ?>

							</div><!-- .recent-posts-text-wrap -->

						</div><!-- .recent-posts-item -->

					<?php endwhile; ?>

				</div><!-- .recent-posts-wrapper -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_News_Block_Widget' ) ) :

	/**
	 * News Block Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_News_Block_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_news_block',
				'description'                 => esc_html__( 'Displays news block.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'         => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category' => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
			);

			parent::__construct( 'magazine-power-news-block', esc_html__( 'MP: News Block', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$qargs = array(
				'posts_per_page'      => 5,
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );

			?>
			<?php if ( $the_query->have_posts() ) : ?>

				<div class="news-block-widget">
					<div class="inner-wrapper">

						<?php
						while ( $the_query->have_posts() ) :
							$the_query->the_post();
							?>

							<?php
							$current_count = $the_query->current_post;
							$is_special    = ( $current_count < 2 ) ? true : false;

							$extra_class = ( true === $is_special ) ? 'item-half' : '';
							?>
							<div class="block-item <?php echo esc_attr( $extra_class ); ?>">
								<div class="block-item-inner">
									<div class="block-item-thumb">
										<a class="images-zoom" href="<?php the_permalink(); ?>">
											<?php if ( has_post_thumbnail() ) : ?>
												<?php the_post_thumbnail( 'magazine-power-large' ); ?>
											<?php else : ?>
												<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
											<?php endif; ?>
										</a>
									</div><!-- .block-item-thumb -->
								<div class="block-content">
									<?php
									$category = magazine_power_get_single_post_category( get_the_ID() );
									?>
									<?php if ( ! empty( $category ) ) : ?>
										<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
									<?php endif; ?>
									<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
									<div class="entry-meta">
										<span class="posted-on"><?php the_time( get_option( 'date_format' ) ); ?></span>
										<?php
										if ( comments_open( get_the_ID() ) ) {
											echo '<span class="comments-link">';
											comments_popup_link( '0', '1', '%' );
											echo '</span>';
										}
										?>
									</div><!-- .entry-meta -->
								</div><!-- .block-content -->
								</div><!-- .block-item-inner -->
							</div><!-- .block-item -->
						<?php endwhile; ?>

					</div><!-- .inner-wrapper -->
				</div><!-- .news-block-widget -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>

			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_News_Slider_Widget' ) ) :

	/**
	 * News Slider Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_News_Slider_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'   => 'magazine_power_widget_news_slider',
				'description' => esc_html__( 'Displays news slider', 'magazine-power-pro' ),
			);

			$fields = array(
				'title'          => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 3,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 10,
				),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'magazine-power-large',
					'options' => magazine_power_get_image_sizes_options( false ),
				),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'magazine-power-pro' ),
					'description' => esc_html__( 'in words', 'magazine-power-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 15,
					'min'         => 0,
					'max'         => 200,
				),
			);

			parent::__construct( 'magazine-power-news-slider', esc_html__( 'MP: News Slider', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$this->render_slider( $params );

			echo $args['after_widget'];
		}

		/**
		 * Render slider.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		protected function render_slider( $params ) {
			$carousel_args = array(
				'slidesToShow'   => 1,
				'slidesToScroll' => 1,
				'dots'           => false,
				'prevArrow'      => '<span class="left-arrow carousel-arrow"><i class="fas fa-angle-left" aria-hidden="true"></i></span>',
				'nextArrow'      => '<span class="right-arrow carousel-arrow"><i class="fas fa-angle-right" aria-hidden="true"></i></span>',
				'responsive'     => array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow' => 1,
						),
					),
					array(
						'breakpoint' => 768,
						'settings'   => array(
							'slidesToShow' => 1,
						),
					),
					array(
						'breakpoint' => 480,
						'settings'   => array(
							'slidesToShow' => 1,
						),
					),
				),
			);

			$carousel_args_encoded = wp_json_encode( $carousel_args );

			$qargs = array(
				'posts_per_page'      => absint( $params['post_number'] ),
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );
			?>

			<?php if ( $the_query->have_posts() ) : ?>

				<div class="magazine-power-carousel" data-slick='<?php echo $carousel_args_encoded; ?>'>

					<?php
					while ( $the_query->have_posts() ) :
						$the_query->the_post();
						?>

						<article class="carousel-item">
							<div class="carousel-item-inner">
								<div class="carousel-item-thumb">
									<?php if ( has_post_thumbnail() ) : ?>
										<a class="images-zoom" href="<?php the_permalink(); ?>"><?php the_post_thumbnail( esc_attr( $params['featured_image'] ) ); ?></a>
									<?php else : ?>
										<a href="<?php the_permalink(); ?>"><img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
									<?php endif; ?>
								</div><!-- .carousel-item-thumb -->
								<div class="carousel-item-content">
									<?php
									$category = magazine_power_get_single_post_category( get_the_ID() );
									?>
									<?php if ( ! empty( $category ) ) : ?>
										<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
									<?php endif; ?>

									<h3 class="carousel-item-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
									<div class="carousel-item-meta entry-meta">
										<span class="posted-on"><?php the_time( get_option( 'date_format' ) ); ?></span>
										<?php
										if ( comments_open( get_the_ID() ) ) {
											echo '<span class="comments-link">';
											comments_popup_link( '0', '1', '%' );
											echo '</span>';
										}
										?>
									</div><!-- .carousel-item-meta -->
									<?php if ( absint( $params['excerpt_length'] ) > 0 ) : ?>
										<div class="carousel-item-excerpt">
											<?php
											$excerpt = magazine_power_the_excerpt( absint( $params['excerpt_length'] ) );
											echo wp_kses_post( wpautop( $excerpt ) );
											?>
										</div><!-- .carousel-item-excerpt -->
									<?php endif; ?>
								</div><!-- .carousel-item-content -->
							</div><!-- .carousel-item-inner -->
						</article><!-- .carousel-item -->

					<?php endwhile; ?>

					<?php wp_reset_postdata(); ?>

				</div><!-- .magazine-power-carousel -->

			<?php endif; ?>
			<?php
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_News_Carousel_Widget' ) ) :

	/**
	 * News Carousel Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_News_Carousel_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'   => 'magazine_power_widget_news_carousel',
				'description' => esc_html__( 'Displays news carousel', 'magazine-power-pro' ),
			);

			$fields = array(
				'title'          => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 6,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 10,
				),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'magazine-power-large',
					'options' => magazine_power_get_image_sizes_options( false ),
				),
			);

			parent::__construct( 'magazine-power-news-carousel', esc_html__( 'MP: News Carousel', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$this->render_carousel( $params );

			echo $args['after_widget'];
		}

		/**
		 * Render carousel.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		protected function render_carousel( $params ) {
			$carousel_args = array(
				'slidesToShow'   => 3,
				'slidesToScroll' => 1,
				'dots'           => false,
				'prevArrow'      => '<span class="left-arrow carousel-arrow"><i class="fas fa-angle-left" aria-hidden="true"></i></span>',
				'nextArrow'      => '<span class="right-arrow carousel-arrow"><i class="fas fa-angle-right" aria-hidden="true"></i></span>',
				'responsive'     => array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow' => 3,
						),
					),
					array(
						'breakpoint' => 768,
						'settings'   => array(
							'slidesToShow' => 2,
						),
					),
					array(
						'breakpoint' => 480,
						'settings'   => array(
							'slidesToShow' => 1,
						),
					),
				),
			);

			$carousel_args_encoded = wp_json_encode( $carousel_args );

			$qargs = array(
				'posts_per_page'      => absint( $params['post_number'] ),
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			);

			if ( absint( $params['post_category'] ) > 0 ) {
				$qargs['cat'] = absint( $params['post_category'] );
			}

			$the_query = new WP_Query( $qargs );
			?>

			<?php if ( $the_query->have_posts() ) : ?>

				<div class="magazine-power-carousel" data-slick='<?php echo $carousel_args_encoded; ?>'>

					<?php
					while ( $the_query->have_posts() ) :
						$the_query->the_post();
						?>
						<article class="carousel-item">
							<div class="carousel-item-inner">
								<div class="carousel-item-thumb">
									<a class="images-zoom" href="<?php the_permalink(); ?>"><?php if ( has_post_thumbnail() ) : ?>
										<?php the_post_thumbnail( 'large' ); ?>
									<?php else : ?>
										<img src="<?php echo esc_url( get_template_directory_uri() . '/images/no-image-large.png' ); ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
									<?php endif; ?>
									</a>
									<div class="carousel-item-content">
										<?php
										$category = magazine_power_get_single_post_category( get_the_ID() );
										?>
										<?php if ( ! empty( $category ) ) : ?>
											<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
										<?php endif; ?>

										<h3 class="carousel-item-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
										<div class="carousel-item-meta entry-meta">
											<span class="posted-on"><?php the_time( get_option( 'date_format' ) ); ?></span>
											<?php
											if ( comments_open( get_the_ID() ) ) {
												echo '<span class="comments-link">';
												comments_popup_link( '0', '1', '%' );
												echo '</span>';
											}
											?>
										</div><!-- .carousel-item-meta -->
									</div><!-- .carousel-item-content -->
								</div><!-- .carousel-item-thumb -->
							</div><!-- .carousel-item-inner -->
						</article><!-- .carousel-item -->

					<?php endwhile; ?>

					<?php wp_reset_postdata(); ?>
				</div><!-- .magazine-power-carousel -->
			<?php endif; ?>
			<?php
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Tabbed_Widget' ) ) :

	/**
	 * Tabbed Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Tabbed_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'   => 'magazine_power_widget_tabbed',
				'description' => esc_html__( 'Tabbed widget.', 'magazine-power-pro' ),
			);

			$fields = array(
				'popular_heading'  => array(
					'label' => esc_html__( 'Popular', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'popular_number'   => array(
					'label'   => esc_html__( 'No. of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'css'     => 'max-width:60px;',
					'default' => 5,
					'min'     => 1,
					'max'     => 10,
				),
				'recent_heading'   => array(
					'label' => esc_html__( 'Recent', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'recent_number'    => array(
					'label'   => esc_html__( 'No. of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'css'     => 'max-width:60px;',
					'default' => 5,
					'min'     => 1,
					'max'     => 10,
				),
				'comments_heading' => array(
					'label' => esc_html__( 'Comments', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'comments_number'  => array(
					'label'   => esc_html__( 'No. of Comments:', 'magazine-power-pro' ),
					'type'    => 'number',
					'css'     => 'max-width:60px;',
					'default' => 5,
					'min'     => 1,
					'max'     => 10,
				),
			);

			parent::__construct( 'magazine-power-tabbed', esc_html__( 'MP: Tabbed', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );
			$tab_id = 'tabbed-' . $this->number;

			echo $args['before_widget'];
			?>
			<div class="tabbed-container">
				<ul class="etabs">
					<li class="tab tab-popular"><a href="#<?php echo esc_attr( $tab_id ); ?>-popular"><?php esc_html_e( 'Popular', 'magazine-power-pro' ); ?></a></li>
					<li class="tab tab-recent"><a href="#<?php echo esc_attr( $tab_id ); ?>-recent"><?php esc_html_e( 'Recent', 'magazine-power-pro' ); ?></a></li>
					<li class="tab tab-comments"><a href="#<?php echo esc_attr( $tab_id ); ?>-comments"><?php esc_html_e( 'Comments', 'magazine-power-pro' ); ?></a></li>
				</ul>
				<div id="<?php echo esc_attr( $tab_id ); ?>-popular" class="tab-content">
					<?php $this->render_news( 'popular', $params ); ?>
				</div>
				<div id="<?php echo esc_attr( $tab_id ); ?>-recent" class="tab-content">
					<?php $this->render_news( 'recent', $params ); ?>
				</div>
				<div id="<?php echo esc_attr( $tab_id ); ?>-comments" class="tab-content">
					<?php $this->render_comments( $params ); ?>
				</div>
			</div><!-- .tabbed-container -->
			<?php
			echo $args['after_widget'];
		}

		/**
		 * Render news.
		 *
		 * @since 1.0.0
		 *
		 * @param array $type Type.
		 * @param array $params Parameters.
		 * @return void
		 */
		protected function render_news( $type, $params ) {
			if ( ! in_array( $type, array( 'popular', 'recent' ), true ) ) {
				return;
			}

			switch ( $type ) {
				case 'popular':
					$qargs = array(
						'posts_per_page'      => absint( $params['popular_number'] ),
						'no_found_rows'       => true,
						'ignore_sticky_posts' => true,
						'orderby'             => 'comment_count',
					);
					break;

				case 'recent':
					$qargs = array(
						'posts_per_page'      => absint( $params['recent_number'] ),
						'no_found_rows'       => true,
						'ignore_sticky_posts' => true,
					);
					break;

				default:
					break;
			}

			$the_query = new WP_Query( $qargs );
			?>

			<?php if ( $the_query->have_posts() ) : ?>

				<ul class="news-list">
					<?php
					while ( $the_query->have_posts() ) :
						$the_query->the_post();
						?>
						<li class="news-item">
							<div class="news-thumb">
								<a class="news-item-thumb" href="<?php the_permalink(); ?>">
								<?php if ( has_post_thumbnail() ) : ?>
									<?php $image = wp_get_attachment_image_src( get_post_thumbnail_id() ); ?>
									<?php if ( ! empty( $image ) ) : ?>
										<img src="<?php echo esc_url( $image[0] ); ?>" alt="<?php the_title_attribute(); ?>" />
									<?php endif; ?>
								<?php else : ?>
									<img src="<?php echo get_template_directory_uri() . '/images/no-image-65.png'; ?>" alt="<?php esc_attr_e( 'No Image', 'magazine-power-pro' ); ?>" />
								<?php endif; ?>
								</a><!-- .news-item-thumb -->
							</div><!-- .news-thumb -->
							<div class="news-content">
								<h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
								<div class="entry-meta"><span class="posted-on"><?php the_time( get_option( 'date_format' ) ); ?></span>
								</div>
							</div><!-- .news-content -->
						</li><!-- .news-item -->
					<?php endwhile; ?>
				</ul><!-- .news-list -->

				<?php wp_reset_postdata(); ?>

			<?php endif; ?>
			<?php
		}

		/**
		 * Render comments.
		 *
		 * @since 1.0.0
		 *
		 * @param array $params Parameters.
		 * @return void
		 */
		protected function render_comments( $params ) {
			$comment_args = array(
				'number'      => absint( $params['comments_number'] ),
				'status'      => 'approve',
				'post_status' => 'publish',
			);

			$comments = get_comments( $comment_args );
			?>

			<?php if ( ! empty( $comments ) ) : ?>
				<ul class="comments-list">
					<?php foreach ( $comments as $key => $comment ) : ?>
						<li>
							<div class="comments-thumb">
								<?php $comment_author_url = get_comment_author_url( $comment ); ?>
								<?php if ( ! empty( $comment_author_url ) ) : ?>
									<a href="<?php echo esc_url( $comment_author_url ); ?>"><?php echo get_avatar( $comment, 65 ); ?></a>
								<?php else : ?>
									<?php echo get_avatar( $comment, 65 ); ?>
								<?php endif; ?>
							</div><!-- .comments-thumb -->
							<div class="comments-content">
								<?php echo get_comment_author_link( $comment ); ?>&nbsp;<?php echo esc_html_x( 'on', 'Tabbed Widget', 'magazine-power-pro' ); ?>&nbsp;<a href="<?php echo esc_url( get_comment_link( $comment ) ); ?>"><?php echo get_the_title( $comment->comment_post_ID ); ?></a>
							</div><!-- .comments-content -->
						</li>
					<?php endforeach; ?>
				</ul><!-- .comments-list -->
			<?php endif; ?>
			<?php
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Categorized_News_Widget' ) ) :

	/**
	 * Categorized News Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Categorized_News_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_categorized_news',
				'description'                 => esc_html__( 'Displays categorized news.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'          => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_category'  => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'post_layout'    => array(
					'label'    => esc_html__( 'Post Layout:', 'magazine-power-pro' ),
					'type'     => 'select',
					'default'  => 1,
					'adjacent' => true,
					'options'  => magazine_power_get_numbers_dropdown_options( 1, 2, esc_html__( 'Layout', 'magazine-power-pro' ) . ' ' ),
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 4,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 10,
				),
				'major_heading'  => array(
					'label' => esc_html__( 'Main Block', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'featured_image' => array(
					'label'   => esc_html__( 'Featured Image:', 'magazine-power-pro' ),
					'type'    => 'select',
					'default' => 'magazine-power-thumb',
					'options' => magazine_power_get_image_sizes_options(),
				),
				'excerpt_length' => array(
					'label'       => esc_html__( 'Excerpt Length:', 'magazine-power-pro' ),
					'description' => esc_html__( 'in words', 'magazine-power-pro' ),
					'type'        => 'number',
					'css'         => 'max-width:60px;',
					'default'     => 20,
					'min'         => 0,
					'max'         => 200,
				),
			);

			parent::__construct( 'magazine-power-categorized-news', esc_html__( 'MP: Categorized News', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$major_post_id = null;
			?>
			<div class="categorized-news-widget categorized-news-layout-<?php echo esc_attr( $params['post_layout'] ); ?>">

				<div class="inner-wrapper">
					<div class="categorized-major">
						<?php
						$qargs = array(
							'posts_per_page'      => 1,
							'no_found_rows'       => true,
							'ignore_sticky_posts' => true,
						);

						if ( absint( $params['post_category'] ) > 0 ) {
							$qargs['cat'] = absint( $params['post_category'] );
						}

						$the_query = new WP_Query( $qargs );
						?>

						<?php if ( $the_query->have_posts() ) : ?>

							<?php
							while ( $the_query->have_posts() ) :
								$the_query->the_post();
								?>
								<?php $major_post_id = get_the_ID(); ?>

								<div class="categorized-news-item">
									<div class="categorize-news-item">
									<?php if ( 'disable' !== $params['featured_image'] && has_post_thumbnail() ) : ?>
										<div class="categorized-news-thumb">
											<a class="images-zoom" href="<?php the_permalink(); ?>">
												<?php the_post_thumbnail( esc_attr( $params['featured_image'] ) ); ?>
											</a>
										</div><!-- .categorized-news-thumb -->
									<?php endif; ?>
									<div class="categorized-news-text-wrap">
										<div class="categorized-news-text-content">
											<h3 class="categorized-news-title">
												<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
											</h3><!-- .categorized-news-title -->
										</div><!-- .categorized-news-text-content -->
												<div class="categorized-news-meta entry-meta">
												<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>

											</div><!-- .categorized-news-meta -->
										<?php if ( absint( $params['excerpt_length'] ) > 0 ) : ?>
											<div class="categorized-news-excerpt">
												<?php
												$excerpt = magazine_power_the_excerpt( absint( $params['excerpt_length'] ) );
												echo wp_kses_post( wpautop( $excerpt ) );
												?>
											</div><!-- .categorized-news-excerpt -->
										<?php endif; ?>
									</div><!-- .categorized-news-text-wrap -->
									</div><!-- .categorize-news-item" -->
								</div><!-- .categorized-news-item -->
							<?php endwhile; ?>

							<?php wp_reset_postdata(); ?>
						<?php endif; ?>
					</div><!-- .categorized-major -->

					<div class="categorized-minor">
						<div class="categorized-minor-news-wrapper">
							<?php
							$qargs = array(
								'posts_per_page'      => absint( $params['post_number'] ),
								'no_found_rows'       => true,
								'ignore_sticky_posts' => true,
							);

							if ( $major_post_id ) {
								$qargs['post__not_in']   = array( $major_post_id );
								$qargs['posts_per_page'] = absint( $params['post_number'] - 1 );
							}

							if ( absint( $params['post_category'] ) > 0 ) {
								$qargs['cat'] = absint( $params['post_category'] );
							}

							$the_query = new WP_Query( $qargs );
							?>

							<?php if ( $the_query->have_posts() ) : ?>

								<?php
								while ( $the_query->have_posts() ) :
									$the_query->the_post();
									?>
									<div class="categorized-news-item">
										<?php if ( has_post_thumbnail() ) : ?>
											<div class="categorized-news-thumb">
												<a href="<?php the_permalink(); ?>"><?php the_post_thumbnail( 'medium' ); ?></a>
											</div><!-- .categorized-news-thumb -->
										<?php endif; ?>
										<div class="categorized-news-text-wrap">
											<div class="categorized-news-text-content">
												<h3 class="categorized-news-title">
													<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
												</h3><!-- .categorized-news-title -->
											</div><!-- .categorized-news-text-content -->
											<div class="categorized-news-meta entry-meta">
												<span class="posted-on"><?php the_time( 'j M Y' ); ?></span>

											</div><!-- .categorized-news-meta -->
										</div><!-- .categorized-news-text-wrap -->
									</div><!-- .categorized-news-item -->
								<?php endwhile; ?>

								<?php wp_reset_postdata(); ?>
							<?php endif; ?>

						</div><!-- .categorized-minor-news-wrapper -->
					</div><!-- .categorized-minor -->

				</div><!-- .inner-wrapper -->

			</div><!-- .categorized-news-widget -->
			<?php
			echo $args['after_widget'];
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Categorized_Columns_Widget' ) ) :

	/**
	 * Categorized Columns Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Categorized_Columns_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_categorized_columns',
				'description'                 => esc_html__( 'Displays categorized news in columns.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'           => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'post_number'    => array(
					'label'   => esc_html__( 'Number of Posts:', 'magazine-power-pro' ),
					'type'    => 'number',
					'default' => 5,
					'css'     => 'max-width:60px;',
					'min'     => 1,
					'max'     => 10,
				),
				'block1_heading'  => array(
					'label' => esc_html__( 'Block 1', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'block1_title'    => array(
					'label' => esc_html__( 'Block Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'block1_category' => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'block1_enable'   => array(
					'label'   => esc_html__( 'Enable Block', 'magazine-power-pro' ),
					'type'    => 'checkbox',
					'default' => true,
				),
				'block2_heading'  => array(
					'label' => esc_html__( 'Block 2', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'block2_title'    => array(
					'label' => esc_html__( 'Block Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'block2_category' => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'block2_enable'   => array(
					'label'   => esc_html__( 'Enable Block', 'magazine-power-pro' ),
					'type'    => 'checkbox',
					'default' => true,
				),
				'block3_heading'  => array(
					'label' => esc_html__( 'Block 2', 'magazine-power-pro' ),
					'type'  => 'heading',
				),
				'block3_title'    => array(
					'label' => esc_html__( 'Block Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'block3_category' => array(
					'label'           => esc_html__( 'Select Category:', 'magazine-power-pro' ),
					'type'            => 'dropdown-taxonomies',
					'show_option_all' => esc_html__( 'All Categories', 'magazine-power-pro' ),
				),
				'block3_enable'   => array(
					'label'   => esc_html__( 'Enable Block', 'magazine-power-pro' ),
					'type'    => 'checkbox',
					'default' => false,
				),
			);

			parent::__construct( 'magazine-power-categorized-columns', esc_html__( 'MP: Categorized Columns', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}

			$blocks_details = array();

			for ( $i = 1; $i <= 3; $i++ ) {
				$item = array();

				$status = $params[ "block{$i}_enable" ];

				if ( true === $status ) {
					$item['title']    = $params[ "block{$i}_title" ];
					$item['category'] = $params[ "block{$i}_category" ];
					$blocks_details[] = $item;
				}
			}

			$column_number = absint( count( $blocks_details ) );

			if ( $column_number > 0 ) {
				echo '<div class="categorized-column-' . absint( $column_number ) . '"> <div class="inner-wrapper">';
				foreach ( $blocks_details as $block ) {
					$this->render_block( $block, $params );
				}
				echo '</div></div>';
			}

			echo $args['after_widget'];
		}

		/**
		 * Render block.
		 *
		 * @since 1.0.0
		 *
		 * @param array $block  Block detail.
		 * @param array $params Parameters.
		 * @return void
		 */
		protected function render_block( $block, $params ) {
			?>
				<div class="categorized-wrapper categorized-column">
					<div class="categorized-column-wrapper">
						<?php if ( ! empty( $block['title'] ) ) : ?>
							<h2 class="widget-title"><?php echo esc_html( $block['title'] ); ?></h2>
						<?php endif; ?>

						<?php
						$qargs = array(
							'posts_per_page'      => absint( $params['post_number'] ),
							'no_found_rows'       => true,
							'ignore_sticky_posts' => true,
						);

						if ( absint( $block['category'] ) > 0 ) {
							$qargs['cat'] = absint( $block['category'] );
						}

						$the_query = new WP_Query( $qargs );
						?>

						<?php if ( $the_query->have_posts() ) : ?>

							<?php
							while ( $the_query->have_posts() ) :
								$the_query->the_post();
								?>

								<?php
								$first_class = ( 0 === $the_query->current_post ) ? 'first-item' : '';
								$image_size  = ( 0 === $the_query->current_post ) ? 'magazine-power-large' : 'thumbnail';
								?>

								<div class="categorized-columns-item <?php echo esc_attr( $first_class ); ?>">
									<div class="categorized-columns-inner">
										<div class="categorized-columns-thumb">
											<a class="images-zoom" href="<?php the_permalink(); ?>">
												<?php the_post_thumbnail( $image_size ); ?>
											</a>
										</div><!-- .categorized-columns-thumb -->
										<div class="categorized-columns-content">

												<?php if ( 0 === $the_query->current_post ) : ?>
													<?php
													$category = magazine_power_get_single_post_category( get_the_ID() );
													?>
													<?php if ( ! empty( $category ) ) : ?>
														<span class="cat-links"><a href="<?php echo esc_url( $category['url'] ); ?>"><?php echo esc_html( $category['name'] ); ?></a></span>
													<?php endif; ?>
												<?php endif; ?>
											<h3 class="categorized-columns-item-title">
												<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
											</h3><!-- .categorized-columns-item-title -->

											<div class="categorized-columns-item-meta entry-meta">
												<span class="posted-on"><?php the_time( get_option( 'date_format' ) ); ?></span>
												<?php
												if ( comments_open( get_the_ID() ) ) {
													echo '<span class="comments-link">';
													comments_popup_link( '0', '1', '%' );
													echo '</span>';
												}
												?>
											</div><!-- .categorized-columns-item-meta -->
										</div><!-- .categorized-columns-content -->
									</div><!-- .categorized-columns-inner -->
								</div><!-- .categorized-columns-item -->

							<?php endwhile; ?>
							<?php wp_reset_postdata(); ?>
						<?php endif; ?>
					</div><!-- .tegorized-column-wrapper -->
				</div><!-- .categorized-column -->
			<?php
		}
	}

endif;

if ( ! class_exists( 'Magazine_Power_Call_To_Action_Widget' ) ) :

	/**
	 * Call To Action Widget Class.
	 *
	 * @since 1.0.0
	 */
	class Magazine_Power_Call_To_Action_Widget extends Magazine_Power_Widget_Base {

		/**
		 * Sets up a new widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$opts = array(
				'classname'                   => 'magazine_power_widget_call_to_action',
				'description'                 => esc_html__( 'Call To Action Widget.', 'magazine-power-pro' ),
				'customize_selective_refresh' => true,
			);

			$fields = array(
				'title'                 => array(
					'label' => esc_html__( 'Title:', 'magazine-power-pro' ),
					'type'  => 'text',
					'class' => 'widefat',
				),
				'description'           => array(
					'label' => esc_html__( 'Description:', 'magazine-power-pro' ),
					'type'  => 'textarea',
					'class' => 'widefat',
				),
				'primary_button_text'   => array(
					'label'   => esc_html__( 'Primary Button Text:', 'magazine-power-pro' ),
					'default' => esc_html__( 'Learn more', 'magazine-power-pro' ),
					'type'    => 'text',
					'class'   => 'widefat',
				),
				'primary_button_url'    => array(
					'label' => esc_html__( 'Primary Button URL:', 'magazine-power-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
				),
				'secondary_button_text' => array(
					'label'   => esc_html__( 'Secondary Button Text:', 'magazine-power-pro' ),
					'default' => '',
					'type'    => 'text',
					'class'   => 'widefat',
				),
				'secondary_button_url'  => array(
					'label' => esc_html__( 'Secondary Button URL:', 'magazine-power-pro' ),
					'type'  => 'url',
					'class' => 'widefat',
				),
			);

			parent::__construct( 'magazine-power-call-to-action', esc_html__( 'MP: Call To Action', 'magazine-power-pro' ), $opts, array(), $fields );
		}

		/**
		 * Outputs the content for the current widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args     Display arguments.
		 * @param array $instance Settings for the current widget instance.
		 */
		public function widget( $args, $instance ) {
			$params = $this->get_params( $instance );

			echo $args['before_widget'];

			if ( ! empty( $params['title'] ) ) {
				echo $args['before_title'] . esc_html( $params['title'] ) . $args['after_title'];
			}
			?>
			<div class="call-to-action-content">
				<?php if ( ! empty( $params['description'] ) ) : ?>
					<div class="call-to-action-description">
						<?php echo wp_kses_post( wpautop( $params['description'] ) ); ?>
					</div><!-- .call-to-action-description -->
				<?php endif; ?>
				<?php if ( ! empty( $params['primary_button_text'] ) || ! empty( $params['secondary_button_text'] ) ) : ?>
					<div class="call-to-action-buttons">
						<?php if ( ! empty( $params['primary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['primary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-primary"><?php echo esc_html( $params['primary_button_text'] ); ?></a>
						<?php endif; ?>
						<?php if ( ! empty( $params['secondary_button_text'] ) ) : ?>
							<a href="<?php echo esc_url( $params['secondary_button_url'] ); ?>" class="custom-button btn-call-to-action btn-call-to-secondary"><?php echo esc_html( $params['secondary_button_text'] ); ?></a>
						<?php endif; ?>
					</div><!-- .call-to-action-buttons -->
				<?php endif; ?>
			</div><!-- .call-to-action-content -->
			<?php
			echo $args['after_widget'];
		}
	}

endif;
