<?php

if( ! function_exists( 'wen_business_sanitize_select' ) ) :

  /**
   * Sanitize select
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_select( $input, $setting ) {

    // Ensure input is a slug
    $input = sanitize_key( $input );

    // Get list of choices from the control associated with the setting
    $choices = $setting->manager->get_control( $setting->id )->choices;

    // If the input is a valid key, return it; otherwise, return the default
    return ( array_key_exists( $input, $choices ) ? $input : $setting->default );

  }

endif;


if( ! function_exists( 'wen_business_sanitize_checkbox' ) ) :

  /**
   * Sanitize checkbox
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_checkbox( $checked ) {

    return ( ( isset( $checked ) && true == $checked ) ? true : false );

  }

endif;

if( ! function_exists( 'wen_business_sanitize_positive_integer' ) ) :

  /**
   * Sanitize positive integer
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_positive_integer( $input, $setting ) {

    $input = absint( $input );

    // If the input is an absolute integer, return it;
    // otherwise, return the default
    return ( $input ? $input : $setting->default );

  }

endif;

if( ! function_exists( 'wen_business_sanitize_number_range' ) ) :

  /**
   * Sanitize number range
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_number_range( $input, $setting ) {

    // Ensure input is an absolute integer
    $input = absint( $input );

    // Get the input attributes
    // associated with the setting
    $atts = $setting->manager->get_control( $setting->id )->input_attrs;

    // Get min
    $min = ( isset( $atts['min'] ) ? $atts['min'] : $input );

    // Get max
    $max = ( isset( $atts['max'] ) ? $atts['max'] : $input );

    // Get Step
    $step = ( isset( $atts['step'] ) ? $atts['step'] : 1 );

    // If the input is within the valid range,
    // return it; otherwise, return the default
    return ( $min <= $input && $input <= $max && is_int( $input / $step ) ? $input : $setting->default );

  }

endif;

if( ! function_exists( 'wen_business_sanitize_footer_content' ) ) :

  /**
   * Sanitize footer content
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_footer_content( $input ) {

    return ( stripslashes( wp_filter_post_kses( addslashes ( $input ) ) ) );

  }

endif;


if( ! function_exists( 'wen_business_sanitize_featured_slider_number' ) ) :

  /**
   * Sanitize featured slider number
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_featured_slider_number( $input ) {

    $input = absint( $input );

    if ( $input < 1 || $input > 20 ) {
      $input = 3;
    }
    return $input;

  }

endif;

if( ! function_exists( 'wen_business_sanitize_post_id' ) ) :

  /**
   * Sanitize post ID
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_post_id( $input, $setting ) {

    $input = absint( $input );

    return ( 'publish' === get_post_status( $input ) ? $input : $setting->default );

  }

endif;

if( ! function_exists( 'wen_business_sanitize_page_id' ) ) :

  /**
   * Sanitize page ID
   *
   * @since  WEN Business 1.0
   */
  function wen_business_sanitize_page_id( $input, $setting ) {

    $input = absint( $input );

    return ( 'page' === get_post_type( $input ) ? $input : $setting->default );

  }

endif;
