<?php

add_action( 'widgets_init', 'wen_business_load_widgets' );

if ( ! function_exists( 'wen_business_load_widgets' ) ) :

  /**
   * Load widgets
   *
   * @since WEN Business 1.0
   *
   */
  function wen_business_load_widgets()
  {

    // Social widget
    register_widget( 'WEN_Business_Social_Widget' );

    // Advanced Text widget
    register_widget( 'WEN_Business_Advanced_Text_Widget' );

    // Call To Action widget
    register_widget( 'WEN_Business_Call_To_Action_Widget' );

    // Featured Page widget
    register_widget( 'WEN_Business_Featured_Page_Widget' );

    // Latest News widget
    register_widget( 'WEN_Business_Latest_News_Widget' );

    // Testimonial widget
    register_widget( 'WEN_Business_Testimonial_Widget' );

    // Service widget
    register_widget( 'WEN_Business_Service_Widget' );

    // Latest Works widget
    register_widget( 'WEN_Business_Latest_Works_Widget' );

    // Contact widget
    register_widget( 'WEN_Business_Contact_Widget' );

    // Special Pages widget
    register_widget( 'WEN_Business_Special_Pages_Widget' );

    // Ad widget
    register_widget( 'WEN_Business_Ad_Widget' );

  }

endif;


if ( ! class_exists( 'WEN_Business_Social_Widget' ) ) :

  /**
   * Social Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Social_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_social',
                  'description' => __( 'Social Icons Widget', 'wen-business-pro' )
              );

      parent::__construct( 'wen-business-social', __( 'Business Social Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title        = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base);
        $subtitle     = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base);
        $icon_size    = empty($instance['icon_size']) ? 16 : $instance['icon_size'];
        $custom_class = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );


        if ( $custom_class ) {
          $before_widget = str_replace('class="', 'class="'. $custom_class . ' ', $before_widget);
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        $nav_menu_locations = get_nav_menu_locations();
        $menu_id = 0;
        if ( isset( $nav_menu_locations['social'] ) && absint( $nav_menu_locations['social'] ) > 0 ) {
          $menu_id = absint( $nav_menu_locations['social'] );
        }
        if ( $menu_id > 0) {

          $menu_items = wp_get_nav_menu_items( $menu_id );

          if ( ! empty( $menu_items ) ) {
            echo '<ul>';
            foreach ( $menu_items as $m_key => $m ) {
              echo '<li>';
              echo '<a href="' . esc_url( $m->url ) . '" target="_blank">';
              echo '<span class="title screen-reader-text">' . esc_attr( $m->title ) . '</span>';
              echo '</a>';
              echo '</li>';
            }
            echo '</ul>';
            if ( isset( $widget_id ) ) {
              echo '<style type="text/css">#' . $widget_id . ' a:before{font-size:' . esc_attr( $icon_size ). 'px;}</style>';
            }
          }
        }
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']        =   esc_html( strip_tags($new_instance['title']) );
        $instance['subtitle']     =   esc_html( $new_instance['subtitle'] );
        $instance['icon_size']    =   absint( $new_instance['icon_size'] );
        if ( $instance['icon_size'] < 1 || $instance['icon_size'] > 100 ) {
          $instance['icon_size'] = 16;
        }
        $instance['custom_class'] =   esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'        => '',
          'subtitle'     => '',
          'icon_size'    => 16,
          'custom_class' => '',
        ) );
        $title        = esc_attr( $instance['title'] );
        $subtitle     = esc_textarea( $instance['subtitle'] );
        $icon_size    = esc_attr( $instance['icon_size'] );
        $custom_class = esc_attr( $instance['custom_class'] );

        // Fetch nav
        $nav_menu_locations = get_nav_menu_locations();
        $is_menu_set = false;
        if ( isset( $nav_menu_locations['social'] ) && absint( $nav_menu_locations['social'] ) > 0 ) {
          $is_menu_set = true;
        }
        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title'); ?>"><?php _e('Title:', 'wen-business-pro'); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle'); ?>"><?php _e('Sub Title:', 'wen-business-pro'); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle'); ?>" name="<?php echo $this->get_field_name('subtitle'); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'icon_size' ); ?>"><?php _e( 'Icon Size:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'icon_size'); ?>" name="<?php echo $this->get_field_name('icon_size'); ?>" type="text" value="<?php echo esc_attr( $icon_size ); ?>" style="max-width:55px;"/>&nbsp;<em><?php _e( 'px. Default: 16', 'wen-business-pro' ); ?></em>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>

        <?php if ( $is_menu_set ): ?>
          <?php
              $menu_id = $nav_menu_locations['social'];
              $social_menu_object = get_term( $menu_id, 'nav_menu' );
              $args = array(
                  'action' => 'edit',
                  'menu'   => $menu_id,
                  );
              $menu_edit_url = add_query_arg( $args, admin_url( 'nav-menus.php' ) );
           ?>
            <p>
                <?php echo __( 'Social Menu is currently set to', 'wen-business-pro' ) . ': '; ?>
                <strong><a href="<?php echo esc_url( $menu_edit_url );  ?>" ><?php echo $social_menu_object->name; ?></a></strong>
            </p>

          <?php else: ?>
          <?php
              $args = array(
                  'action' => 'locations',
                  );
              $menu_manage_url = add_query_arg( $args, admin_url( 'nav-menus.php' ) );
              $args = array(
                  'action' => 'edit',
                  'menu'   => 0,
                  );
              $menu_create_url = add_query_arg( $args, admin_url( 'nav-menus.php' ) );
           ?>
            <p>
              <?php echo __( 'Social menu is not set.', 'wen-business-pro' ) . ' '; ?><br />
              <strong><a href="<?php echo esc_url( $menu_manage_url );  ?>"><?php echo __( 'Click here to set menu', 'wen-business-pro' ); ?></a></strong>
              <?php echo ' '._x( 'or', 'Social Widget', 'wen-business-pro' ) . ' '; ?>
              <strong><a href="<?php echo esc_url( $menu_create_url );  ?>"><?php echo __( 'Create menu now', 'wen-business-pro' ); ?></a></strong>
            </p>

          <?php endif ?>

        <?php
      }

  }

endif;


if ( ! class_exists( 'WEN_Business_Advanced_Text_Widget' ) ) :

  /**
   * Advanced Text Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Advanced_Text_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_advanced_text',
                  'description' => __( 'Advanced Text Widget', 'wen-business-pro' )
              );

      parent::__construct( 'wen-business-advanced-text', __( 'Business Advanced Text Widget', 'wen-business-pro' ), $opts );
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title        = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle     = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $text         = apply_filters( 'widget_welcome_text', empty( $instance['text'] ) ? '' : $instance['text'], $instance, $this->id_base);
        $custom_class = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        // Run text through do_shortcode
        $text = do_shortcode( $text );

        ?>
        <div class="advanced-text-widget"><?php echo ! empty( $instance['filter'] ) ? wpautop( $text ) : $text; ?></div>
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']    = strip_tags($new_instance['title']);
        $instance['subtitle'] = esc_html( $new_instance['subtitle'] );

        if ( current_user_can( 'unfiltered_html' ) )
          $instance['text'] =  $new_instance['text'];
        else
          $instance['text'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['text'] ) ) ); // wp_filter_post_kses() expects slashed

        $instance['filter']       = isset( $new_instance['filter'] );

        $instance['custom_class'] =   esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'        => '',
          'subtitle'     => '',
          'text'         => '',
          'filter'       =>  0,
          'custom_class' => '',
        ) );
        $title        = strip_tags( $instance['title'] );
        $subtitle     = esc_textarea( $instance['subtitle'] );
        $text         = esc_textarea( $instance['text'] );
        $filter       = esc_attr($instance['filter']);
        $custom_class = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title'); ?>"><?php _e('Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle'); ?>"><?php _e('Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle'); ?>" name="<?php echo $this->get_field_name('subtitle'); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'text'); ?>"><?php _e( 'Text:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="5" id="<?php echo $this->get_field_id( 'text' ); ?>" name="<?php echo $this->get_field_name( 'text' ); ?>"><?php echo $text; ?></textarea>
        </p>
        <p><input id="<?php echo $this->get_field_id('filter'); ?>" name="<?php echo $this->get_field_name('filter'); ?>" type="checkbox" <?php checked(isset($instance['filter']) ? $instance['filter'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id('filter'); ?>"><?php _e( 'Automatically add paragraphs', 'wen-business-pro' ); ?></label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }
  }

endif;

if ( ! class_exists( 'WEN_Business_Call_To_Action_Widget' ) ) :

  /**
   * Call To Action Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Call_To_Action_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_call_to_action',
                  'description' => __( 'Call To Action Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-call-to-action', __( 'Business Call To Action Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title                = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle             = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $text                 = apply_filters( 'widget_welcome_text', empty( $instance['text'] ) ? '' : $instance['text'], $instance, $this->id_base);
        $primary_button_text  = ! empty( $instance['primary_button_text'] ) ? strip_tags( $instance['primary_button_text'] ) : '';
        $primary_button_url   = ! empty( $instance['primary_button_url'] ) ? esc_url( $instance['primary_button_url'] ) : '';
        $secondary_button_text  = ! empty( $instance['secondary_button_text'] ) ? strip_tags( $instance['secondary_button_text'] ) : '';
        $secondary_button_url = ! empty( $instance['secondary_button_url'] ) ? esc_url( $instance['secondary_button_url'] ) : '';
        $open_url_in_new_window    = ! empty( $instance['open_url_in_new_window'] ) ? $instance['open_url_in_new_window'] : false ;
        $custom_class         = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //

        ?>
        <div class="call-to-action-widget"><?php echo ! empty( $instance['filter'] ) ? wpautop( $text ) : $text; ?></div>
        <div class="call-to-action-buttons">
          <?php
            $target_text = '';
            if ( $open_url_in_new_window ) {
              $target_text = ' target="_blank" ';
            }
          ?>
          <?php if ( ! empty( $primary_button_text ) ): ?>
            <a href="<?php echo $primary_button_url; ?>" <?php echo $target_text ?> class="cta-button-primary" title="<?php echo esc_attr( $primary_button_text ); ?>" ><?php echo $primary_button_text; ?></a>
          <?php endif ?>
          <?php if ( ! empty( $secondary_button_text ) ): ?>
            <a href="<?php echo $secondary_button_url; ?>" <?php echo $target_text ?> class="cta-button-secondary" title="<?php echo esc_attr( $secondary_button_text ); ?>" ><?php echo $secondary_button_text; ?></a>
          <?php endif ?>
        </div><!-- .cta-buttons -->
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']    = strip_tags( $new_instance['title'] );
        $instance['subtitle'] = esc_html( $new_instance['subtitle'] );

        if ( current_user_can( 'unfiltered_html' ) )
          $instance['text'] =  $new_instance['text'];
        else
          $instance['text'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['text'] ) ) ); // wp_filter_post_kses() expects slashed

        $instance['filter']                 = isset( $new_instance['filter'] );
        $instance['primary_button_text']    = strip_tags( $new_instance['primary_button_text'] );
        $instance['primary_button_url']     = esc_url( $new_instance['primary_button_url'] );
        $instance['secondary_button_text']  = strip_tags( $new_instance['secondary_button_text'] );
        $instance['secondary_button_url']   = esc_url( $new_instance['secondary_button_url'] );
        $instance['open_url_in_new_window'] = isset( $new_instance['open_url_in_new_window'] );
        $instance['custom_class']           = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'                  => '',
          'subtitle'               => '',
          'text'                   => '',
          'filter'                 => 0,
          'primary_button_text'    => __( 'Read more', 'wen-business-pro' ),
          'primary_button_url'     => '',
          'secondary_button_text'  => '',
          'secondary_button_url'   => '',
          'open_url_in_new_window' => 0,
          'custom_class'           => '',
        ) );
        $title                  = strip_tags( $instance['title'] );
        $subtitle               = esc_textarea( $instance['subtitle'] );
        $text                   = esc_textarea( $instance['text'] );
        $filter                 = esc_attr($instance['filter']);
        $primary_button_text    = esc_html( strip_tags( $instance['primary_button_text'] ) );
        $primary_button_url     = esc_url( $instance['primary_button_url'] );
        $secondary_button_text  = esc_html( strip_tags( $instance['secondary_button_text'] ) );
        $secondary_button_url   = esc_url( $instance['secondary_button_url'] );
        $open_url_in_new_window = esc_attr($instance['open_url_in_new_window']);
        $custom_class           = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title'); ?>"><?php _e('Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle'); ?>"><?php _e('Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle'); ?>" name="<?php echo $this->get_field_name('subtitle'); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'text'); ?>"><?php _e( 'Text:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="5" id="<?php echo $this->get_field_id( 'text' ); ?>" name="<?php echo $this->get_field_name( 'text' ); ?>"><?php echo $text; ?></textarea>
        </p>
        <p><input id="<?php echo $this->get_field_id('filter'); ?>" name="<?php echo $this->get_field_name('filter'); ?>" type="checkbox" <?php checked(isset($instance['filter']) ? $instance['filter'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id('filter'); ?>"><?php _e( 'Automatically add paragraphs', 'wen-business-pro' ); ?></label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'primary_button_text' ); ?>"><?php _e( 'Primary Button Text:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'primary_button_text'); ?>" name="<?php echo $this->get_field_name( 'primary_button_text' ); ?>" type="text" value="<?php echo esc_attr( $primary_button_text ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'primary_button_url' ); ?>"><?php _e( 'Primary Button URL:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'primary_button_url'); ?>" name="<?php echo $this->get_field_name( 'primary_button_url' ); ?>" type="text" value="<?php echo esc_attr( $primary_button_url ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'secondary_button_text' ); ?>"><?php _e( 'Secondary Button Text:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'secondary_button_text'); ?>" name="<?php echo $this->get_field_name( 'secondary_button_text' ); ?>" type="text" value="<?php echo esc_attr( $secondary_button_text ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'secondary_button_url' ); ?>"><?php _e( 'Secondary Button URL:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'secondary_button_url'); ?>" name="<?php echo $this->get_field_name( 'secondary_button_url' ); ?>" type="text" value="<?php echo esc_attr( $secondary_button_url ); ?>" />
        </p>
        <p><input id="<?php echo $this->get_field_id('open_url_in_new_window'); ?>" name="<?php echo $this->get_field_name('open_url_in_new_window'); ?>" type="checkbox" <?php checked(isset($instance['open_url_in_new_window']) ? $instance['open_url_in_new_window'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id('open_url_in_new_window'); ?>"><?php _e( 'Open URL in New Window', 'wen-business-pro' ); ?></label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>



        <?php
      }

  }

endif;


if ( ! class_exists( 'WEN_Business_Featured_Page_Widget' ) ) :

  /**
   * Featured Page Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Featured_Page_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_featured_page',
                  'description' => __( 'Featured Page Widget', 'wen-business-pro' )
              );

      parent::__construct( 'wen-business-featured-page', __( 'Business Featured Page Widget', 'wen-business-pro' ), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title          = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $use_page_title = ! empty( $instance['use_page_title'] ) ? $instance['use_page_title'] : false ;
        $subtitle       = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $featured_page  = ! empty( $instance['featured_page'] ) ? $instance['featured_page'] : 0;
        $featured_post  = ! empty( $instance['featured_post'] ) ? $instance['featured_post'] : 0;
        $content_type   = ! empty( $instance['content_type'] ) ? $instance['content_type'] : 'full';
        $featured_image = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'disable';
        $custom_class   = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        // ID validation
        $our_post_object = null;
        $our_id = '';
        if ( absint( $featured_post ) > 0 ) {
          $our_id = absint( $featured_post );
        }
        if ( absint( $featured_page ) > 0 ) {
          $our_id = absint( $featured_page );
        }
        if ( absint( $our_id ) > 0 ) {
          $raw_object = get_post( $our_id );
          if ( ! in_array( $raw_object->post_type, array( 'attachment', 'nav_menu_item', 'revision' ) ) ) {
            $our_post_object = $raw_object;
          }
        }
        if ( null == $our_post_object ) {
          // No valid object; bail now;
          return;
        }

        global $post;
        // Setup global post
        $post = $our_post_object;
        setup_postdata( $post );

        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        // Override title if checkbox is selected
        if ( false != $use_page_title ) {
          $title = get_the_title( $post );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        ?>
        <?php
          // Display featured image
          if ( 'disable' != $featured_image && has_post_thumbnail() ) {
            the_post_thumbnail( $featured_image, array( 'class' => 'aligncenter' ) );
          }
        ?>
        <div class="featured-page-widget entry-content"><?php if ( 'short' == $content_type ) { the_excerpt(); } else { the_content(); } ?></div>
        <?php
        // Reset
        wp_reset_postdata();
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']          = strip_tags($new_instance['title']);
        $instance['use_page_title'] = isset( $new_instance['use_page_title'] );
        $instance['subtitle']       = esc_html( $new_instance['subtitle'] );
        $instance['featured_page']  = absint( $new_instance['featured_page'] );
        $instance['featured_post']  = absint( $new_instance['featured_post'] );
        if ( $instance['featured_post'] <= 0 ) {
          $instance['featured_post'] = '';
        }
        $instance['content_type']   = esc_attr( $new_instance['content_type'] );
        $instance['featured_image'] = esc_attr( $new_instance['featured_image'] );
        $instance['custom_class']   = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'          => '',
          'use_page_title' => 1,
          'subtitle'       => '',
          'featured_page'  => '',
          'featured_post'  => '',
          'content_type'   => 'full',
          'featured_image' => 'disable',
          'custom_class'   => '',
        ) );
        $title          = strip_tags( $instance['title'] );
        $use_page_title = esc_attr( $instance['use_page_title'] );
        $subtitle       = esc_textarea( $instance['subtitle'] );
        $featured_page  = absint( $instance['featured_page'] );
        $featured_post  = esc_attr( $instance['featured_post'] );
        $content_type   = esc_attr( $instance['content_type'] );
        $featured_image = esc_attr($instance['featured_image']);
        $custom_class   = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title'); ?>"><?php _e('Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p><input id="<?php echo $this->get_field_id( 'use_page_title' ); ?>" name="<?php echo $this->get_field_name( 'use_page_title' ); ?>" type="checkbox" <?php checked(isset($instance['use_page_title']) ? $instance['use_page_title'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'use_page_title' ); ?>"><?php _e( 'Use Page/Post Title as Widget Title', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle'); ?>"><?php _e('Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle'); ?>" name="<?php echo $this->get_field_name('subtitle'); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_page'); ?>"><?php _e( 'Select Page:', 'wen-business-pro' ); ?></label>
          <?php
            wp_dropdown_pages( array(
              'id'               => $this->get_field_id( 'featured_page' ),
              'name'             => $this->get_field_name( 'featured_page' ),
              'selected'         => $featured_page,
              'show_option_none' => __( '&mdash; Select &mdash;', 'wen-business-pro' ),
              )
            );
          ?>
        </p>
        <h4><?php _ex( 'OR', 'Featured Page Widget', 'wen-business-pro' ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_post' ); ?>"><?php _e( 'Enter Post ID:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'featured_post'); ?>" name="<?php echo $this->get_field_name('featured_post'); ?>" type="text" value="<?php echo esc_attr( $featured_post ); ?>" style="max-width:65px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'content_type' ); ?>"><?php _e( 'Show Content:', 'wen-business-pro' ); ?></label>
          <select id="<?php echo $this->get_field_id( 'content_type' ); ?>" name="<?php echo $this->get_field_name( 'content_type' ); ?>">
            <option value="short" <?php selected( $content_type, 'short' ) ?>><?php _e( 'Short', 'wen-business-pro' ) ?></option>
            <option value="full" <?php selected( $content_type, 'full' ) ?>><?php _e( 'Full', 'wen-business-pro' ) ?></option>
          </select>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_image' ); ?>"><?php _e( 'Featured Image:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_image_sizes( array(
              'id'       => $this->get_field_id( 'featured_image' ),
              'name'     => $this->get_field_name( 'featured_image' ),
              'selected' => $featured_image,
              )
            );
          ?>

        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }

    function dropdown_image_sizes( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = $this->get_image_sizes_options();

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    private function get_image_sizes_options(){

      global $_wp_additional_image_sizes;
      $get_intermediate_image_sizes = get_intermediate_image_sizes();
      $choices = array();
      $choices['disable'] = __( 'No Image', 'wen-business-pro' );
      foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
        $choices[ $_size ] = $_size . ' ('. get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
      }
      $choices['full'] = __( 'full (original)', 'wen-business-pro' );
      if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {

        foreach ($_wp_additional_image_sizes as $key => $size ) {
          $choices[ $key ] = $key . ' ('. $size['width'] . 'x' . $size['height'] . ')';
        }

      }
      return $choices;
    }

  }

endif;


if ( ! class_exists( 'WEN_Business_Latest_News_Widget' ) ) :

  /**
   * Latest News Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Latest_News_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_latest_news',
                  'description' => __( 'Latest News Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-latest-news', __( 'Business Latest News Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title          = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle       = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $post_category     = ! empty( $instance['post_category'] ) ? $instance['post_category'] : 0;
        $post_column       = ! empty( $instance['post_column'] ) ? $instance['post_column'] : 4;
        $featured_image    = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'thumbnail';
        $post_number       = ! empty( $instance['post_number'] ) ? $instance['post_number'] : 4;
        $excerpt_length    = ! empty( $instance['excerpt_length'] ) ? $instance['excerpt_length'] : 40;
        $post_order_by     = ! empty( $instance['post_order_by'] ) ? $instance['post_order_by'] : 'date';
        $post_order        = ! empty( $instance['post_order'] ) ? $instance['post_order'] : 'desc';
        $more_text         = ! empty( $instance['more_text'] ) ? $instance['more_text'] : __( 'Read more','wen-business-pro' );
        $disable_date      = ! empty( $instance['disable_date'] ) ? $instance['disable_date'] : false ;
        $disable_comment   = ! empty( $instance['disable_comment'] ) ? $instance['disable_comment'] : false ;
        $disable_excerpt   = ! empty( $instance['disable_excerpt'] ) ? $instance['disable_excerpt'] : false ;
        $disable_more_text = ! empty( $instance['disable_more_text'] ) ? $instance['disable_more_text'] : false ;
        $custom_class   = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        // Validation
        // Order
        if ( in_array( $post_order, array( 'asc', 'desc' ) ) ) {
          $post_order = strtoupper( $post_order );
        }
        else{
          $post_order = 'DESC';
        }
        // Order By
        switch ( $post_order_by ) {
          case 'date':
            $post_order_by = 'date';
            break;
          case 'title':
            $post_order_by = 'title';
            break;
          case 'comment-count':
            $post_order_by = 'comment_count';
            break;
          case 'random':
            $post_order_by = 'rand';
            break;
          case 'menu-order':
            $post_order_by = 'menu_order';
            break;
          default:
            $post_order_by = 'date';
            break;
        }
        // Column class
        switch ( $post_column ) {
          case 1:
            $column_class = 'col-sm-12';
            break;
          case 2:
            $column_class = 'col-sm-6';
            break;
          case 3:
            $column_class = 'col-sm-4';
            break;
          case 4:
            $column_class = 'col-sm-3';
            break;
          case 5:
            $column_class = 'col-sm-5ths';
            break;
          case 6:
            $column_class = 'col-sm-2';
            break;
          default:
            $column_class = '';
            break;
        }


        // Add Custom class
        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        ?>
        <?php
          $qargs = array(
            'posts_per_page' => $post_number,
            'no_found_rows'  => true,
            'orderby'        => $post_order_by,
            'order'          => $post_order,
            );
          if ( absint( $post_category ) > 0  ) {
            $qargs['cat'] = $post_category;
          }

          $all_posts = get_posts( $qargs );
        ?>
        <?php if ( ! empty( $all_posts ) ): ?>


          <?php global $post; ?>

          <div class="latest-news-widget">

            <div class="row">

              <?php foreach ( $all_posts as $key => $post ): ?>
                <?php setup_postdata( $post ); ?>

                <div class="<?php echo esc_attr( $column_class ); ?>">
                  <div class="latest-news-item">

                      <?php if ( 'disable' != $featured_image && has_post_thumbnail() ): ?>
                        <div class="latest-news-thumb">
                          <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>">
                            <?php
                              $img_attributes = array( 'class' => 'aligncenter' );
                              the_post_thumbnail( $featured_image, $img_attributes );
                            ?>
                          </a>
                        </div><!-- .latest-news-thumb -->
                      <?php endif ?>
                      <div class="latest-news-text-wrap">
                        <h3 class="latest-news-title">
                          <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>"><?php the_title(); ?></a>
                        </h3><!-- .latest-news-title -->

                        <?php if ( false == $disable_date || ( false == $disable_comment && comments_open( get_the_ID() ) ) ): ?>
                          <div class="latest-news-meta">

                            <?php if ( false == $disable_date ): ?>
                              <span class="latest-news-date"><?php the_time( get_option('date_format') ); ?></span><!-- .latest-news-date -->
                            <?php endif ?>

                            <?php if ( false == $disable_comment ): ?>
                              <?php
                              if ( comments_open( get_the_ID() ) ) {
                                echo '<span class="latest-news-comments">';
                                comments_popup_link( '<span class="leave-reply">' . __( 'No Comment', 'wen-business-pro' ) . '</span>', __( '1 Comment', 'wen-business-pro' ), __( '% Comments', 'wen-business-pro' ) );
                                echo '</span>';
                              }
                              ?>
                            <?php endif ?>

                          </div><!-- .latest-news-meta -->
                        <?php endif ?>

                        <?php if ( false == $disable_excerpt ): ?>
                          <div class="latest-news-summary">
                            <?php
                            $excerpt = wen_business_the_excerpt( $excerpt_length, $post );
                            echo wp_kses_post( wpautop( $excerpt ) );
                            ?>
                          </div><!-- .latest-news-summary -->
                        <?php endif ?>
                        <?php if ( false == $disable_more_text ): ?>
                          <div class="latest-news-read-more"><a href="<?php the_permalink(); ?>" class="read-more" title="<?php the_title_attribute(); ?>"><?php echo esc_html( $more_text ); ?></a></div><!-- .latest-news-read-more -->
                        <?php endif ?>
                      </div><!-- .latest-news-text-wrap -->
                  </div><!-- .latest-news-item -->

                </div><!-- .latest-news-item .col-sm-3 -->

              <?php endforeach ?>

            </div><!-- .row -->

          </div><!-- .latest-news-widget -->

          <?php wp_reset_postdata(); // Reset ?>

        <?php endif; ?>
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']             = strip_tags($new_instance['title']);
        $instance['subtitle']          = esc_html( $new_instance['subtitle'] );
        $instance['post_category']     = absint( $new_instance['post_category'] );
        $instance['post_number']       = absint( $new_instance['post_number'] );
        $instance['post_column']       = absint( $new_instance['post_column'] );
        $instance['excerpt_length']    = absint( $new_instance['excerpt_length'] );
        $instance['post_order_by']     = esc_attr( $new_instance['post_order_by'] );
        $instance['post_order']        = esc_attr( $new_instance['post_order'] );
        $instance['featured_image']    = esc_attr( $new_instance['featured_image'] );
        $instance['disable_date']      = isset( $new_instance['disable_date'] );
        $instance['disable_comment']   = isset( $new_instance['disable_comment'] );
        $instance['disable_excerpt']   = isset( $new_instance['disable_excerpt'] );
        $instance['disable_more_text'] = isset( $new_instance['disable_more_text'] );
        $instance['more_text']         = esc_attr( $new_instance['more_text'] );
        $instance['custom_class']      = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'             => '',
          'subtitle'          => '',
          'post_category'     => '',
          'post_column'       => 4,
          'featured_image'    => 'thumbnail',
          'post_number'       => 4,
          'excerpt_length'    => 40,
          'post_order_by'     => 'date',
          'post_order'        => 'desc',
          'more_text'         => __( 'Read more', 'wen-business-pro' ),
          'disable_date'      => 0,
          'disable_comment'   => 0,
          'disable_excerpt'   => 0,
          'disable_more_text' => 0,
          'custom_class'      => '',
        ) );
        $title             = strip_tags( $instance['title'] );
        $subtitle          = esc_textarea( $instance['subtitle'] );
        $post_category     = absint( $instance['post_category'] );
        $post_column       = absint( $instance['post_column'] );
        $featured_image    = esc_attr( $instance['featured_image'] );
        $post_number       = absint( $instance['post_number'] );
        $excerpt_length    = absint( $instance['excerpt_length'] );
        $post_order_by     = esc_attr( $instance['post_order_by'] );
        $post_order        = esc_attr( $instance['post_order'] );
        $more_text         = strip_tags( $instance['more_text'] );
        $disable_date      = esc_attr( $instance['disable_date'] );
        $disable_comment   = esc_attr( $instance['disable_comment'] );
        $disable_excerpt   = esc_attr( $instance['disable_excerpt'] );
        $disable_more_text = esc_attr( $instance['disable_more_text'] );
        $custom_class      = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e( 'Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle' ); ?>" name="<?php echo $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_category' ); ?>"><?php _e( 'Category:', 'wen-business-pro' ); ?></label>
          <?php
            $cat_args = array(
                'orderby'         => 'name',
                'hide_empty'      => 0,
                'taxonomy'        => 'category',
                'name'            => $this->get_field_name('post_category'),
                'id'              => $this->get_field_id('post_category'),
                'selected'        => $post_category,
                'show_option_all' => __( 'All Categories','wen-business-pro' ),
              );
            wp_dropdown_categories( $cat_args );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_column' ); ?>"><?php _e('Number of Columns:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_post_columns( array(
              'id'       => $this->get_field_id( 'post_column' ),
              'name'     => $this->get_field_name( 'post_column' ),
              'selected' => $post_column,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_image' ); ?>"><?php _e( 'Featured Image:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_image_sizes( array(
              'id'       => $this->get_field_id( 'featured_image' ),
              'name'     => $this->get_field_name( 'featured_image' ),
              'selected' => $featured_image,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_number' ); ?>"><?php _e('Number of Posts:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'post_number' ); ?>" name="<?php echo $this->get_field_name( 'post_number' ); ?>" type="number" value="<?php echo esc_attr( $post_number ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'excerpt_length' ); ?>"><?php _e('Excerpt Length:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'excerpt_length' ); ?>" name="<?php echo $this->get_field_name( 'excerpt_length' ); ?>" type="number" value="<?php echo esc_attr( $excerpt_length ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order_by' ); ?>"><?php _e( 'Order By:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_post_order_by( array(
              'id'       => $this->get_field_id( 'post_order_by' ),
              'name'     => $this->get_field_name( 'post_order_by' ),
              'selected' => $post_order_by,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order' ); ?>"><?php _e( 'Order:', 'wen-business-pro' ); ?></label>
          <select id="<?php echo $this->get_field_id( 'post_order' ); ?>" name="<?php echo $this->get_field_name( 'post_order' ); ?>">
            <option value="asc" <?php selected( $post_order, 'asc' ) ?>><?php _e( 'Ascending', 'wen-business-pro' ) ?></option>
            <option value="desc" <?php selected( $post_order, 'desc' ) ?>><?php _e( 'Descending', 'wen-business-pro' ) ?></option>
          </select>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'more_text' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'more_text'); ?>" name="<?php echo $this->get_field_name( 'more_text' ); ?>" type="text" value="<?php echo esc_attr( $more_text ); ?>" />
        </p>
        <p><input id="<?php echo $this->get_field_id( 'disable_date' ); ?>" name="<?php echo $this->get_field_name( 'disable_date' ); ?>" type="checkbox" <?php checked(isset($instance['disable_date']) ? $instance['disable_date'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_date' ); ?>"><?php _e( 'Disable Date', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p><input id="<?php echo $this->get_field_id( 'disable_comment' ); ?>" name="<?php echo $this->get_field_name( 'disable_comment' ); ?>" type="checkbox" <?php checked(isset($instance['disable_comment']) ? $instance['disable_comment'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_comment' ); ?>"><?php _e( 'Disable Comment', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <input id="<?php echo $this->get_field_id( 'disable_excerpt' ); ?>" name="<?php echo $this->get_field_name( 'disable_excerpt' ); ?>" type="checkbox" <?php checked(isset($instance['disable_excerpt']) ? $instance['disable_excerpt'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_excerpt' ); ?>"><?php _e( 'Disable Excerpt', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <input id="<?php echo $this->get_field_id( 'disable_more_text' ); ?>" name="<?php echo $this->get_field_name( 'disable_more_text' ); ?>" type="checkbox" <?php checked(isset($instance['disable_more_text']) ? $instance['disable_more_text'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_more_text' ); ?>"><?php _e( 'Disable More Text', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }

    function dropdown_post_columns( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = array(
        '1' => sprintf( __( '%d Column','wen-business-pro' ), 1 ),
        '2' => sprintf( __( '%d Columns','wen-business-pro' ), 2 ),
        '3' => sprintf( __( '%d Columns','wen-business-pro' ), 3 ),
        '4' => sprintf( __( '%d Columns','wen-business-pro' ), 4 ),
      );

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    function dropdown_post_order_by( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = array(
        'date'          => __( 'Date','wen-business-pro' ),
        'title'         => __( 'Title','wen-business-pro' ),
        'comment-count' => __( 'Comment Count','wen-business-pro' ),
        'menu-order'    => __( 'Menu Order','wen-business-pro' ),
        'random'        => __( 'Random','wen-business-pro' ),
      );

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    function dropdown_image_sizes( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = $this->get_image_sizes_options();

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    private function get_image_sizes_options(){

      global $_wp_additional_image_sizes;
      $get_intermediate_image_sizes = get_intermediate_image_sizes();
      $choices = array();
      $choices['disable'] = __( 'No Image', 'wen-business-pro' );
      foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
        $choices[ $_size ] = $_size . ' ('. get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
      }
      $choices['full'] = __( 'full (original)', 'wen-business-pro' );
      if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {

        foreach ($_wp_additional_image_sizes as $key => $size ) {
          $choices[ $key ] = $key . ' ('. $size['width'] . 'x' . $size['height'] . ')';
        }

      }
      return $choices;
    }

  }

endif;

if ( ! class_exists( 'WEN_Business_Testimonial_Widget' ) ) :

  /**
   * Testimonial Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Testimonial_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_testimonial',
                  'description' => __( 'Testimonial Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-testimonial', __( 'Business Testimonial Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title          = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle       = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $post_category       = ! empty( $instance['post_category'] ) ? $instance['post_category'] : 0;
        $featured_image      = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'thumbnail';
        $post_number         = ! empty( $instance['post_number'] ) ? $instance['post_number'] : 4;
        $excerpt_length      = ! empty( $instance['excerpt_length'] ) ? $instance['excerpt_length'] : 40;
        $post_order_by       = ! empty( $instance['post_order_by'] ) ? $instance['post_order_by'] : 'date';
        $post_order          = ! empty( $instance['post_order'] ) ? $instance['post_order'] : 'desc';
        $transition_delay    = ! empty( $instance['transition_delay'] ) ? $instance['transition_delay'] : 3;
        $transition_duration = ! empty( $instance['transition_duration'] ) ? $instance['transition_duration'] : 1;
        $disable_pager       = ! empty( $instance['disable_pager'] ) ? $instance['disable_pager'] : false ;
        $custom_class        = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        // Validation
        // Order
        if ( in_array( $post_order, array( 'asc', 'desc' ) ) ) {
          $post_order = strtoupper( $post_order );
        }
        else{
          $post_order = 'DESC';
        }
        // Order By
        switch ( $post_order_by ) {
          case 'date':
            $post_order_by = 'date';
            break;
          case 'title':
            $post_order_by = 'title';
            break;
          case 'random':
            $post_order_by = 'rand';
            break;
          case 'menu-order':
            $post_order_by = 'menu_order';
            break;
          default:
            $post_order_by = 'date';
            break;
        }


        // Add Custom class
        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        ?>
        <?php
          $qargs = array(
            'posts_per_page' => $post_number,
            'no_found_rows'  => true,
            'orderby'        => $post_order_by,
            'order'          => $post_order,
            );
          if ( absint( $post_category ) > 0  ) {
            $qargs['cat'] = $post_category;
          }

          $all_posts = get_posts( $qargs );
        ?>
        <?php if ( ! empty( $all_posts ) ): ?>

          <?php global $post; ?>

          <?php
            // Cycle data
            $slide_data = array(
              'fx'             => 'fade',
              'speed'          => $transition_duration * 1000,
              'pause-on-hover' => 'true',
              'log'            => 'false',
              'swipe'          => 'true',
              'auto-height'    => 'container',
              'slides'         => '> article',
            );
            $slide_data['timeout'] = $transition_delay * 1000;
            $slide_attributes_text = '';
            foreach ($slide_data as $key => $item) {
              $slide_attributes_text .= ' ';
              $slide_attributes_text .= ' data-cycle-'.esc_attr( $key );
              $slide_attributes_text .= '="'.esc_attr( $item ).'"';
            }


          ?>

          <div class="testimonial-widget">

            <div class="cycle-slideshow" <?php echo $slide_attributes_text; ?> >


              <?php foreach ( $all_posts as $key => $post ): ?>
                <?php setup_postdata( $post ); ?>

                <article class="testimonial-item">

                  <?php if ( 'disable' != $featured_image && has_post_thumbnail() ): ?>
                    <div class="testimonial-thumb">
                        <?php
                          the_post_thumbnail( $featured_image );
                        ?>
                    </div><!-- .testimonial-thumb -->
                  <?php endif ?>
                  <div class="testimonial-text-wrap">
                    <div class="testimonial-summary">
                      <?php
                      $excerpt = wen_business_the_excerpt( $excerpt_length, $post );
                      echo wp_kses_post( wpautop( $excerpt ) );
                      ?>
                    </div><!-- .testimonial-summary -->
                    <h3 class="testimonial-title"><?php the_title(); ?></h3><!-- .testimonial-title -->
                  </div><!-- .testimonial-text-wrap -->

                </article><!-- .testimonial-item -->

              <?php endforeach ?>

              <?php if ( false == $disable_pager ): ?>
                <div class="cycle-pager"></div>
              <?php endif ?>

            </div><!-- .cycle-slideshow -->


          </div><!-- .testimonial-widget -->

          <?php wp_reset_postdata(); // Reset ?>

        <?php endif; ?>
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']               = strip_tags($new_instance['title']);
        $instance['subtitle']            = esc_html( $new_instance['subtitle'] );
        $instance['post_category']       = absint( $new_instance['post_category'] );
        $instance['featured_image']      = esc_attr( $new_instance['featured_image'] );
        $instance['post_number']         = absint( $new_instance['post_number'] );
        $instance['excerpt_length']      = absint( $new_instance['excerpt_length'] );
        $instance['post_order_by']       = esc_attr( $new_instance['post_order_by'] );
        $instance['post_order']          = esc_attr( $new_instance['post_order'] );
        $instance['transition_delay']    = absint( $new_instance['transition_delay'] );
        $instance['transition_duration'] = absint( $new_instance['transition_duration'] );
        $instance['disable_pager']       = isset( $new_instance['disable_pager'] );
        $instance['custom_class']        = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'               => '',
          'subtitle'            => '',
          'post_category'       => '',
          'featured_image'      => 'thumbnail',
          'post_number'         => 4,
          'excerpt_length'      => 20,
          'post_order_by'       => 'date',
          'post_order'          => 'desc',
          'transition_delay'    => 3,
          'transition_duration' => 1,
          'disable_pager'       => 0,
          'custom_class'        => '',
        ) );
        $title               = strip_tags( $instance['title'] );
        $subtitle            = esc_textarea( $instance['subtitle'] );
        $post_category       = absint( $instance['post_category'] );
        $featured_image      = esc_attr( $instance['featured_image'] );
        $post_number         = absint( $instance['post_number'] );
        $excerpt_length      = absint( $instance['excerpt_length'] );
        $post_order_by       = esc_attr( $instance['post_order_by'] );
        $post_order          = esc_attr( $instance['post_order'] );
        $transition_delay    = absint( $instance['transition_delay'] );
        $transition_duration = absint( $instance['transition_duration'] );
        $disable_pager       = esc_attr( $instance['disable_pager'] );
        $custom_class        = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e( 'Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle' ); ?>" name="<?php echo $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_category' ); ?>"><?php _e( 'Category:', 'wen-business-pro' ); ?></label>
          <?php
            $cat_args = array(
                'orderby'         => 'name',
                'hide_empty'      => 0,
                'taxonomy'        => 'category',
                'name'            => $this->get_field_name('post_category'),
                'id'              => $this->get_field_id('post_category'),
                'selected'        => $post_category,
                'show_option_all' => __( 'All Categories','wen-business-pro' ),
              );
            wp_dropdown_categories( $cat_args );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_image' ); ?>"><?php _e( 'Featured Image:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_image_sizes( array(
              'id'       => $this->get_field_id( 'featured_image' ),
              'name'     => $this->get_field_name( 'featured_image' ),
              'selected' => $featured_image,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_number' ); ?>"><?php _e('Number of Posts:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'post_number' ); ?>" name="<?php echo $this->get_field_name( 'post_number' ); ?>" type="number" value="<?php echo esc_attr( $post_number ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'excerpt_length' ); ?>"><?php _e('Excerpt Length:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'excerpt_length' ); ?>" name="<?php echo $this->get_field_name( 'excerpt_length' ); ?>" type="number" value="<?php echo esc_attr( $excerpt_length ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order_by' ); ?>"><?php _e( 'Order By:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_post_order_by( array(
              'id'       => $this->get_field_id( 'post_order_by' ),
              'name'     => $this->get_field_name( 'post_order_by' ),
              'selected' => $post_order_by,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order' ); ?>"><?php _e( 'Order:', 'wen-business-pro' ); ?></label>
          <select id="<?php echo $this->get_field_id( 'post_order' ); ?>" name="<?php echo $this->get_field_name( 'post_order' ); ?>">
            <option value="asc" <?php selected( $post_order, 'asc' ) ?>><?php _e( 'Ascending', 'wen-business-pro' ) ?></option>
            <option value="desc" <?php selected( $post_order, 'desc' ) ?>><?php _e( 'Descending', 'wen-business-pro' ) ?></option>
          </select>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'transition_delay' ); ?>"><?php _e( 'Transition Delay:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'transition_delay' ); ?>" name="<?php echo $this->get_field_name( 'transition_delay' ); ?>" type="text" value="<?php echo esc_attr( $transition_delay ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'transition_duration' ); ?>"><?php _e( 'Transition Duration:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'transition_duration' ); ?>" name="<?php echo $this->get_field_name( 'transition_duration' ); ?>" type="text" value="<?php echo esc_attr( $transition_duration ); ?>" />
        </p>
        <p><input id="<?php echo $this->get_field_id( 'disable_pager' ); ?>" name="<?php echo $this->get_field_name( 'disable_pager' ); ?>" type="checkbox" <?php checked(isset($instance['disable_pager']) ? $instance['disable_pager'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_pager' ); ?>"><?php _e( 'Disable Pager', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }

    function dropdown_post_order_by( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = array(
        'date'          => __( 'Date','wen-business-pro' ),
        'title'         => __( 'Title','wen-business-pro' ),
        'menu-order'    => __( 'Menu Order','wen-business-pro' ),
        'random'        => __( 'Random','wen-business-pro' ),
      );

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    function dropdown_image_sizes( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = $this->get_image_sizes_options();

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    private function get_image_sizes_options(){

      global $_wp_additional_image_sizes;
      $get_intermediate_image_sizes = get_intermediate_image_sizes();
      $choices = array();
      $choices['disable'] = __( 'No Image', 'wen-business-pro' );
      foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
        $choices[ $_size ] = $_size . ' ('. get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
      }
      $choices['full'] = __( 'full (original)', 'wen-business-pro' );
      if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {

        foreach ($_wp_additional_image_sizes as $key => $size ) {
          $choices[ $key ] = $key . ' ('. $size['width'] . 'x' . $size['height'] . ')';
        }

      }
      return $choices;
    }

  }

endif;

if ( ! class_exists( 'WEN_Business_Service_Widget' ) ) :

  /**
   * Service Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Service_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_service',
                  'description' => __( 'Service Widget', 'wen-business-pro' )
              );
      $control_ops = array( 'width' => 400 );

      parent::__construct('wen-business-service', __( 'Business Service Widget', 'wen-business-pro'), $opts, $control_ops );
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title        = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle     = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $custom_class = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        $block_title_1       = ! empty( $instance['block_title_1'] ) ? $instance['block_title_1'] : '';
        $block_icon_1        = ! empty( $instance['block_icon_1'] ) ? $instance['block_icon_1'] : 'fa-cogs';
        $block_description_1 = ! empty( $instance['block_description_1'] ) ? $instance['block_description_1'] : '';
        $block_read_more_1   = ! empty( $instance['block_read_more_1'] ) ? $instance['block_read_more_1'] : __( 'Read more', 'wen-business-pro' );
        $block_url_1         = ! empty( $instance['block_url_1'] ) ? $instance['block_url_1'] : '';

        $block_title_2       = ! empty( $instance['block_title_2'] ) ? $instance['block_title_2'] : '';
        $block_icon_2        = ! empty( $instance['block_icon_2'] ) ? $instance['block_icon_2'] : 'fa-cogs';
        $block_description_2 = ! empty( $instance['block_description_2'] ) ? $instance['block_description_2'] : '';
        $block_read_more_2   = ! empty( $instance['block_read_more_2'] ) ? $instance['block_read_more_2'] : __( 'Read more', 'wen-business-pro' );
        $block_url_2         = ! empty( $instance['block_url_2'] ) ? $instance['block_url_2'] : '';

        $block_title_3       = ! empty( $instance['block_title_3'] ) ? $instance['block_title_3'] : '';
        $block_icon_3        = ! empty( $instance['block_icon_3'] ) ? $instance['block_icon_3'] : 'fa-cogs';
        $block_description_3 = ! empty( $instance['block_description_3'] ) ? $instance['block_description_3'] : '';
        $block_read_more_3   = ! empty( $instance['block_read_more_3'] ) ? $instance['block_read_more_3'] : __( 'Read more', 'wen-business-pro' );
        $block_url_3         = ! empty( $instance['block_url_3'] ) ? $instance['block_url_3'] : '';

        $block_title_4       = ! empty( $instance['block_title_4'] ) ? $instance['block_title_4'] : '';
        $block_icon_4        = ! empty( $instance['block_icon_4'] ) ? $instance['block_icon_4'] : 'fa-cogs';
        $block_description_4 = ! empty( $instance['block_description_4'] ) ? $instance['block_description_4'] : '';
        $block_read_more_4   = ! empty( $instance['block_read_more_4'] ) ? $instance['block_read_more_4'] : __( 'Read more', 'wen-business-pro' );
        $block_url_4         = ! empty( $instance['block_url_4'] ) ? $instance['block_url_4'] : '';

        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        // Arrange data
        $service_arr = array();
        for ( $i=0; $i < 4 ; $i++ ) {
          $block = ( $i + 1 );
          $service_arr[ $i ] = array(
            'title'       => ${"block_title_" . $block},
            'icon'        => ${"block_icon_" . $block},
            'description' => ${"block_description_" . $block},
            'read_more'   => ${"block_read_more_" . $block},
            'url'         => ${"block_url_" . $block},
          );
        }
        // Clean up data
        $refined_arr = array();
        foreach ($service_arr as $key => $item) {
          if ( ! empty( $item['title'] ) ) {
            $refined_arr[] = $item;
          }
        }

        // Render content
        if ( ! empty( $refined_arr ) ) {
          $this->render_widget_content( $refined_arr );
        }

        //
        echo $after_widget;

    }

    function render_widget_content( $service_arr ){

      $column = count( $service_arr );
      switch ( $column ) {
        case 1:
          $block_item_class = 'col-sm-12';
          break;

        case 2:
          $block_item_class = 'col-sm-6';
          break;

        case 3:
          $block_item_class = 'col-sm-4';
          break;

        case 4:
          $block_item_class = 'col-sm-3';
          break;

        default:
          $block_item_class = '';
          break;
      }
      ?>
      <div class="service-block-list row">

        <?php foreach ( $service_arr as $key => $service ): ?>
          <?php
            $link_open  = '';
            $link_close = '';
            if ( ! empty( $service['url'] ) && ! empty( $service['read_more'] ) ) {
              $link_open  = '<a href="' . esc_url( $service['url'] ) . '" title="' . esc_attr( $service['title'] ) . '">';
              $link_close = '</a>';
            }
          ?>

          <div class="service-block-item <?php echo esc_attr( $block_item_class ); ?>">
            <div class="service-block-inner">

              <i class="<?php echo 'fa ' . esc_attr( $service['icon'] ); ?>"></i>
              <h3 class="service-item-title">
                <?php printf('%s%s%s', $link_open, esc_html( $service['title'] ), $link_close ); ?>
              </h3>
              <div class="service-block-item-excerpt">
                <?php echo esc_html( $service['description'] ); ?>
              </div><!-- .service-block-item-excerpt -->

              <?php if ( ! empty( $link_open ) ): ?>
                <a href="<?php echo esc_url( $service['url'] ); ?>" class="read-more" title="<?php echo esc_attr( $service['title'] )?>" ><?php echo esc_html( $service['read_more'] ); ?></a>
              <?php endif ?>

            </div><!-- .service-block-inner -->
          </div><!-- .service-block-item -->

        <?php endforeach ?>

      </div><!-- .service-block-list -->

      <?php


    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']    = strip_tags($new_instance['title']);
        $instance['subtitle'] = esc_html( $new_instance['subtitle'] );

        $instance['block_title_1']       = esc_html( $new_instance['block_title_1'] );
        $instance['block_icon_1']        = esc_attr( $new_instance['block_icon_1'] );
        $instance['block_description_1'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['block_description_1'] ) ) );
        $instance['block_read_more_1']   = esc_html( $new_instance['block_read_more_1'] );
        $instance['block_url_1']         = esc_url( $new_instance['block_url_1'] );

        $instance['block_title_2']       = esc_html( $new_instance['block_title_2'] );
        $instance['block_icon_2']        = esc_attr( $new_instance['block_icon_2'] );
        $instance['block_description_2'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['block_description_2'] ) ) );
        $instance['block_read_more_2']   = esc_html( $new_instance['block_read_more_2'] );
        $instance['block_url_2']         = esc_url( $new_instance['block_url_2'] );

        $instance['block_title_3']       = esc_html( $new_instance['block_title_3'] );
        $instance['block_icon_3']        = esc_attr( $new_instance['block_icon_3'] );
        $instance['block_description_3'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['block_description_3'] ) ) );
        $instance['block_read_more_3']   = esc_html( $new_instance['block_read_more_3'] );
        $instance['block_url_3']         = esc_url( $new_instance['block_url_3'] );

        $instance['block_title_4']       = esc_html( $new_instance['block_title_4'] );
        $instance['block_icon_4']        = esc_attr( $new_instance['block_icon_4'] );
        $instance['block_description_4'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['block_description_4'] ) ) );
        $instance['block_read_more_4']   = esc_html( $new_instance['block_read_more_4'] );
        $instance['block_url_4']         = esc_url( $new_instance['block_url_4'] );

        $instance['custom_class'] =   esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'               => '',
          'subtitle'            => '',

          'block_title_1'       => '',
          'block_icon_1'        => 'fa-cogs',
          'block_description_1' => '',
          'block_read_more_1'   => __( 'Read more', 'wen-business-pro' ),
          'block_url_1'         => '',

          'block_title_2'       => '',
          'block_icon_2'        => 'fa-cogs',
          'block_description_2' => '',
          'block_read_more_2'   => __( 'Read more', 'wen-business-pro' ),
          'block_url_2'         => '',

          'block_title_3'       => '',
          'block_icon_3'        => 'fa-cogs',
          'block_description_3' => '',
          'block_read_more_3'   => __( 'Read more', 'wen-business-pro' ),
          'block_url_3'         => '',

          'block_title_4'       => '',
          'block_icon_4'        => 'fa-cogs',
          'block_description_4' => '',
          'block_read_more_4'   => __( 'Read more', 'wen-business-pro' ),
          'block_url_4'         => '',

          'custom_class'        => '',
        ) );
        $title               = strip_tags( $instance['title'] );
        $subtitle            = esc_textarea( $instance['subtitle'] );

        $block_title_1       = esc_html( $instance['block_title_1'] );
        $block_icon_1        = esc_attr( $instance['block_icon_1'] );
        $block_description_1 = esc_textarea( $instance['block_description_1'] );
        $block_read_more_1   = esc_html( $instance['block_read_more_1'] );
        $block_url_1         = esc_url( $instance['block_url_1'] );

        $block_title_2       = esc_html( $instance['block_title_2'] );
        $block_icon_2        = esc_attr( $instance['block_icon_2'] );
        $block_description_2 = esc_textarea( $instance['block_description_2'] );
        $block_read_more_2   = esc_html( $instance['block_read_more_2'] );
        $block_url_2         = esc_url( $instance['block_url_2'] );

        $block_title_3       = esc_html( $instance['block_title_3'] );
        $block_icon_3        = esc_attr( $instance['block_icon_3'] );
        $block_description_3 = esc_textarea( $instance['block_description_3'] );
        $block_read_more_3   = esc_html( $instance['block_read_more_3'] );
        $block_url_3         = esc_url( $instance['block_url_3'] );

        $block_title_4       = esc_html( $instance['block_title_4'] );
        $block_icon_4        = esc_attr( $instance['block_icon_4'] );
        $block_description_4 = esc_textarea( $instance['block_description_4'] );
        $block_read_more_4   = esc_html( $instance['block_read_more_4'] );
        $block_url_4         = esc_url( $instance['block_url_4'] );

        $custom_class        = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e('Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e( 'Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle' ); ?>" name="<?php echo $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <h4 class="block-heading"><?php printf( __( 'Block %d','wen-business-pro' ), 1 ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_title_1' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_title_1' ); ?>" name="<?php echo $this->get_field_name( 'block_title_1' ); ?>" type="text" value="<?php echo esc_attr( $block_title_1 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_icon_1' ); ?>"><?php _e( 'Icon:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_icon_1' ); ?>" name="<?php echo $this->get_field_name( 'block_icon_1' ); ?>" type="text" value="<?php echo esc_attr( $block_icon_1 ); ?>" />&nbsp;<em><?php _e( 'eg: fa-cogs', 'wen-business-pro' ); ?>&nbsp;<a href="<?php echo esc_url( 'http://fontawesome.io/cheatsheet/' ); ?>" target="_blank" title="<?php _e( 'View Reference', 'wen-business-pro' ); ?>"><?php _e( 'Reference', 'wen-business-pro' ); ?></a></em>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_description_1' ); ?>"><?php _e( 'Intro:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="3" id="<?php echo $this->get_field_id( 'block_description_1' ); ?>" name="<?php echo $this->get_field_name( 'block_description_1' ); ?>"><?php echo $block_description_1; ?></textarea>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_read_more_1' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_read_more_1' ); ?>" name="<?php echo $this->get_field_name( 'block_read_more_1' ); ?>" type="text" value="<?php echo esc_attr( $block_read_more_1 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_url_1' ); ?>"><?php _e( 'URL:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_url_1' ); ?>" name="<?php echo $this->get_field_name( 'block_url_1' ); ?>" type="text" value="<?php echo esc_attr( $block_url_1 ); ?>" style="min-width:310px;" />
        </p>
        <h4 class="block-heading"><?php printf( __( 'Block %d','wen-business-pro' ), 2 ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_title_2' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_title_2' ); ?>" name="<?php echo $this->get_field_name( 'block_title_2' ); ?>" type="text" value="<?php echo esc_attr( $block_title_2 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_icon_2' ); ?>"><?php _e( 'Icon:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_icon_2' ); ?>" name="<?php echo $this->get_field_name( 'block_icon_2' ); ?>" type="text" value="<?php echo esc_attr( $block_icon_2 ); ?>" />&nbsp;<em><?php _e( 'eg: fa-cogs', 'wen-business-pro' ); ?>&nbsp;<a href="<?php echo esc_url( 'http://fontawesome.io/cheatsheet/' ); ?>" target="_blank" title="<?php _e( 'View Reference', 'wen-business-pro' ); ?>"><?php _e( 'Reference', 'wen-business-pro' ); ?></a></em>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_description_2' ); ?>"><?php _e( 'Intro:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="3" id="<?php echo $this->get_field_id( 'block_description_2' ); ?>" name="<?php echo $this->get_field_name( 'block_description_2' ); ?>"><?php echo $block_description_2; ?></textarea>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_read_more_2' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_read_more_2' ); ?>" name="<?php echo $this->get_field_name( 'block_read_more_2' ); ?>" type="text" value="<?php echo esc_attr( $block_read_more_2 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_url_2' ); ?>"><?php _e( 'URL:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_url_2' ); ?>" name="<?php echo $this->get_field_name( 'block_url_2' ); ?>" type="text" value="<?php echo esc_attr( $block_url_2 ); ?>" style="min-width:310px;" />
        </p>
        <h4 class="block-heading"><?php printf( __( 'Block %d','wen-business-pro' ), 3 ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_title_3' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_title_3' ); ?>" name="<?php echo $this->get_field_name( 'block_title_3' ); ?>" type="text" value="<?php echo esc_attr( $block_title_3 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_icon_3' ); ?>"><?php _e( 'Icon:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_icon_3' ); ?>" name="<?php echo $this->get_field_name( 'block_icon_3' ); ?>" type="text" value="<?php echo esc_attr( $block_icon_3 ); ?>" />&nbsp;<em><?php _e( 'eg: fa-cogs', 'wen-business-pro' ); ?>&nbsp;<a href="<?php echo esc_url( 'http://fontawesome.io/cheatsheet/' ); ?>" target="_blank" title="<?php _e( 'View Reference', 'wen-business-pro' ); ?>"><?php _e( 'Reference', 'wen-business-pro' ); ?></a></em>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_description_3' ); ?>"><?php _e( 'Intro:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="3" id="<?php echo $this->get_field_id( 'block_description_3' ); ?>" name="<?php echo $this->get_field_name( 'block_description_3' ); ?>"><?php echo $block_description_3; ?></textarea>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_read_more_3' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_read_more_3' ); ?>" name="<?php echo $this->get_field_name( 'block_read_more_3' ); ?>" type="text" value="<?php echo esc_attr( $block_read_more_3 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_url_3' ); ?>"><?php _e( 'URL:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_url_3' ); ?>" name="<?php echo $this->get_field_name( 'block_url_3' ); ?>" type="text" value="<?php echo esc_attr( $block_url_3 ); ?>" style="min-width:310px;" />
        </p>
        <h4 class="block-heading"><?php printf( __( 'Block %d','wen-business-pro' ), 4 ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_title_4' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_title_4' ); ?>" name="<?php echo $this->get_field_name( 'block_title_4' ); ?>" type="text" value="<?php echo esc_attr( $block_title_4 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_icon_4' ); ?>"><?php _e( 'Icon:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_icon_4' ); ?>" name="<?php echo $this->get_field_name( 'block_icon_4' ); ?>" type="text" value="<?php echo esc_attr( $block_icon_4 ); ?>" />&nbsp;<em><?php _e( 'eg: fa-cogs', 'wen-business-pro' ); ?>&nbsp;<a href="<?php echo esc_url( 'http://fontawesome.io/cheatsheet/' ); ?>" target="_blank" title="<?php _e( 'View Reference', 'wen-business-pro' ); ?>"><?php _e( 'Reference', 'wen-business-pro' ); ?></a></em>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_description_4' ); ?>"><?php _e( 'Intro:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="3" id="<?php echo $this->get_field_id( 'block_description_4' ); ?>" name="<?php echo $this->get_field_name( 'block_description_4' ); ?>"><?php echo $block_description_4; ?></textarea>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_read_more_4' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input  id="<?php echo $this->get_field_id( 'block_read_more_4' ); ?>" name="<?php echo $this->get_field_name( 'block_read_more_4' ); ?>" type="text" value="<?php echo esc_attr( $block_read_more_4 ); ?>" style="min-width:310px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'block_url_4' ); ?>"><?php _e( 'URL:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'block_url_4' ); ?>" name="<?php echo $this->get_field_name( 'block_url_4' ); ?>" type="text" value="<?php echo esc_attr( $block_url_4 ); ?>" style="min-width:310px;" />
        </p>
        <?php
      }
  }

endif;


if ( ! class_exists( 'WEN_Business_Latest_Works_Widget' ) ) :

  /**
   * Latest Works Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Latest_Works_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_latest_works',
                  'description' => __( 'Latest Works Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-latest-works', __( 'Business Latest Works Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title          = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle       = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $post_category     = ! empty( $instance['post_category'] ) ? $instance['post_category'] : 0;
        $post_column       = ! empty( $instance['post_column'] ) ? $instance['post_column'] : 4;
        $featured_image    = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'thumbnail';
        $post_number       = ! empty( $instance['post_number'] ) ? $instance['post_number'] : 4;
        $excerpt_length    = ! empty( $instance['excerpt_length'] ) ? $instance['excerpt_length'] : 40;
        $post_order_by     = ! empty( $instance['post_order_by'] ) ? $instance['post_order_by'] : 'date';
        $post_order        = ! empty( $instance['post_order'] ) ? $instance['post_order'] : 'desc';
        $more_text         = ! empty( $instance['more_text'] ) ? $instance['more_text'] : __( 'Read more','wen-business-pro' );
        $disable_date      = ! empty( $instance['disable_date'] ) ? $instance['disable_date'] : false ;
        $disable_comment   = ! empty( $instance['disable_comment'] ) ? $instance['disable_comment'] : false ;
        $disable_excerpt   = ! empty( $instance['disable_excerpt'] ) ? $instance['disable_excerpt'] : false ;
        $disable_more_text = ! empty( $instance['disable_more_text'] ) ? $instance['disable_more_text'] : false ;
        $custom_class   = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        // Validation
        // Order
        if ( in_array( $post_order, array( 'asc', 'desc' ) ) ) {
          $post_order = strtoupper( $post_order );
        }
        else{
          $post_order = 'DESC';
        }
        // Order By
        switch ( $post_order_by ) {
          case 'date':
            $post_order_by = 'date';
            break;
          case 'title':
            $post_order_by = 'title';
            break;
          case 'comment-count':
            $post_order_by = 'comment_count';
            break;
          case 'random':
            $post_order_by = 'rand';
            break;
          case 'menu-order':
            $post_order_by = 'menu_order';
            break;
          default:
            $post_order_by = 'date';
            break;
        }
        // Column class
        switch ( $post_column ) {
          case 1:
            $column_class = 'col-sm-12';
            break;
          case 2:
            $column_class = 'col-sm-6';
            break;
          case 3:
            $column_class = 'col-sm-4';
            break;
          case 4:
            $column_class = 'col-sm-3';
            break;
          case 5:
            $column_class = 'col-sm-5ths';
            break;
          case 6:
            $column_class = 'col-sm-2';
            break;
          default:
            $column_class = '';
            break;
        }


        // Add Custom class
        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        ?>
        <?php
          $qargs = array(
            'posts_per_page' => $post_number,
            'no_found_rows'  => true,
            'orderby'        => $post_order_by,
            'order'          => $post_order,
            'meta_query'     => array(
                array( 'key' => '_thumbnail_id' ), //Show only posts with featured images
              )
            );
          if ( absint( $post_category ) > 0  ) {
            $qargs['cat'] = $post_category;
          }

          $all_posts = get_posts( $qargs );
        ?>
        <?php if ( ! empty( $all_posts ) ): ?>


          <?php global $post; ?>

          <div class="latest-works-widget">

            <div class="row">

              <?php foreach ( $all_posts as $key => $post ): ?>
                <?php setup_postdata( $post ); ?>

                <div class="latest-works-item <?php echo esc_attr( $column_class ); ?>">

                  <div class="latest-works-thumb">
                    <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>">
                      <?php
                        $img_attributes = array( 'class' => 'aligncenter' );
                        the_post_thumbnail( $featured_image, $img_attributes );
                      ?>
                    </a>
                  </div><!-- .latest-works-thumb -->


                </div><!-- .latest-works-item -->

              <?php endforeach ?>

            </div><!-- .row -->

          </div><!-- .latest-works-widget -->

          <?php wp_reset_postdata(); // Reset ?>

        <?php endif; ?>
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']             = strip_tags($new_instance['title']);
        $instance['subtitle']          = esc_html( $new_instance['subtitle'] );
        $instance['post_category']     = absint( $new_instance['post_category'] );
        $instance['post_number']       = absint( $new_instance['post_number'] );
        $instance['post_column']       = absint( $new_instance['post_column'] );
        $instance['post_order_by']     = esc_attr( $new_instance['post_order_by'] );
        $instance['post_order']        = esc_attr( $new_instance['post_order'] );
        $instance['featured_image']    = esc_attr( $new_instance['featured_image'] );
        $instance['custom_class']      = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'             => '',
          'subtitle'          => '',
          'post_category'     => '',
          'post_column'       => 4,
          'featured_image'    => 'thumbnail',
          'post_number'       => 4,
          'post_order_by'     => 'date',
          'post_order'        => 'desc',
          'custom_class'      => '',
        ) );
        $title             = strip_tags( $instance['title'] );
        $subtitle          = esc_textarea( $instance['subtitle'] );
        $post_category     = absint( $instance['post_category'] );
        $post_column       = absint( $instance['post_column'] );
        $featured_image    = esc_attr( $instance['featured_image'] );
        $post_number       = absint( $instance['post_number'] );
        $post_order_by     = esc_attr( $instance['post_order_by'] );
        $post_order        = esc_attr( $instance['post_order'] );
        $custom_class      = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e( 'Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle' ); ?>" name="<?php echo $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_category' ); ?>"><?php _e( 'Category:', 'wen-business-pro' ); ?></label>
          <?php
            $cat_args = array(
                'orderby'         => 'name',
                'hide_empty'      => 0,
                'taxonomy'        => 'category',
                'name'            => $this->get_field_name('post_category'),
                'id'              => $this->get_field_id('post_category'),
                'selected'        => $post_category,
                'show_option_all' => __( 'All Categories','wen-business-pro' ),
              );
            wp_dropdown_categories( $cat_args );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_column' ); ?>"><?php _e('Number of Columns:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_post_columns( array(
              'id'       => $this->get_field_id( 'post_column' ),
              'name'     => $this->get_field_name( 'post_column' ),
              'selected' => $post_column,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_image' ); ?>"><?php _e( 'Featured Image:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_image_sizes( array(
              'id'       => $this->get_field_id( 'featured_image' ),
              'name'     => $this->get_field_name( 'featured_image' ),
              'selected' => $featured_image,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_number' ); ?>"><?php _e('Number of Posts:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'post_number' ); ?>" name="<?php echo $this->get_field_name( 'post_number' ); ?>" type="number" value="<?php echo esc_attr( $post_number ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order_by' ); ?>"><?php _e( 'Order By:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_post_order_by( array(
              'id'       => $this->get_field_id( 'post_order_by' ),
              'name'     => $this->get_field_name( 'post_order_by' ),
              'selected' => $post_order_by,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'post_order' ); ?>"><?php _e( 'Order:', 'wen-business-pro' ); ?></label>
          <select id="<?php echo $this->get_field_id( 'post_order' ); ?>" name="<?php echo $this->get_field_name( 'post_order' ); ?>">
            <option value="asc" <?php selected( $post_order, 'asc' ) ?>><?php _e( 'Ascending', 'wen-business-pro' ) ?></option>
            <option value="desc" <?php selected( $post_order, 'desc' ) ?>><?php _e( 'Descending', 'wen-business-pro' ) ?></option>
          </select>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }

    function dropdown_post_columns( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = array(
        '1' => sprintf( __( '%d Column','wen-business-pro' ), 1 ),
        '2' => sprintf( __( '%d Columns','wen-business-pro' ), 2 ),
        '3' => sprintf( __( '%d Columns','wen-business-pro' ), 3 ),
        '4' => sprintf( __( '%d Columns','wen-business-pro' ), 4 ),
      );

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    function dropdown_post_order_by( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = array(
        'date'          => __( 'Date','wen-business-pro' ),
        'title'         => __( 'Title','wen-business-pro' ),
        'comment-count' => __( 'Comment Count','wen-business-pro' ),
        'menu-order'    => __( 'Menu Order','wen-business-pro' ),
        'random'        => __( 'Random','wen-business-pro' ),
      );

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    function dropdown_image_sizes( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = $this->get_image_sizes_options();

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

    private function get_image_sizes_options(){

      global $_wp_additional_image_sizes;
      $get_intermediate_image_sizes = get_intermediate_image_sizes();
      $choices = array();
      foreach ( array( 'thumbnail', 'medium', 'large' ) as $key => $_size ) {
        $choices[ $_size ] = $_size . ' ('. get_option( $_size . '_size_w' ) . 'x' . get_option( $_size . '_size_h' ) . ')';
      }
      $choices['full'] = __( 'full (original)', 'wen-business-pro' );
      if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {

        foreach ($_wp_additional_image_sizes as $key => $size ) {
          $choices[ $key ] = $key . ' ('. $size['width'] . 'x' . $size['height'] . ')';
        }

      }
      return $choices;
    }

  }

endif;

if ( ! class_exists( 'WEN_Business_Contact_Widget' ) ) :

  /**
   * Contact Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Contact_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_contact',
                  'description' => __( 'Contact Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-contact', __( 'Business Contact Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title           = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle        = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $contact_address = ! empty( $instance['contact_address'] ) ? $instance['contact_address'] : '';
        $contact_phone   = ! empty( $instance['contact_phone'] ) ? $instance['contact_phone'] : '';
        $contact_email   = ! empty( $instance['contact_email'] ) ? $instance['contact_email'] : '';
        $contact_time    = ! empty( $instance['contact_time'] ) ? $instance['contact_time'] : '';
        $custom_class    = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        $contact_arr = array();
        // Arrange contact address
        if ( ! empty( $contact_address )) {
          $contact_arr['address'] = ! empty( $instance['filter'] ) ? wpautop( $contact_address ) : $contact_address;
        }
        if ( ! empty( $contact_phone )) {
          $contact_arr['phone'] = $contact_phone;
        }
        if ( ! empty( $contact_email )) {
          $contact_arr['email'] = $contact_email;
        }
        if ( ! empty( $contact_time )) {
          $contact_arr['time'] = $contact_time;
        }
        // Render now
        if ( ! empty( $contact_arr ) ){
          echo '<div class="contact-widget">';
          $this->render_contact( $contact_arr );
          echo '</div>';
        }
        //
        echo $after_widget;

    }

    function render_contact( $contact_arr ){
      if ( empty( $contact_arr ) ) {
        return;
      }
      echo '<ul>';
      foreach ( $contact_arr as $key => $c ) {
        echo '<li class="contact-' . esc_attr( $key ) . '">';
        echo $c;
        echo '</li>';
      }
      echo '</ul>';
    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']    = strip_tags($new_instance['title']);
        $instance['subtitle'] = esc_html( $new_instance['subtitle'] );

        if ( current_user_can( 'unfiltered_html' ) )
          $instance['contact_address'] =  $new_instance['contact_address'];
        else
          $instance['contact_address'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['contact_address'] ) ) ); // wp_filter_post_kses() expects slashed

        $instance['filter']        = isset( $new_instance['filter'] );
        $instance['contact_phone'] = esc_html( $new_instance['contact_phone'] );
        $instance['contact_email'] = esc_html( $new_instance['contact_email'] );
        $instance['contact_time']  = esc_html( $new_instance['contact_time'] );
        $instance['custom_class']  = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'           => '',
          'subtitle'        => '',
          'contact_address' => '',
          'filter'          => 1,
          'contact_phone'   => '',
          'contact_email'   => '',
          'contact_time'    => '',
          'custom_class'    => '',
        ) );
        $title           = strip_tags( $instance['title'] );
        $subtitle        = esc_textarea( $instance['subtitle'] );
        $contact_address = esc_textarea( $instance['contact_address'] );
        $filter          = esc_attr( $instance['filter'] );
        $contact_phone   = esc_textarea( $instance['contact_phone'] );
        $contact_email   = esc_textarea( $instance['contact_email'] );
        $contact_time    = esc_textarea( $instance['contact_time'] );
        $custom_class    = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e('Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e('Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle'); ?>" name="<?php echo $this->get_field_name('subtitle'); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'contact_address' ); ?>"><?php _e( 'Address:', 'wen-business-pro' ); ?></label>
          <textarea class="widefat" rows="5" id="<?php echo $this->get_field_id( 'contact_address' ); ?>" name="<?php echo $this->get_field_name( 'contact_address' ); ?>"><?php echo $contact_address; ?></textarea>
        </p>
        <p><input id="<?php echo $this->get_field_id('filter'); ?>" name="<?php echo $this->get_field_name('filter'); ?>" type="checkbox" <?php checked(isset($instance['filter']) ? $instance['filter'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id('filter'); ?>"><?php _e( 'Automatically add paragraphs', 'wen-business-pro' ); ?></label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'contact_phone' ); ?>"><?php _e( 'Phone:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'contact_phone'); ?>" name="<?php echo $this->get_field_name('contact_phone'); ?>" type="text" value="<?php echo esc_attr( $contact_phone ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'contact_email' ); ?>"><?php _e( 'Email:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'contact_email'); ?>" name="<?php echo $this->get_field_name('contact_email'); ?>" type="email" value="<?php echo esc_attr( $contact_email ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'contact_time' ); ?>"><?php _e( 'Time:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'contact_time'); ?>" name="<?php echo $this->get_field_name('contact_time'); ?>" type="text" value="<?php echo esc_attr( $contact_time ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }
  }

endif;


if ( ! class_exists( 'WEN_Business_Special_Pages_Widget' ) ) :

  /**
   * Special Pages Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Special_Pages_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_special_pages',
                  'description' => __( 'Special Pages Widget', 'wen-business-pro' )
              );

      parent::__construct('wen-business-special-pages', __( 'Business Special Pages Widget', 'wen-business-pro'), $opts);
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title               = apply_filters( 'widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base );
        $subtitle            = apply_filters('widget_subtitle', empty($instance['subtitle']) ? '' : $instance['subtitle'], $instance, $this->id_base );
        $featured_image      = ! empty( $instance['featured_image'] ) ? $instance['featured_image'] : 'thumbnail';
        $special_page_1      = ! empty( $instance['special_page_1'] ) ? $instance['special_page_1'] : 0;
        $special_page_2      = ! empty( $instance['special_page_2'] ) ? $instance['special_page_2'] : 0;
        $special_page_3      = ! empty( $instance['special_page_3'] ) ? $instance['special_page_3'] : 0;
        $special_page_4      = ! empty( $instance['special_page_4'] ) ? $instance['special_page_4'] : 0;
        $excerpt_length      = ! empty( $instance['excerpt_length'] ) ? $instance['excerpt_length'] : 40;
        $more_text           = ! empty( $instance['more_text'] ) ? $instance['more_text'] : __( 'Read more','wen-business-pro' );
        $special_button_text = ! empty( $instance['special_button_text'] ) ? $instance['special_button_text'] : __( 'Know more','wen-business-pro' );
        $special_button_url  = ! empty( $instance['special_button_url'] ) ? $instance['special_button_url'] : '';
        $disable_excerpt     = ! empty( $instance['disable_excerpt'] ) ? $instance['disable_excerpt'] : false ;
        $disable_more_text   = ! empty( $instance['disable_more_text'] ) ? $instance['disable_more_text'] : false ;
        $custom_class        = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );

        // Validation
        $pages_arr = array();
        for ( $i = 1; $i <= 4 ; $i++ ) {
          $pg = absint( ${'special_page_' . $i } );
          if ( $pg > 0 ) {
            $pages_arr[] = $pg;
          }
        }

        // Column class
        $post_column = count( $pages_arr );
        switch ( $post_column ) {
          case 1:
            $column_class = 'col-sm-12';
            break;
          case 2:
            $column_class = 'col-sm-6';
            break;
          case 3:
            $column_class = 'col-sm-4';
            break;
          case 4:
            $column_class = 'col-sm-3';
            break;
          case 5:
            $column_class = 'col-sm-5ths';
            break;
          case 6:
            $column_class = 'col-sm-2';
            break;
          default:
            $column_class = '';
            break;
        }


        // Add Custom class
        if ( $custom_class ) {
          $before_widget = str_replace( 'class="', 'class="'. $custom_class . ' ', $before_widget );
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;

        // Sub Title
        if ( $subtitle ) printf( '%s%s%s', '<h4 class="widget-subtitle">', $subtitle, '</h4>' );
        //
        ?>
        <?php
          $all_posts = null;

          if ( count( $pages_arr ) > 0 ) {
            $qargs = array(
              'no_found_rows' => true,
              'post_type'     => 'page',
              'post__in'      => $pages_arr,
              'orderby'       => 'post__in',
              );

            $all_posts = get_posts( $qargs );
          }

        ?>
        <?php if ( ! empty( $all_posts ) ): ?>


          <?php global $post; ?>

          <div class="special-pages-widget">

            <div class="row">

              <?php foreach ( $all_posts as $key => $post ): ?>
                <?php setup_postdata( $post ); ?>

                <div class="<?php echo esc_attr( $column_class ); ?>">
                  <div class="special-pages-item">

                      <?php if ( 'disable' != $featured_image && has_post_thumbnail() ): ?>
                        <div class="special-pages-thumb">
                          <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>">
                            <?php
                              $img_attributes = array( 'class' => 'aligncenter' );
                              the_post_thumbnail( $featured_image, $img_attributes );
                            ?>
                          </a>
                        </div><!-- .special-pages-thumb -->
                      <?php endif ?>
                      <div class="special-pages-text-wrap">
                        <h3 class="special-pages-title">
                          <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>"><?php the_title(); ?></a>
                        </h3><!-- .special-pages-title -->

                        <?php if ( false == $disable_excerpt ): ?>
                          <div class="special-pages-summary">
                            <?php
                            $excerpt = wen_business_the_excerpt( $excerpt_length, $post );
                            echo wp_kses_post( wpautop( $excerpt ) );
                            ?>
                          </div><!-- .special-pages-summary -->
                        <?php endif ?>
                        <?php if ( false == $disable_more_text ): ?>
                          <div class="special-pages-read-more"><a href="<?php the_permalink(); ?>" class="read-more" title="<?php the_title_attribute(); ?>"><?php echo esc_html( $more_text ); ?></a></div><!-- .special-pages-read-more -->
                        <?php endif ?>
                      </div><!-- .special-pages-text-wrap -->
                  </div><!-- .special-pages-item -->

                </div><!-- .special-pages-item .col-sm-3 -->

              <?php endforeach ?>

            </div><!-- .row -->

            <?php if ( ! empty( $special_button_text ) && ! empty( $special_button_url ) ): ?>

              <div class="special-button-wrap">
                <div class="special-button-wrap-inner">
                  <a href="<?php echo esc_url( $special_button_url ); ?>" class="special-button"><?php echo esc_html( $special_button_text ); ?></a>
                </div><!-- .special-button-wrap-inner -->
              </div><!-- .special-button-wrap -->

            <?php endif ?>

          </div><!-- .special-pages-widget -->

          <?php wp_reset_postdata(); // Reset ?>

        <?php endif; ?>
        <?php
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']               = strip_tags($new_instance['title']);
        $instance['subtitle']            = esc_html( $new_instance['subtitle'] );
        $instance['special_page_1']      = absint( $new_instance['special_page_1'] );
        $instance['special_page_2']      = absint( $new_instance['special_page_2'] );
        $instance['special_page_3']      = absint( $new_instance['special_page_3'] );
        $instance['special_page_4']      = absint( $new_instance['special_page_4'] );
        $instance['excerpt_length']      = absint( $new_instance['excerpt_length'] );
        $instance['featured_image']      = esc_attr( $new_instance['featured_image'] );
        $instance['disable_excerpt']     = isset( $new_instance['disable_excerpt'] );
        $instance['disable_more_text']   = isset( $new_instance['disable_more_text'] );
        $instance['more_text']           = esc_attr( $new_instance['more_text'] );
        $instance['special_button_text'] = esc_attr( $new_instance['special_button_text'] );
        $instance['special_button_url']  = esc_url( $new_instance['special_button_url'] );
        $instance['custom_class']        = esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'               => '',
          'subtitle'            => '',
          'special_page_1'      => 0,
          'special_page_2'      => 0,
          'special_page_3'      => 0,
          'special_page_4'      => 0,
          'featured_image'      => 'thumbnail',
          'excerpt_length'      => 40,
          'more_text'           => __( 'Read more', 'wen-business-pro' ),
          'disable_excerpt'     => 0,
          'disable_more_text'   => 0,
          'special_button_text' => __( 'Know more', 'wen-business-pro' ),
          'special_button_url'  => '',
          'custom_class'        => '',
        ) );
        $title               = strip_tags( $instance['title'] );
        $subtitle            = esc_textarea( $instance['subtitle'] );
        $special_page_1      = absint( $instance['special_page_1'] );
        $special_page_2      = absint( $instance['special_page_2'] );
        $special_page_3      = absint( $instance['special_page_3'] );
        $special_page_4      = absint( $instance['special_page_4'] );
        $featured_image      = esc_attr( $instance['featured_image'] );
        $excerpt_length      = absint( $instance['excerpt_length'] );
        $more_text           = strip_tags( $instance['more_text'] );
        $disable_excerpt     = esc_attr( $instance['disable_excerpt'] );
        $disable_more_text   = esc_attr( $instance['disable_more_text'] );
        $special_button_text = strip_tags( $instance['special_button_text'] );
        $special_button_url  = esc_url( $instance['special_button_url'] );
        $custom_class        = esc_attr( $instance['custom_class'] );
        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'subtitle' ); ?>"><?php _e( 'Sub Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'subtitle' ); ?>" name="<?php echo $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_page_1' ); ?>"><?php printf( __( 'Page #%d:', 'wen-business-pro' ), 1 ); ?></label>
          <?php
            wp_dropdown_pages( array(
              'id'               => $this->get_field_id( 'special_page_1' ),
              'name'             => $this->get_field_name( 'special_page_1' ),
              'selected'         => $special_page_1,
              'show_option_none' => __( '&mdash; Select &mdash;', 'wen-business-pro' ),
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_page_2' ); ?>"><?php printf( __( 'Page #%d:', 'wen-business-pro' ), 2 ); ?></label>
          <?php
            wp_dropdown_pages( array(
              'id'               => $this->get_field_id( 'special_page_2' ),
              'name'             => $this->get_field_name( 'special_page_2' ),
              'selected'         => $special_page_2,
              'show_option_none' => __( '&mdash; Select &mdash;', 'wen-business-pro' ),
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_page_3' ); ?>"><?php printf( __( 'Page #%d:', 'wen-business-pro' ), 3 ); ?></label>
          <?php
            wp_dropdown_pages( array(
              'id'               => $this->get_field_id( 'special_page_3' ),
              'name'             => $this->get_field_name( 'special_page_3' ),
              'selected'         => $special_page_3,
              'show_option_none' => __( '&mdash; Select &mdash;', 'wen-business-pro' ),
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_page_4' ); ?>"><?php printf( __( 'Page #%d:', 'wen-business-pro' ), 4 ); ?></label>
          <?php
            wp_dropdown_pages( array(
              'id'               => $this->get_field_id( 'special_page_4' ),
              'name'             => $this->get_field_name( 'special_page_4' ),
              'selected'         => $special_page_4,
              'show_option_none' => __( '&mdash; Select &mdash;', 'wen-business-pro' ),
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'featured_image' ); ?>"><?php _e( 'Featured Image:', 'wen-business-pro' ); ?></label>
          <?php
            $this->dropdown_image_sizes( array(
              'id'       => $this->get_field_id( 'featured_image' ),
              'name'     => $this->get_field_name( 'featured_image' ),
              'selected' => $featured_image,
              )
            );
          ?>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'excerpt_length' ); ?>"><?php _e('Excerpt Length:', 'wen-business-pro' ); ?></label>
          <input class="widefat1" id="<?php echo $this->get_field_id( 'excerpt_length' ); ?>" name="<?php echo $this->get_field_name( 'excerpt_length' ); ?>" type="number" value="<?php echo esc_attr( $excerpt_length ); ?>" min="1" style="max-width:50px;" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'more_text' ); ?>"><?php _e( 'More Text:', 'wen-business-pro' ); ?></label>
          <input id="<?php echo $this->get_field_id( 'more_text'); ?>" name="<?php echo $this->get_field_name( 'more_text' ); ?>" type="text" value="<?php echo esc_attr( $more_text ); ?>" />
        </p>
        <p>
          <input id="<?php echo $this->get_field_id( 'disable_excerpt' ); ?>" name="<?php echo $this->get_field_name( 'disable_excerpt' ); ?>" type="checkbox" <?php checked(isset($instance['disable_excerpt']) ? $instance['disable_excerpt'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_excerpt' ); ?>"><?php _e( 'Disable Excerpt', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <input id="<?php echo $this->get_field_id( 'disable_more_text' ); ?>" name="<?php echo $this->get_field_name( 'disable_more_text' ); ?>" type="checkbox" <?php checked(isset($instance['disable_more_text']) ? $instance['disable_more_text'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'disable_more_text' ); ?>"><?php _e( 'Disable More Text', 'wen-business-pro' ); ?>
          </label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_button_text' ); ?>"><?php _e( 'Special Button Text:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'special_button_text'); ?>" name="<?php echo $this->get_field_name( 'special_button_text' ); ?>" type="text" value="<?php echo esc_attr( $special_button_text ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'special_button_url' ); ?>"><?php _e( 'Special Button URL:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'special_button_url'); ?>" name="<?php echo $this->get_field_name( 'special_button_url' ); ?>" type="text" value="<?php echo esc_url( $special_button_url ); ?>" />
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class'); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <?php
      }

    function dropdown_image_sizes( $args ){
      $defaults = array(
        'id'       => '',
        'name'     => '',
        'selected' => 0,
        'echo'     => 1,
      );

      $r = wp_parse_args( $args, $defaults );
      $output = '';

      $choices = wen_business_get_image_sizes_options();

      if ( ! empty( $choices ) ) {

        $output = "<select name='" . esc_attr( $r['name'] ) . "' id='" . esc_attr( $r['id'] ) . "'>\n";
        foreach ( $choices as $key => $choice ) {
          $output .= '<option value="' . esc_attr( $key ) . '" ';
          $output .= selected( $r['selected'], $key, false );
          $output .= '>' . esc_html( $choice ) . '</option>\n';
        }
        $output .= "</select>\n";
      }

      if ( $r['echo'] ) {
        echo $output;
      }
      return $output;

    }

  }

endif;



if ( ! class_exists( 'WEN_Business_Ad_Widget' ) ) :

  /**
   * Ad Widget Class
   *
   * @since WEN Business 1.0
   *
   */
  class WEN_Business_Ad_Widget extends WP_Widget {

    function __construct() {
      $opts = array(
                  'classname'   => 'wen_business_widget_ad',
                  'description' => __( 'Ad Widget', 'wen-business-pro' )
              );

      parent::__construct( 'wen-business-ad', __( 'Business Ad', 'wen-business-pro' ), $opts );
    }


    function widget( $args, $instance ) {
        extract( $args );

        $title              = apply_filters('widget_title', empty($instance['title']) ? '' : $instance['title'], $instance, $this->id_base);
        $ad_code           = ! empty( $instance['ad_code'] ) ? $instance['ad_code'] : '';
        $image_url          = empty($instance['image_url']) ? '' : $instance['image_url'];
        $link_url           = empty($instance['link_url']) ? '' : $instance['link_url'];
        $open_in_new_window = empty($instance['open_in_new_window']) ? '' : $instance['open_in_new_window'];
        $alt_text           = empty($instance['alt_text']) ? '' : $instance['alt_text'];
        $custom_class       = apply_filters( 'widget_custom_class', empty( $instance['custom_class'] ) ? '' : $instance['custom_class'], $instance, $this->id_base );


        if ( $custom_class ) {
          $before_widget = str_replace('class="', 'class="'. $custom_class . ' ', $before_widget);
        }

        echo $before_widget;

        // Title
        if ( $title ) echo $before_title . $title . $after_title;
        //
        $ad_content = '';
        if ( ! empty( $ad_code ) ) {
          $ad_content = $ad_code;
        }
        if ( empty( $ad_code ) && ! empty( $image_url ) ) {
          // make html
          $html = '';
          $img_html = '<img src="' . esc_url( $image_url ) . '" alt="'. esc_attr( $alt_text ) . '" />';
          $link_open = '';
          $link_close = '';
          if ( ! empty( $link_url ) ) {
            $target_text = ( true == $open_in_new_window ) ? ' target="_blank" ' : '';
            $link_open = '<a href="' . esc_url( $link_url ) . '" ' . $target_text . '>';
            $link_close = '</a>';
          }
          $ad_content = $link_open . $img_html .  $link_close;
        }
        if ( ! empty( $ad_content ) ) {
          echo '<div class="ad-widget">';
          echo $ad_content;
          echo '</div>';
        }
        //
        echo $after_widget;

    }

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $instance['title']        =   esc_html( strip_tags($new_instance['title']) );
        if ( current_user_can( 'unfiltered_html' ) ){
          $instance['ad_code'] =  $new_instance['ad_code'];
        }
        else{
          $instance['ad_code'] = stripslashes( wp_filter_post_kses( addslashes($new_instance['ad_code']) ) ); // wp_filter_post_kses() expects slashed
        }
        $instance['image_url']          =   esc_url( $new_instance['image_url'] );
        $instance['link_url']           =   esc_url( $new_instance['link_url'] );
        $instance['alt_text']           =   esc_attr( $new_instance['alt_text'] );
        $instance['open_in_new_window'] = isset( $new_instance['open_in_new_window'] );
        $instance['custom_class']       =   esc_attr( $new_instance['custom_class'] );

        return $instance;
    }

      function form( $instance ) {

        //Defaults
        $instance = wp_parse_args( (array) $instance, array(
          'title'              => '',
          'ad_code'            => '',
          'image_url'          => '',
          'link_url'           => '',
          'open_in_new_window' => true,
          'alt_text'           => '',
          'custom_class'       => '',
        ) );
        $title              = esc_attr( $instance['title'] );
        $ad_code            = esc_textarea( $instance['ad_code'] );
        $image_url          = esc_url( $instance['image_url'] );
        $link_url           = esc_url( $instance['link_url'] );
        $open_in_new_window = esc_attr( $instance['open_in_new_window'] );
        $alt_text           = esc_attr( $instance['alt_text'] );
        $custom_class       = esc_attr( $instance['custom_class'] );

        ?>
        <p>
          <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <hr />
        <h4><?php _e( 'Option 1 : Ad Code', 'wen-business-pro' ); ?></h4>
        <p>
          <label for="<?php echo $this->get_field_id( 'ad_code' ); ?>"><?php _e( 'Adv Code:', 'wen-business-pro' ); ?></label>
           <textarea class="widefat" rows="3" id="<?php echo $this->get_field_id( 'ad_code' ); ?>" name="<?php echo $this->get_field_name( 'ad_code' ); ?>"><?php echo $ad_code; ?></textarea>
        </p>
        <hr />
        <h4><?php _e( 'Option 2 : Image', 'wen-business-pro' ); ?></h4>
        <div>
          <label for="<?php echo $this->get_field_id( 'image_url' ); ?>"><?php _e( 'Image URL:', 'wen-business-pro' ); ?></label><br/>
          <input id="<?php echo $this->get_field_id( 'image_url' ); ?>" name="<?php echo $this->get_field_name( 'image_url' ); ?>" type="text" value="<?php echo esc_attr( $image_url ); ?>" class="img" />
          <input type="button" class="select-img button button-primary" value="<?php _e('Upload', 'wen-business-pro' ); ?>" data-uploader_title="<?php _e( 'Select Image', 'wen-business-pro' ); ?>" data-uploader_button_text="<?php _e( 'Choose Image', 'wen-business-pro' ); ?>" />
          <?php
            $full_image_url = '';
            if (! empty( $image_url ) ){
              $full_image_url = $image_url;
            }
            $wrap_style = '';
            if ( empty( $full_image_url ) ) {
              $wrap_style = ' style="display:none;" ';
            }
          ?>
          <div class="ad-preview-wrap" <?php echo $wrap_style; ?>>
            <img src="<?php echo esc_url( $full_image_url ); ?>" alt="<?php _e( 'Preview', 'wen-business-pro' ); ?>" style="max-width: 100%;"  />
          </div><!-- .ad-preview-wrap -->

        </div>
        <p>
          <label for="<?php echo $this->get_field_id( 'link_url' ); ?>"><?php _e( 'Link URL:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'link_url' ); ?>" name="<?php echo $this->get_field_name( 'link_url' ); ?>" type="text" value="<?php echo esc_attr( $link_url ); ?>" />
        </p>
        <p><input id="<?php echo $this->get_field_id( 'open_in_new_window' ); ?>" name="<?php echo $this->get_field_name( 'open_in_new_window' ); ?>" type="checkbox" <?php checked( isset( $instance['open_in_new_window'] ) ? $instance['open_in_new_window'] : 0); ?> />&nbsp;<label for="<?php echo $this->get_field_id( 'open_in_new_window' ); ?>"><?php _e( 'Open in new window', 'wen-business-pro' ); ?></label>
        </p>
        <p>
          <label for="<?php echo $this->get_field_id( 'alt_text' ); ?>"><?php _e( 'Alt Text:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'alt_text' ); ?>" name="<?php echo $this->get_field_name( 'alt_text' ); ?>" type="text" value="<?php echo esc_attr( $alt_text ); ?>" />
        </p>
        <hr />
        <p>
          <label for="<?php echo $this->get_field_id( 'custom_class' ); ?>"><?php _e( 'Custom Class:', 'wen-business-pro' ); ?></label>
          <input class="widefat" id="<?php echo $this->get_field_id( 'custom_class' ); ?>" name="<?php echo $this->get_field_name( 'custom_class' ); ?>" type="text" value="<?php echo esc_attr( $custom_class ); ?>" />
        </p>
        <p>
          <em><strong><?php _e( 'Note:', 'wen-business-pro' ); ?></strong>&nbsp;<?php _e( 'Option 2 will be ignored if Ad Code is available.', 'wen-business-pro' ); ?></em>
        </p>

        <?php
      }

  }

endif;
