<?php
/**
 * Gallery Options
 *
 * @package WEN_Travel
 */

/**
 * Add gallery options to theme options
 *
 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
 */
function wen_travel_gallery_options( $wp_customize ) {
	$wp_customize->add_section( 'wen_travel_gallery_options', array(
			'title' => esc_html__( 'Gallery', 'wen-travel-pro' ),
			'panel' => 'wen_travel_theme_options',
		)
	);

	wen_travel_register_option( $wp_customize, array(
            'name'              => 'wen_travel_gallery_note',
            'sanitize_callback' => 'sanitize_text_field',
            'custom_control'    => 'Wen_Travel_Note_Control',
            'active_callback'   => 'wen_travel_is_catch_gallery_inactive',
            'label'             => sprintf( esc_html__( 'For Gallery, install %1$sCatch Gallery%2$s Plugin', 'wen-travel-pro' ),
                '<a target="_blank" href="https://wordpress.org/plugins/catch-gallery/">',
                '</a>'
            ),
            'section'           => 'wen_travel_gallery_options',
            'type'              => 'description',
            'priority'          => 1,
        )
    );


	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_visibility',
			'default'           => 'disabled',
			'sanitize_callback' => 'wen_travel_sanitize_select',
			'active_callback'	=> 'wen_travel_is_catch_gallery_active',
			'choices'           => wen_travel_section_visibility_options(),
			'label'             => esc_html__( 'Enable on', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'type'              => 'select',
		)
	);

	$types = wen_travel_section_type_options();

	// Unset image as gallery does not support it.
	unset( $types['custom'] );

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_type',
			'default'           => 'page',
			'sanitize_callback' => 'wen_travel_sanitize_select',
			'active_callback'   => 'wen_travel_is_gallery_active',
			'choices'           => $types,
			'label'             => esc_html__( 'Type', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'type'              => 'select',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery',
			'default'           => '0',
			'sanitize_callback' => 'wen_travel_sanitize_post',
			'active_callback'   => 'wen_travel_is_gallery_page_content_active',
			'label'             => esc_html__( 'Page', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'type'              => 'dropdown-pages',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_post',
			'default'           => 0,
			'sanitize_callback' => 'wen_travel_sanitize_post',
			'active_callback'   => 'wen_travel_is_gallery_post_content_active',
			'label'             => esc_html__( 'Post', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'choices'           => wen_travel_generate_post_array(),
			'type'              => 'select',
		)
	);

	// create an empty array.
	$cats = array();

	$cats['0'] = esc_html__( '-- Select --', 'wen-travel-pro' );

	// we loop over the categories and set the names and
	// labels we need.
	foreach ( get_categories() as $categories => $category ) {
		$cats[ $category->term_id ] = $category->name;
	}

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_category',
			'default'           => '0',
			'sanitize_callback' => 'absint',
			'active_callback'   => 'wen_travel_is_gallery_category_content_active',
			'label'             => esc_html__( 'Category', 'wen-travel-pro' ),
			'type'              => 'select',
			'choices'           => $cats,
			'section'           => 'wen_travel_gallery_options',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_description',
			'sanitize_callback' => 'wp_kses_post',
			'active_callback'   => 'wen_travel_is_gallery_post_page_category_content_active',
			'label'             => esc_html__( 'Description', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'type'              => 'textarea',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_display_gallery_title',
			'sanitize_callback' => 'wen_travel_sanitize_checkbox',
			'active_callback'   => 'wen_travel_is_gallery_post_page_category_content_active',
			'label'             => esc_html__( 'Display title', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'custom_control'    => 'Wen_Travel_Toggle_Control',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_text',
			'sanitize_callback' => 'sanitize_text_field',
			'active_callback'   => 'wen_travel_is_gallery_active',
			'label'             => esc_html__( 'Button Text', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'type'              => 'text',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_link',
			'sanitize_callback' => 'esc_url_raw',
			'active_callback'   => 'wen_travel_is_gallery_active',
			'label'             => esc_html__( 'Button Link', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
		)
	);

	wen_travel_register_option( $wp_customize, array(
			'name'              => 'wen_travel_gallery_target',
			'sanitize_callback' => 'wen_travel_sanitize_checkbox',
			'active_callback'   => 'wen_travel_is_gallery_active',
			'label'             => esc_html__( 'Open Link in New Window/Tab', 'wen-travel-pro' ),
			'section'           => 'wen_travel_gallery_options',
			'custom_control'    => 'Wen_Travel_Toggle_Control',
		)
	);
}
add_action( 'customize_register', 'wen_travel_gallery_options' );

/** Active Callback Functions **/
if ( ! function_exists( 'wen_travel_is_gallery_active' ) ) :
	/**
	* Return true if gallery content is active
	*
	* @since Wen Travel Pro 1.0
	*/
	function wen_travel_is_gallery_active( $control ) {
		$enable = $control->manager->get_setting( 'wen_travel_gallery_visibility' )->value();

		return wen_travel_check_section( $enable );
	}
endif;

if ( ! function_exists( 'wen_travel_is_gallery_post_content_active' ) ) :
	/**
	* Return true if post content is active
	*
	* @since Wen Travel Pro 1.0
	*/
	function wen_travel_is_gallery_post_content_active( $control ) {
		$type = $control->manager->get_setting( 'wen_travel_gallery_type' )->value();

		return ( wen_travel_is_gallery_active( $control ) && 'post' === $type );
	}
endif;

if ( ! function_exists( 'wen_travel_is_gallery_page_content_active' ) ) :
	/**
	* Return true if gallery page content is active
	*
	* @since Wen Travel Pro 1.0
	*/
	function wen_travel_is_gallery_page_content_active( $control ) {
		$type = $control->manager->get_setting( 'wen_travel_gallery_type' )->value();

		return ( wen_travel_is_gallery_active( $control ) && 'page' === $type );
	}
endif;

if ( ! function_exists( 'wen_travel_is_gallery_category_content_active' ) ) :
	/**
	* Return true if gallery category content is active
	*
	* @since Wen Travel Pro 1.0
	*/
	function wen_travel_is_gallery_category_content_active( $control ) {
		$type = $control->manager->get_setting( 'wen_travel_gallery_type' )->value();

		return ( wen_travel_is_gallery_active( $control ) && 'category' === $type );
	}
endif;

if ( ! function_exists( 'wen_travel_is_gallery_post_page_category_content_active' ) ) :
	/**
	* Return true if gallery post/page/category content is active
	*
	* @since Wen Travel Pro 1.0
	*/
	function wen_travel_is_gallery_post_page_category_content_active( $control ) {
		$type = $control->manager->get_setting( 'wen_travel_gallery_type' )->value();

		return ( wen_travel_is_gallery_active( $control ) && ( 'page' === $type || 'post' === $type || 'category' === $type ) );
	}
endif;

if ( ! function_exists( 'wen_travel_is_catch_gallery_inactive' ) ) :
    /**
    *
    * @since Wen Travel Pro 1.2.2
    */
    function wen_travel_is_catch_gallery_inactive( $control ) {
        return ! ( defined( 'CATCH_GALLERY_VERSION' ) || defined( 'CATCH_GALLERY_PRO_VERSION' ) );
    }
endif;

if ( ! function_exists( 'wen_travel_is_catch_gallery_active' ) ) :
    /**
    *
    * @since Wen Travel Pro 1.2.2
    */
    function wen_travel_is_catch_gallery_active( $control ) {
        return  ( defined( 'CATCH_GALLERY_VERSION' ) || defined( 'CATCH_GALLERY_PRO_VERSION' ) );
    }
endif;
