<?php
/**
 * The template for displaying meta box in page/post
 *
 * This adds Select Sidebar, Header Featured Image Options, Single Page/Post Image
 * This is only for the design purpose and not used to save any content
 *
 * @package WEN_Travel
 */



/**
 * Class to Renders and save metabox options
 *
 * @since Wen Travel Pro 1.0
 */
class Wen_Travel_Metabox {
	private $meta_box;

	private $fields;

	/**
	* Constructor
	*
	* @since Wen Travel Pro 1.0
	*
	* @access public
	*
	*/
	public function __construct( $meta_box_id, $meta_box_title, $post_type ) {

		$this->meta_box = array (
							'id'        => $meta_box_id,
							'title'     => $meta_box_title,
							'post_type' => $post_type,
							);

		$this->fields = array(
			'wen-travel-header-image',
			'wen-travel-sidebar-option',
			'wen-travel-featured-image',
		);


		// Add metaboxes
		add_action( 'add_meta_boxes', array( $this, 'add' ) );

		add_action( 'save_post', array( $this, 'save' ) );
	}

	/**
	* Add Meta Box for multiple post types.
	*
	* @since Wen Travel Pro 1.0
	*
	* @access public
	*/
	public function add( $post_type ) {
		add_meta_box( $this->meta_box['id'], $this->meta_box['title'], array( $this, 'show' ), $post_type, 'side', 'high' );
	}

	/**
	* Renders metabox
	*
	* @since Wen Travel Pro 1.0
	*
	* @access public
	*/
	public function show() {
		global $post;

		$sidebar_options = array(
			'default-sidebar'        => esc_html__( 'Default Sidebar', 'wen-travel-pro' ),
			'optional-sidebar-one'   => esc_html__( 'Optional Sidebar One', 'wen-travel-pro' ),
			'optional-sidebar-two'   => esc_html__( 'Optional Sidebar Two', 'wen-travel-pro' ),
			'optional-sidebar-three' => esc_html__( 'Optional Sidebar three', 'wen-travel-pro' ),
		);

		$header_image_options = array(
			'default' => esc_html__( 'Default', 'wen-travel-pro' ),
			'enable'  => esc_html__( 'Enable', 'wen-travel-pro' ),
			'disable' => esc_html__( 'Disable', 'wen-travel-pro' ),
		);

		$featured_image_options = array(
			'default'         => esc_html__( 'Default', 'wen-travel-pro' ),
			'disabled'        => esc_html__( 'Disabled', 'wen-travel-pro' ),
			'post-thumbnail'  => esc_html__( 'Post Thumbnail (470x470)', 'wen-travel-pro' ),
			'wen-travel-slider' => esc_html__( 'Slider Image (1920x954)', 'wen-travel-pro' ),
			'full'            => esc_html__( 'Original Image Size', 'wen-travel-pro' ),
		);


		// Use nonce for verification
		wp_nonce_field( basename( __FILE__ ), 'wen_travel_custom_meta_box_nonce' );

		// Begin the field table and loop  ?>
		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-sidebar-option"><?php esc_html_e( 'Select Sidebar', 'wen-travel-pro' ); ?></label></p>
		<select class="widefat" name="wen-travel-sidebar-option" id="wen-travel-sidebar-option">
			 <?php
				$meta_value = get_post_meta( $post->ID, 'wen-travel-sidebar-option', true );
				
				if ( empty( $meta_value ) ){
					$meta_value='default';
				}
				
				foreach ( $sidebar_options as $field =>$label ) {	
				?>
					<option value="<?php echo esc_attr( $field ); ?>" <?php selected( $meta_value, $field ); ?>><?php echo esc_html( $label ); ?></option>
				<?php
				} // end foreach
			?>
		</select>

		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-header-image"><?php esc_html_e( 'Header Featured Image Options', 'wen-travel-pro' ); ?></label></p>
		<select class="widefat" name="wen-travel-header-image" id="wen-travel-header-image">
			 <?php
				$meta_value = get_post_meta( $post->ID, 'wen-travel-header-image', true );
				
				if ( empty( $meta_value ) ){
					$meta_value='default';
				}
				
				foreach ( $header_image_options as $field =>$label ) {	
				?>
					<option value="<?php echo esc_attr( $field ); ?>" <?php selected( $meta_value, $field ); ?>><?php echo esc_html( $label ); ?></option>
				<?php
				} // end foreach
			?>
		</select>

		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-featured-image"><?php esc_html_e( 'Single Page/Post Image', 'wen-travel-pro' ); ?></label></p>
		<select class="widefat" name="wen-travel-featured-image" id="wen-travel-featured-image">
			 <?php
				$meta_value = get_post_meta( $post->ID, 'wen-travel-featured-image', true );
				
				if ( empty( $meta_value ) ){
					$meta_value='default';
				}
				
				foreach ( $featured_image_options as $field =>$label ) {	
				?>
					<option value="<?php echo esc_attr( $field ); ?>" <?php selected( $meta_value, $field ); ?>><?php echo esc_html( $label ); ?></option>
				<?php
				} // end foreach
			?>
		</select>

		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-event-date-day"><?php esc_html_e( 'Events Date Day', 'wen-travel-pro' ); ?></label></p>
		<?php
			$value = get_post_meta( $post->ID, 'wen-travel-event-date-day', true );

			if ( ! $value ) {
				$value = 1;
			}
		?>
		<input class="widefat ct-event-date" type="number" min="1" max="31" name="wen-travel-event-date-day" value="<?php echo esc_attr( $value ); ?>"/>

		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-event-date-month"><?php esc_html_e( 'Events Date Month', 'wen-travel-pro' ); ?></label></p>
		<?php
			$month_value = get_post_meta( $post->ID, 'wen-travel-event-date-month', true );

			if ( ! $month_value ) {
				$month_value = 1;
			} 
		?>
		<input class="widefat ct-event-time" type="number" min="1" max="12" name="wen-travel-event-date-month" value="<?php echo esc_attr( $month_value ); ?>"/>

		<p class="post-attributes-label-wrapper"><label class="post-attributes-label" for="wen-travel-event-date-month"><?php esc_html_e( 'Events Date Year', 'wen-travel-pro' ); ?></label></p>
		<?php
			$year_value = get_post_meta( $post->ID, 'wen-travel-event-date-year', true );

			if ( ! $year_value ) {
				$year_value = 2019;
			}
		?>
		<input class="widefat ct-event-time" type="number" min="1000" max="9999" name="wen-travel-event-date-year" value="<?php echo esc_attr( $year_value ); ?>"/>
	<?php
	}

	/**
	 * Save custom metabox data
	 *
	 * @action save_post
	 *
	 * @since Wen Travel Pro 1.0
	 *
	 * @access public
	 */
	public function save( $post_id ) {
		global $post_type;

		$post_type_object = get_post_type_object( $post_type );

		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )                      // Check Autosave
		|| ( ! isset( $_POST['post_ID'] ) || $post_id != $_POST['post_ID'] )        // Check Revision
		|| ( ! in_array( $post_type, $this->meta_box['post_type'] ) )                  // Check if current post type is supported.
		|| ( ! check_admin_referer( basename( __FILE__ ), 'wen_travel_custom_meta_box_nonce') )    // Check nonce - Security
		|| ( ! current_user_can( $post_type_object->cap->edit_post, $post_id ) ) )  // Check permission
		{
		  return $post_id;
		}

		foreach ( $this->fields as $field ) {
			$new = $_POST[ $field ];

			delete_post_meta( $post_id, $field );

			if ( '' == $new || array() == $new ) {
				return;
			} else {
				if ( ! update_post_meta ( $post_id, $field, sanitize_key( $new ) ) ) {
					add_post_meta( $post_id, $field, sanitize_key( $new ), true );
				}
			}
		} // end foreach

		//Validation for event extra options
		$date_day = $_POST['wen-travel-event-date-day'];
		if ( '' != $date_day ) {
			if ( ! update_post_meta( $post_id, 'wen-travel-event-date-day', absint( $date_day ) ) ) {
				add_post_meta( $post_id, 'wen-travel-event-date-day', absint( $date_day ), true );
			}
		}

		$date_month = $_POST['wen-travel-event-date-month'];
		if ( '' != $date_month ) {
			if ( ! update_post_meta( $post_id, 'wen-travel-event-date-month', absint( $date_month ) ) ) {
				add_post_meta( $post_id, 'wen-travel-event-date-month', absint( $date_month ), true );
			}
		}

		$date_year = $_POST['wen-travel-event-date-year'];
		if ( '' != $date_year ) {
			if ( ! update_post_meta( $post_id, 'wen-travel-event-date-year', absint( $date_year ) ) ) {
				add_post_meta( $post_id, 'wen-travel-event-date-year', absint( $date_year ), true );
			}
		}
	}
}

$wen_travel_metabox = new Wen_Travel_Metabox(
	'wen-travel-options',                  //metabox id
	esc_html__( 'Wen Travel Pro Options', 'wen-travel-pro' ), //metabox title
	array( 'page', 'post' )             //metabox post types
);
