<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package WEN_Travel
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @since Wen Travel Pro 1.0
 *
 * @param array $classes Classes for the body element.
 * @return array (Maybe) filtered body classes.
 */
function wen_travel_body_classes( $classes ) {
	// Adds a class of custom-background-image to sites with a custom background image.
	if ( get_background_image() ) {
		$classes[] = 'custom-background-image';
	}

	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Always add a front-page class to the front page.
	if ( is_front_page() && ! is_home() ) {
		$classes[] = 'page-template-front-page';
	}

	// Adds a class of (full-width|box) to blogs.
	if ( 'boxed' === get_theme_mod( 'wen_travel_layout_type' ) ) {
		$classes[] = 'boxed-layout';
	} else {
		$classes[] = 'fluid-layout';
	}
	$menu = get_theme_mod( 'wen_travel_menu_type' );
	// Adds a class of navigation-(default|classic) to blogs.
	if ( 'classic' === get_theme_mod( 'wen_travel_menu_type', 'classic' ) ) {
		$classes[] = 'navigation-classic';
	} else {
		$classes[] = 'navigation-default';
	}

	// Adds a class with respect to layout selected.
	$layout  = wen_travel_get_theme_layout();
	$sidebar = wen_travel_get_sidebar_id();

	$layout_class = "no-sidebar content-width-layout";

	if ( 'no-sidebar-full-width' === $layout ) {
		$layout_class = 'no-sidebar full-width-layout';
	} elseif ( 'left-sidebar' === $layout ) {
		if ( '' !== $sidebar ) {
			$layout_class = 'two-columns-layout content-right';
		}
	} elseif ( 'right-sidebar' === $layout ) {
		if ( '' !== $sidebar ) {
			$layout_class = 'two-columns-layout content-left';
		}
	}

	$classes[] = $layout_class;

	$classes[] = esc_attr( get_theme_mod( 'wen_travel_content_layout', 'excerpt' ) );

	$classes[] = 'header-media-fluid';

	$enable_sticky_playlist = get_theme_mod( 'wen_travel_sticky_playlist_visibility', 'disabled' );

	if ( wen_travel_check_section( $enable_sticky_playlist ) ) {
		$classes[] = 'sticky-playlist-enabled';
	}

	$enable_slider = wen_travel_check_section( get_theme_mod( 'wen_travel_slider_option', 'disabled' ) );

	$header_image = wen_travel_featured_overall_image();

	if ( 'disable' !== $header_image ) {
		$classes[] = 'has-header-media';
	}

	if ( ! wen_travel_has_header_media_text() ) {
		$classes[] = 'header-media-text-disabled';
	}

	// Add a class if there is a custom header.
	if ( has_header_image() ) {
		$classes[] = 'has-header-image';
	}

	$enable_slider = wen_travel_check_section( get_theme_mod( 'wen_travel_slider_option', 'disabled' ) );

	$transparent_header = get_theme_mod( 'wen_travel_transparent_header', 1 );
	if ( $transparent_header && ( wen_travel_has_header_media_text() || 'disable' !== $header_image || $enable_slider ) ) {
		$classes[] = 'transparent-header';
	}
	

	// Add class when there is header top bar.
	if ( get_theme_mod( 'wen_travel_disable_header_top' ) ) {
		$classes[] = 'has-header-top-bar';
	}

	// Added color scheme to body class.
	$classes['color-scheme'] = 'color-scheme-' . esc_attr( get_theme_mod( 'color_scheme', 'default' ) );
	$classes['header-style'] = 'header-' . esc_attr( get_theme_mod( 'wen_travel_header_style', 'style-one' ) );

	if ( get_theme_mod( 'wen_travel_gradient_button' ) ) {
		$classes[] = 'gradient-button-enabled';
	}

	if ( class_exists( 'WP_Travel' ) ) {
		$classes[] = 'wp-travel-activated';
	}

	return $classes;
}
add_filter( 'body_class', 'wen_travel_body_classes' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function wen_travel_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'wen_travel_pingback_header' );

if ( ! function_exists( 'wen_travel_comments' ) ) :
	/**
	 * Enable/Disable Comments
	 *
	 * @uses comment_form_default_fields filter
	 * @since Wen Travel Pro 1.0
	 */
	function wen_travel_comments( $open, $post_id ) {
		$comment_select = get_theme_mod( 'wen_travel_comment_option', 'use-wordpress-setting' );

	    if ( 'disable-completely' === $comment_select ) {
			return false;
		} elseif ( 'disable-in-pages' === $comment_select && is_page() ) {
			return false;
		}

	    return $open;
	}
endif; // wen_travel_comments.
add_filter( 'comments_open', 'wen_travel_comments', 10, 2 );

if ( ! function_exists( 'wen_travel_comment_form_fields' ) ) :
	/**
	 * Modify Comment Form Fields
	 *
	 * @uses comment_form_default_fields filter
	 * @since Wen Travel Pro 1.0
	 */
	function wen_travel_comment_form_fields( $fields ) {
	    $disable_website = get_theme_mod( 'wen_travel_website_field', 1  );

	    if ( isset( $fields['url'] ) && ! $disable_website ) {
			unset( $fields['url'] );
		}

		return $fields;
	}
endif; // wen_travel_comment_form_fields.
add_filter( 'comment_form_default_fields', 'wen_travel_comment_form_fields' );

/**
 * Adds font family custom CSS
 */
function wen_travel_get_font_family_css() {
	$font_family_options = wen_travel_font_family_options();

	$fonts = wen_travel_avaliable_fonts();

	$css = array();

	foreach ( $font_family_options as $key => $value ) {
		$option = get_theme_mod( $key, $value['default'] );

		if ( $option !== $value['default'] ) {
			$css[] = $value['selector'] . ' { font-family: ' . $fonts [ $option ]['label'] . '; }';
		}
	}

	$css = implode( PHP_EOL, $css );

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_get_font_family_css', 11 );

/**
 * Adds portfolio background CSS
 */
function wen_travel_portfolio_bg_css() {
	$enable   = get_theme_mod( 'wen_travel_portfolio_option', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$css = '';

	$background = get_theme_mod( 'wen_travel_portfolio_bg_image' );
	
	if ( $background ) {
		$css = '#portfolio-content-section { background-image: url("' . esc_url( $background ) . '"); }';
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_portfolio_bg_css', 11 );

/**
 * Adds portfolio background CSS
 */
function wen_travel_portfolio_two_bg_css() {
	$enable   = get_theme_mod( 'wen_travel_portfolio_two_option', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$css = '';

	$background = get_theme_mod( 'wen_travel_portfolio_two_bg_image' );
	
	if ( $background ) {
		$css = '#portfolio-two-content-section { background-image: url("' . esc_url( $background ) . '"); }';
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_portfolio_two_bg_css', 11 );

/**
 * Adds Attraction background CSS
 */
function wen_travel_attraction_bg_css() {
	$enable   = get_theme_mod( 'wen_travel_attraction_option', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$css = '';

	$background = get_theme_mod( 'wen_travel_attraction_bg_image' );
	
	if ( $background ) {
		$css = '#attraction-content-section { background-image: url("' . esc_url( $background ) . '"); }';
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_attraction_bg_css', 11 );

/**
 * Adds stats background CSS
 */
function wen_travel_stats_bg_css() {
	$enable   = get_theme_mod( 'wen_travel_stats_option', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$css = '';

	$background = get_theme_mod( 'wen_travel_stats_bg_image' );
	
	if ( $background ) {
		$css = '#stats-section .hentry-inner { background-image: url("' . esc_url( $background ) . '"); }';
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_stats_bg_css', 11 );

/**
 * Adds custom overlay for Promotion Headline Background Image
 */
function wen_travel_promo_head_bg_image_overlay_css() {
	$enable   = get_theme_mod( 'wen_travel_promo_head_visibility', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$overlay = get_theme_mod( 'wen_travel_promo_head_background_image_opacity' );

	$css = '';

	$overlay_bg = $overlay / 100;

	if ( $overlay ) {
		$css = '.promotion-section .cover-link:before {
					background-color: rgba(0, 0, 0, ' . esc_attr( $overlay_bg ) . ' );
			    } '; // Dividing by 100 as the option is shown as % for user
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_promo_head_bg_image_overlay_css', 11 );

/**
 * Adds custom overlay for Promotion Headline Two Background Image
 */
function wen_travel_promo_head_two_bg_image_overlay_css() {
	$enable   = get_theme_mod( 'wen_travel_promo_head_two_visibility', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$overlay = get_theme_mod( 'wen_travel_promo_head_two_background_image_opacity' );

	$css = '';

	$overlay_bg = $overlay / 100;

	if ( $overlay ) {
		$css = '#promotion-section-two .cover-link:before {
					background-color: rgba(0, 0, 0, ' . esc_attr( $overlay_bg ) . ' );
			    } '; // Dividing by 100 as the option is shown as % for user
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_promo_head_two_bg_image_overlay_css', 11 );

/**
 * Adds custom overlay for Promotion Headline Background Image
 */
function wen_travel_attraction_bg_image_overlay_css() {
	$enable   = get_theme_mod( 'wen_travel_attraction_option', 'disabled' );
	$bg_image = get_theme_mod( 'wen_travel_attraction_bg_image' );

	if ( ! wen_travel_check_section( $enable ) || ! $bg_image ) {
		// Bail if contact section is disabled.
		return;
	}

	$overlay = get_theme_mod( 'wen_travel_attraction_background_image_opacity', '60' );

	$css = '';

	$overlay_bg = $overlay / 100;

	if ( $overlay ) {
		$css = '.attraction-content-section .wrapper::before {
					background-color: rgba(0, 0, 0, ' . esc_attr( $overlay_bg ) . ' );
			    } '; // Dividing by 100 as the option is shown as % for user
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_attraction_bg_image_overlay_css', 11 );

/**
 * Adds custom overlay for Promotion Headline Background Image
 */
function wen_travel_special_offer_text_bg_image_css() {
	$enable   = get_theme_mod( 'wen_travel_special_offer_visibility', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$background = get_theme_mod( 'wen_travel_special_offer_text_bg_image' );

	$css = '';

	if ( $background ) {
		$css = '#special-offer-section .section-description { background-image: url("' . esc_url( $background ) . '"); }';
	}
	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_special_offer_text_bg_image_css', 11 );

/**
 * Adds custom overlay for Contact Background Image
 */
function wen_travel_contact_bg_css() {
	$enable   = get_theme_mod( 'wen_travel_contact_option', 'disabled' );

	if ( ! wen_travel_check_section( $enable ) ) {
		// Bail if contact section is disabled.
		return;
	}

	$background = '';

	$type  = get_theme_mod( 'wen_travel_contact_type', 'page' );

	if ( 'post' === $type && 'publish' === get_post_status( get_theme_mod( 'wen_travel_contact_post' ) ) ) {
		$background = get_the_post_thumbnail_url( get_theme_mod( 'wen_travel_contact_post' ) );
	} elseif ( 'page' === $type  && 'publish' === get_post_status( get_theme_mod( 'wen_travel_contact_page' ) ) ) {
		$background = get_the_post_thumbnail_url( get_theme_mod( 'wen_travel_contact_page' ) );
	} elseif ( 'custom' === $type ) {
		$background = get_theme_mod( 'wen_travel_contact_bg_image' ); 
	}

	$css = '';

	if ( $background ) {
		$css = '#contact-section { background-image: url("' . esc_url( $background ) . '"); }';
	}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_contact_bg_css', 11 );

/**
 * Adds custom overlay for Header Media
 */
function wen_travel_header_media_image_overlay_css() {
	$overlay = get_theme_mod( 'wen_travel_header_media_image_opacity' );

	$css = '';

	$overlay_bg = $overlay / 100;

	if ( $overlay ) {
	$css = '.custom-header-overlay {
		background-color: rgba(0, 0, 0, ' . esc_attr( $overlay_bg ) . ' );
    } '; // Dividing by 100 as the option is shown as % for user
}

	wp_add_inline_style( 'wen-travel-style', $css );
}
add_action( 'wp_enqueue_scripts', 'wen_travel_header_media_image_overlay_css', 11 );

/**
 * Remove first post from blog as it is already show via recent post template
 */
function wen_travel_alter_home( $query ) {
	if ( $query->is_home() && $query->is_main_query() ) {
		$cats = get_theme_mod( 'wen_travel_front_page_category' );

		if ( is_array( $cats ) && ! in_array( '0', $cats ) ) {
			$query->query_vars['category__in'] = $cats;
		}

		if ( get_theme_mod( 'wen_travel_exclude_slider_post' ) ) {
			$quantity = get_theme_mod( 'wen_travel_slider_number', 4 );

			$post_list	= array();	// list of valid post ids

			for( $i = 1; $i <= $quantity; $i++ ){
				if ( get_theme_mod( 'wen_travel_slider_post_' . $i ) && get_theme_mod( 'wen_travel_slider_post_' . $i ) > 0 ) {
					$post_list = array_merge( $post_list, array( get_theme_mod( 'wen_travel_slider_post_' . $i ) ) );
				}
			}

			if ( ! empty( $post_list ) ) {
	    		$query->query_vars['post__not_in'] = $post_list;
			}
		}
	}
}
add_action( 'pre_get_posts', 'wen_travel_alter_home' );

/**
 * Function to add Scroll Up icon
 */
function wen_travel_scrollup() {
	$disable_scrollup =  ! get_theme_mod( 'wen_travel_disable_scrollup', 1 );

	if ( $disable_scrollup ) {
		return;
	}

	echo '<a href="#masthead" id="scrollup" class="backtotop">' . wen_travel_get_svg( array( 'icon' => 'scrollup' ) ) . '<span class="screen-reader-text">' . esc_html__( 'Scroll Up', 'wen-travel-pro' ) . '</span></a>' ;
} 
add_action( 'wp_footer', 'wen_travel_scrollup', 1 );

if ( ! function_exists( 'wen_travel_content_nav' ) ) :
	/**
	 * Display navigation/pagination when applicable
	 *
	 * @since Wen Travel Pro 1.0
	 */
	function wen_travel_content_nav() {
		global $wp_query;

		// Don't print empty markup in archives if there's only one page.
		if ( $wp_query->max_num_pages < 2 && ( is_home() || is_archive() || is_search() ) ) {
			return;
		}

		$pagination_type = get_theme_mod( 'wen_travel_pagination_type', 'default' );

		if ( ( class_exists( 'Jetpack' ) && Jetpack::is_module_active( 'infinite-scroll' ) ) ) {
			// Support infinite scroll plugins.
			the_posts_navigation();
		} elseif ( 'numeric' === $pagination_type && function_exists( 'the_posts_pagination' ) ) {
			the_posts_pagination( array(
				'prev_text'          => '<span>' . esc_html__( 'Prev', 'wen-travel-pro' ) . '</span>',
				'next_text'          => '<span>' . esc_html__( 'Next', 'wen-travel-pro' ) . '</span>',
				'screen_reader_text' => '<span class="nav-subtitle screen-reader-text">' . esc_html__( 'Page', 'wen-travel-pro' ) . ' </span>',
			) );
		} else {
			the_posts_navigation();
		}
	}
endif; // wen_travel_content_nav

/**
 * Check if a section is enabled or not based on the $value parameter
 * @param  string $value Value of the section that is to be checked
 * @return boolean return true if section is enabled otherwise false
 */
function wen_travel_check_section( $value ) {
	return ( 'entire-site' == $value  || ( is_front_page() && 'homepage' === $value ) );
}

/**
 * Return the first image in a post. Works inside a loop.
 * @param [integer] $post_id [Post or page id]
 * @param [string/array] $size Image size. Either a string keyword (thumbnail, medium, large or full) or a 2-item array representing width and height in pixels, e.g. array(32,32).
 * @param [string/array] $attr Query string or array of attributes.
 * @return [string] image html
 *
 * @since Wen Travel Pro 1.0
 */
function wen_travel_get_first_image( $postID, $size, $attr, $src = false ) {
	ob_start();

	ob_end_clean();

	$image 	= '';

	$output = preg_match_all('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', get_post_field('post_content', $postID ) , $matches);

	if ( isset( $matches[1][0] ) ) {
		// Get first image.
		$first_img = $matches[1][0];

		if ( $src ) {
			//Return url of src is true
			return $first_img;
		}

		return '<img class="wp-post-image" src="'. esc_url( $first_img ) .'">';
	}

	return false;
}

function wen_travel_get_theme_layout() {
	$layout = '';

	if ( is_page_template( 'templates/no-sidebar.php' ) ) {
		$layout = 'no-sidebar';
	} elseif ( is_page_template( 'templates/full-width-page.php' ) ) {
		$layout = 'no-sidebar-full-width';
	} elseif ( is_page_template( 'templates/left-sidebar.php' ) ) {
		$layout = 'left-sidebar';
	} elseif ( is_page_template( 'templates/right-sidebar.php' ) ) {
		$layout = 'right-sidebar';
	} else {
		$layout = get_theme_mod( 'wen_travel_default_layout', 'right-sidebar' );

		if ( is_home() || is_archive() ) {
			$layout = get_theme_mod( 'wen_travel_homepage_archive_layout', 'no-sidebar-full-width' );
		}

		if ( class_exists( 'WooCommerce' ) && ( is_shop() || is_woocommerce() || is_cart() || is_checkout() ) ) {
			$layout = get_theme_mod( 'wen_travel_woocommerce_layout', 'right-sidebar' );
		}
	}

	return $layout;
}

function wen_travel_get_sidebar_id() {
	$sidebar = $id = '';

	$layout = wen_travel_get_theme_layout();

	if ( 'no-sidebar-full-width' === $layout || 'no-sidebar' === $layout ) {
		return $sidebar;
	}

	// WooCommerce Shop Page excluding Cart and checkout.
	if ( class_exists( 'WooCommerce' ) && is_woocommerce() ) {
		$id = get_option( 'woocommerce_shop_page_id' );
	} else {
		// Blog Page or Front Page setting in Reading Settings.
		if ( 'page' == get_option('show_on_front') ) {
	        $id = get_option('show_on_front');
	    } elseif ( is_singular() ) {
	    	global $post;

			$id = $post->ID;
	    	
	    	if ( is_attachment() ) {
				$id = $post->post_parent;
 			}
		}
	}

	$sidebaroptions = get_post_meta( $id, 'wen-travel-sidebar-option', true );

	if ( is_active_sidebar( 'sidebar-woo' ) && class_exists( 'WooCommerce' ) && ( is_woocommerce() || is_cart() || is_checkout() ) ) {
		$sidebar = 'sidebar-woo'; // WooCommerce Sidebar.
	} elseif ( is_active_sidebar( 'sidebar-optional-one' ) && 'optional-sidebar-one' === $sidebaroptions ) {
		$sidebar = 'sidebar-optional-one';
	} elseif ( is_active_sidebar( 'sidebar-optional-two' ) && 'optional-sidebar-two' === $sidebaroptions ) {
		$sidebar = 'sidebar-optional-two';
	} elseif ( is_active_sidebar( 'sidebar-optional-three' ) && 'optional-sidebar-three' === $sidebaroptions ) {
		$sidebar = 'sidebar-optional-three';
	} elseif ( is_active_sidebar( 'sidebar-optional-homepage' ) && ( is_front_page() || ( is_home() && $page_id != $page_for_posts ) ) ) {
		$sidebar = 'sidebar-optional-homepage';
	} elseif ( is_active_sidebar( 'sidebar-optional-archive' ) && ( is_archive() || ( is_home() && $page_id != $page_for_posts ) ) ) {
		$sidebar = 'sidebar-optional-archive';
	} elseif ( is_page() && is_active_sidebar( 'sidebar-optional-page' ) ) {
		$sidebar = 'sidebar-optional-page';
	} elseif ( is_single() && is_active_sidebar( 'sidebar-optional-post' ) ) {
		$sidebar = 'sidebar-optional-post';
	} elseif ( is_active_sidebar( 'sidebar-1' ) ) {
		$sidebar = 'sidebar-1'; // Primary Sidebar.
	}

	return $sidebar;
}

if ( ! function_exists( 'wen_travel_truncate_phrase' ) ) :
	/**
	 * Return a phrase shortened in length to a maximum number of characters.
	 *
	 * Result will be truncated at the last white space in the original string. In this function the word separator is a
	 * single space. Other white space characters (like newlines and tabs) are ignored.
	 *
	 * If the first `$max_characters` of the string does not contain a space character, an empty string will be returned.
	 *
	 * @since Wen Travel Pro 1.0
	 *
	 * @param string $text            A string to be shortened.
	 * @param integer $max_characters The maximum number of characters to return.
	 *
	 * @return string Truncated string
	 */
	function wen_travel_truncate_phrase( $text, $max_characters ) {

		$text = trim( $text );

		if ( mb_strlen( $text ) > $max_characters ) {
			//* Truncate $text to $max_characters + 1
			$text = mb_substr( $text, 0, $max_characters + 1 );

			//* Truncate to the last space in the truncated string
			$text = trim( mb_substr( $text, 0, mb_strrpos( $text, ' ' ) ) );
		}

		return $text;
	}
endif; //wen_travel_truncate_phrase

if ( ! function_exists( 'wen_travel_get_the_content_limit' ) ) :
	/**
	 * Return content stripped down and limited content.
	 *
	 * Strips out tags and shortcodes, limits the output to `$max_char` characters, and appends an ellipsis and more link to the end.
	 *
	 * @since Wen Travel Pro 1.0
	 *
	 * @param integer $max_characters The maximum number of characters to return.
	 * @param string  $more_link_text Optional. Text of the more link. Default is "(more...)".
	 * @param bool    $stripteaser    Optional. Strip teaser content before the more text. Default is false.
	 *
	 * @return string Limited content.
	 */
	function wen_travel_get_the_content_limit( $max_characters, $more_link_text = '(more...)', $stripteaser = false ) {

		$content = get_the_content( '', $stripteaser );

		// Strip tags and shortcodes so the content truncation count is done correctly.
		$content = strip_tags( strip_shortcodes( $content ), apply_filters( 'get_the_content_limit_allowedtags', '<script>,<style>' ) );

		// Remove inline styles / .
		$content = trim( preg_replace( '#<(s(cript|tyle)).*?</\1>#si', '', $content ) );

		// Truncate $content to $max_char
		$content = wen_travel_truncate_phrase( $content, $max_characters );

		// More link?
		if ( $more_link_text ) {
			$link   = apply_filters( 'get_the_content_more_link', sprintf( '<a href="%s" class="more-link">%s</a>', esc_url( get_permalink() ), $more_link_text ), $more_link_text );
			$output = sprintf( '<p>%s %s</p>', $content, $link );
		} else {
			$output = sprintf( '<p>%s</p>', $content );
			$link = '';
		}

		return apply_filters( 'wen_travel_get_the_content_limit', $output, $content, $link, $max_characters );

	}
endif; //wen_travel_get_the_content_limit

if ( ! function_exists( 'wen_travel_content_image' ) ) :
	/**
	 * Template for Featured Image in Archive Content
	 *
	 * To override this in a child theme
	 * simply fabulous-fluid your own wen_travel_content_image(), and that function will be used instead.
	 *
	 * @since Wen Travel Pro 1.0
	 */
	function wen_travel_content_image() {
		if ( has_post_thumbnail() && wen_travel_jetpack_featured_image_display() && is_singular() ) {
			global $post, $wp_query;

			// Get Page ID outside Loop.
			$page_id = $wp_query->get_queried_object_id();

			if ( $post ) {
		 		if ( is_attachment() ) {
					$parent = $post->post_parent;

					$individual_featured_image = get_post_meta( $parent, 'wen-travel-featured-image', true );
				} else {
					$individual_featured_image = get_post_meta( $page_id, 'wen-travel-featured-image', true );
				}
			}

			if ( empty( $individual_featured_image ) ) {
				$individual_featured_image = 'default';
			}

			if ( 'disable' === $individual_featured_image ) {
				return false;
			} else {
				$class = array();

				$image_size = 'post-thumbnail';

				if ( 'default' !== $individual_featured_image ) {
					$image_size = $individual_featured_image;
					$class[]    = 'from-metabox';
				} else {
					$layout = wen_travel_get_theme_layout();

					if ( 'no-sidebar-full-width' === $layout ) {
						$image_size = 'wen-travel-slider';
					}
				}

				$class[] = $individual_featured_image;
				?>
				<div class="post-thumbnail <?php echo esc_attr( implode( ' ', $class ) ); ?>">
					<a href="<?php the_permalink(); ?>">
					<?php the_post_thumbnail( $image_size ); ?>
					</a>
				</div>
		   	<?php
			}
		} // End if ().
	}
endif; // wen_travel_content_image.

if ( ! function_exists( 'wen_travel_enable_homepage_posts' ) ) :
    /**
     * Determine Homepage Content disabled or not
     * @return boolean
     */
    function wen_travel_enable_homepage_posts() {
       if ( ! ( ! get_theme_mod( 'wen_travel_display_homepage_posts', 1 ) && is_front_page() ) ) {
            return true;
        }
        return false;
    }
endif; // wen_travel_enable_homepage_posts.

if ( ! function_exists( 'wen_travel_sections' ) ) :
	/**
	 * Display Sections on header and footer with respect to the section option set in wen_travel_sections_sort
	 */
	function wen_travel_sections( $selector = 'header' ) {
		$sections = get_theme_mod( 'wen_travel_sections_sort', wen_travel_get_default_sections_value() );

		$separated_sections = explode( 'main-content,', $sections );

		$sections = $separated_sections[0];

		if ( 'footer' === $selector ) {
			$sections = $separated_sections[1];
		}

		$sections =  ! empty( $sections ) ? explode( ',', $sections ) : array();

		foreach ( $sections as $section ) {
			$all_sections = wen_travel_get_sortable_sections();

			if ( isset( $all_sections[ $section ]['template-part'] ) ) {
				get_template_part( $all_sections[ $section ]['template-part'] );
			}
		}
	}
endif;

if ( ! function_exists( 'wen_travel_get_sortable_sections' ) ) :
	/**
	 * Returns list of sortable sections
	 */
	function wen_travel_get_sortable_sections() {
		$sortable_sections = array(
				'header-media'          => array(
					'label'         => esc_html__( 'Header Media', 'wen-travel-pro' ),
					'section'       => 'header_image',
					'template-part' => 'template-parts/header/header-media',
				),
				'slider'                => array(
					'label'         => esc_html__( 'Slider', 'wen-travel-pro' ),
					'section'       => 'wen_travel_featured_slider',
					'template-part' => 'template-parts/slider/display-slider',
				),
				'hero-content'          => array(
					'label'         => esc_html__( 'Hero Content', 'wen-travel-pro' ),
					'section'       => 'wen_travel_hero_content_options',
					'template-part' => 'template-parts/hero-content/content-hero',
				),
				'service'              => array(
					'label'         => esc_html__( 'Services', 'wen-travel-pro' ),
					'section'       => 'wen_travel_service',
					'template-part' => 'template-parts/service/display-service',
				),
				'countdown'              => array(
					'label'         => esc_html__( 'Countdown', 'wen-travel-pro' ),
					'section'       => 'wen_travel_countdown',
					'template-part' => 'template-parts/countdown/display-countdown',
				),
				'special-offer'              => array(
					'label'         => esc_html__( 'Special Offer', 'wen-travel-pro' ),
					'section'       => 'wen_travel_special_offer_options',
					'template-part' => 'template-parts/special-offer/content-special',
				),	
				'portfolio'             => array(
					'label'         => esc_html__( 'Portfolio', 'wen-travel-pro' ),
					'section'       => 'wen_travel_portfolio',
					'template-part' => 'template-parts/portfolio/display-portfolio',
				),
				'portfolio-two'             => array(
					'label'         => esc_html__( 'Portfolio Two', 'wen-travel-pro' ),
					'section'       => 'wen_travel_portfolio_two',
					'template-part' => 'template-parts/portfolio-two/display-portfolio',
				),
				'collection'             => array(
					'label'         => esc_html__( 'Collection', 'wen-travel-pro' ),
					'section'       => 'wen_travel_collection',
					'template-part' => 'template-parts/collection/display-collection',
				),
				'attraction'             => array(
					'label'         => esc_html__( 'Attraction', 'wen-travel-pro' ),
					'section'       => 'wen_travel_attraction',
					'template-part' => 'template-parts/attraction/display-attraction',
				),
				'stats'                 => array(
					'label'         => esc_html__( 'Stats', 'wen-travel-pro' ),
					'section'       => 'wen_travel_stats',
					'template-part' => 'template-parts/stats/display-stats',
				),
				'testimonial'           => array(
					'label'         => esc_html__( 'Testimonial', 'wen-travel-pro' ),
					'section'       => 'wen_travel_testimonials',
					'template-part' => 'template-parts/testimonial/display-testimonial',
				),
				'promotion-headline'    => array(
					'label'         => esc_html__( 'Promotion headline', 'wen-travel-pro' ),
					'section'       => 'wen_travel_promotion_headline',
					'template-part' => 'template-parts/promotion-headline/content-promotion',
				),
				'promotion-headline-two'    => array(
					'label'         => esc_html__( 'Promotion headline Two', 'wen-travel-pro' ),
					'section'       => 'wen_travel_promotion_headline_two',
					'template-part' => 'template-parts/promotion-headline-two/content-promotion-two',
				),
				'contact'               => array(
					'label'         => esc_html__( 'Contact Section', 'wen-travel-pro' ),
					'section'       => 'wen_travel_contact',
					'template-part' => 'template-parts/contact/display-contact',
				),
				'team'                  => array(
					'label'         => esc_html__( 'Team', 'wen-travel-pro' ),
					'section'       => 'wen_travel_team',
					'template-part' => 'template-parts/team/display-team',
				),
				'gallery'               => array(
					'label'         => esc_html__( 'Gallery', 'wen-travel-pro' ),
					'section'       => 'wen_travel_gallery_options',
					'template-part' => 'template-parts/gallery/content-gallery',
				),
				'main-content'          => array(
					'label' => esc_html__( 'Main Content', 'wen-travel-pro' ),
				),
				'recent-posts'          => array(
					'label'         => esc_html__( 'Recent Posts ( Only on homepage )', 'wen-travel-pro' ),
					'section'       => 'wen_travel_homepage_options',
					'template-part' => 'template-parts/recent-posts/front-recent-posts',
				),
				'newsletter'            => array(
					'label'         => esc_html__( 'Newsletter', 'wen-travel-pro' ),
					'section'       => 'sidebar-widgets-sidebar-newsletter',
					'template-part' => 'template-parts/footer/footer-newsletter',
				),
				'woo-products-showcase' => array(
					'label'         => esc_html__( 'Woo Products Showcase', 'wen-travel-pro' ),
					'section'       => 'wen_travel_woo_products_showcase',
					'template-part' => 'template-parts/woo-products-showcase/display-products',
				),
				'featured-content'      => array(
					'label'         => esc_html__( 'Featured Content', 'wen-travel-pro' ),
					'section'       => 'wen_travel_featured_content',
					'template-part' => 'template-parts/featured-content/display-featured',
				),
				'latest-package'      => array(
					'label'         => esc_html__( 'Latest Package', 'wen-travel-pro' ),
					'section'       => 'wen_travel_latest_package',
					'template-part' => 'template-parts/latest-package/display-latest-package',
				),
				'logo-slider'           => array(
					'label'         => esc_html__( 'Logo Slider', 'wen-travel-pro' ),
					'section'       => 'wen_travel_logo_slider',
					'template-part' => 'template-parts/logo-slider/display-logo-slider',
				),
				'instagram'             => array(
					'label'         => esc_html__( 'Instagram', 'wen-travel-pro' ),
					'section'       => 'sidebar-widgets-sidebar-instagram',
					'template-part' => 'template-parts/footer/footer-instagram',
				),
		);

		$sortable_sections = apply_filters( 'wen_travel_sortable_sections', $sortable_sections );

		return $sortable_sections;
	}
endif;

function wen_travel_team_social( $link ) {
    // Get supported social icons.
    $social_icons = wen_travel_social_links_icons();

    foreach ( $social_icons as $attr => $value ) {
        if ( false !== strpos( $link, $attr ) ) {
            return wen_travel_get_svg( array( 'icon' => esc_attr( $value ) ) );
        }
    }
}

if ( ! function_exists( 'wen_travel_team_social_links' ) ) :
	/**
	 * Displays team social links html
	 */
	function wen_travel_team_social_links( $counter ) {
		?>
		<div class="artist-social-profile">
            <nav class="social-navigation" role="navigation" aria-label="Social Menu">
                <div class="menu-social-container">
                    <ul id="menu-social-menu" class="social-links-menu">
				        <?php
				        $social_link_one = get_theme_mod( 'wen_travel_team_social_link_one_' . $counter );
				    	if ( $social_link_one ): ?>
				            <li class="menu-item-one">
				                <a target="_blank" rel="nofollow" href="<?php echo esc_url( $social_link_one ); ?>"><?php echo wen_travel_team_social( $social_link_one ); // WPCS: XSS ok. ?></a>
				            </li>
				    	<?php endif;  ?>

						<?php
				    	$social_link_two = get_theme_mod( 'wen_travel_team_social_link_two_' . $counter );
				    	if ( $social_link_two ): ?>
				            <li class="menu-item-two">
				                <a target="_blank" rel="nofollow" href="<?php echo esc_url( $social_link_two ); ?>"><?php echo wen_travel_team_social( $social_link_two ); // WPCS: XSS ok. ?></a>
				            </li>
				    	<?php endif;  ?>

				    	<?php
				    	$social_link_three = get_theme_mod( 'wen_travel_team_social_link_three_' . $counter );
				    	if ( $social_link_three ): ?>
				            <li class="menu-item-three">
				                <a target="_blank" rel="nofollow" href="<?php echo esc_url( $social_link_three ); ?>"><?php echo wen_travel_team_social( $social_link_three ); // WPCS: XSS ok. ?></a>
				            </li>
				    	<?php endif;  ?>

				    	<?php
				    	$social_link_four = get_theme_mod( 'wen_travel_team_social_link_four_' . $counter );
				    	if ( $social_link_four ): ?>
				            <li class="menu-item-four">
				                <a target="_blank" rel="nofollow" href="<?php echo esc_url( $social_link_four ); ?>"><?php echo wen_travel_team_social( $social_link_four ); // WPCS: XSS ok. ?></a>
				            </li>
				    	<?php endif; ?>
				    </ul>
				</div>
			</nav>
		</div><!-- .artist-social-profile -->
		<?php
	}
endif;

if ( ! function_exists( 'wen_travel_post_thumbnail' ) ) :
	/**
	 * $image_size post thumbnail size
	 * $type html, html-with-bg, url
	 * $echo echo true/false
	 * $no_thumb display no-thumb image or not
	 */
	function wen_travel_post_thumbnail( $image_size = 'post-thumbnail', $type = 'html', $echo = true, $no_thumb = false ) {
		$image = $image_url = '';
		
		if ( has_post_thumbnail() ) {
			$image_url = get_the_post_thumbnail_url( get_the_ID(), $image_size );
			$image     = get_the_post_thumbnail( get_the_ID(), $image_size );
		} else {
			if ( $no_thumb ) {
				if ( is_array( $image_size ) ) {
					$image_url  = trailingslashit( get_template_directory_uri() ) . 'images/no-thumb-' . $image_size[0] . 'x' . $image_size[1] . '.jpg';
				} else {
					global $_wp_additional_image_sizes;

					$image_url  = trailingslashit( get_template_directory_uri() ) . 'images/no-thumb-' . $_wp_additional_image_sizes[ $image_size ]['width'] . 'x' . $_wp_additional_image_sizes[ $image_size ]['height'] . '.jpg';
				}
				$image      = '<img src="' . esc_url( $image_url ) . '" alt="" />';
			}

			// Get the first image in page, returns false if there is no image.
			$first_image_url = wen_travel_get_first_image( get_the_ID(), $image_size, '', true );

			// Set value of image as first image if there is an image present in the page.
			if ( $first_image_url ) {
				$image_url = $first_image_url;
				$image = '<img class="wp-post-image" src="'. esc_url( $image_url ) .'">';
			}
		}

		if ( ! $image_url ) {
			// Bail if there is no image url at this stage.
			return;
		}

		if ( 'url' === $type ) {
			return $image_url;
		}

		$output = '<div';

		if ( 'html-with-bg' === $type ) {
			$output .= ' class="post-thumbnail-background" style="background-image: url( ' . esc_url( $image_url ) . ' )"';
		} else {
			$output .= ' class="post-thumbnail"';
		}

		$output .= '>';

		$output .= '<a class="cover-link" href="' . esc_url( get_the_permalink() ) . '" title="' . the_title_attribute( 'echo=0' ) . '">';

		if ( 'html-with-bg' !== $type ) {
			$output .= $image;
		}

		$output .= '</a></div><!-- .post-thumbnail -->';

		if ( ! $echo ) {
			return $output;
		}

		echo $output;
	}
endif;

if ( ! function_exists( 'wen_travel_content_display' ) ) :
	/**
	 * Displays excerpt, content or nothing according to option.
	 */
	function wen_travel_content_display( $show_content, $echo = true ) {
		$output = '';

		if ( $echo ) {
			if ( 'excerpt' === $show_content ) {
				?>
				<div class="entry-summary">
					<?php the_excerpt(); ?>
				</div><!-- .entry-content -->
				<?php
			} elseif ( 'full-content' === $show_content ) {
				?>
				<div class="entry-content">
					<?php the_content(); ?>
				</div><!-- .entry-content -->
				<?php
			}

			return;
		} else {
			if ( 'excerpt' === $show_content ) {
				$output = '<div class="entry-summary"><p>'. get_the_excerpt() . '</p></div>';
			} elseif ( 'full-content' === $show_content ) {
				$output = '<div class="entry-content">'. get_the_content() . '</div>';
			}
		}

		return wp_kses_post( $output );
	}
endif;
